from qgis.PyQt.QtCore import QPointF, QRect, Qt, pyqtSignal
from qgis.PyQt.QtGui import QPainter, QPixmap
from qgis.PyQt.QtWidgets import QWidget

from panoramax.logic.image_viewer import compute_center_x_for_heading


class PanoramaxWidget(QWidget):
    """Widget to display a selected geolocated panoramic picture"""

    # signal to emit axis orientation on 360 picture
    orientationChanged = pyqtSignal(float)

    def __init__(self, parent=None):
        super().__init__(parent)
        self.setMinimumHeight(250)
        self.setMinimumWidth(400)
        self.setMouseTracking(True)
        self.source = None
        self.pano = None
        self.center = None
        self.sourceRect = QRect()
        self.mousePos = None
        self.initial_heading = 0
        self.field_of_view = None
        self.is_360 = True

    def setPixmap(self, pixmap, field_of_view: int):
        if pixmap is None:
            return
        self.source = pixmap
        self.field_of_view = field_of_view if field_of_view else 80
        self.is_360 = self.field_of_view == 360

        w, h = self.source.width(), self.source.height()
        self.pano = QPixmap(w * 3, h)
        qp = QPainter(self.pano)
        if self.is_360:
            qp.drawPixmap(0, 0, self.source)
            qp.drawPixmap(w, 0, self.source)
            qp.drawPixmap(2 * w, 0, self.source)
        else:
            qp.drawPixmap(w, 0, self.source)
        qp.end()
        self.center = QPointF(self.pano.rect().center())
        self.sourceRect = QRect(0, 0, self.width(), self.height())
        self.sourceRect.moveCenter(self.center.toPoint())
        self.update()

    def moveCenter(self, dx, dy):
        """
        Moves the center of the panoramic image by the given dx and dy offsets.
        :param dx: Horizontal offset.
        :param dy: Vertical offset.
        :type dx: float
        :type dy: float
        :return: None
        :rtype: None
        """
        if self.source is None:
            return
        self.center += QPointF(dx, dy)

        min_y = self.sourceRect.height() * 0.5
        max_y = self.source.height() - self.height() * 0.5
        if self.center.y() < min_y:
            self.center.setY(int(min_y))
        elif self.center.y() > max_y:
            self.center.setY(int(max_y))

        w = self.source.width()
        if self.is_360:
            min_x = w * 0.5
            max_x = w * 2.5
            if self.center.x() < min_x:
                self.center.setX(max_x)
            elif self.center.x() > max_x:
                self.center.setX(min_x)
        else:
            # Limit horizontal scrolling when not 360 picture
            min_x = w
            max_x = w * 2
            if self.center.x() < min_x:
                self.center.setX(min_x)
            elif self.center.x() > max_x:
                self.center.setX(max_x)
        self.sourceRect.moveCenter(self.center.toPoint())
        self.update()
        # compute angle between initial center (w*1.5) and actual center (center.x)
        if self.is_360:
            angle = ((self.center.x() - w * 1.5) / w) * 360
        else:
            angle = ((self.center.x() - w * 1.5) / w) * self.field_of_view
        angle = (self.initial_heading + angle) % 360
        self.orientationChanged.emit(angle)

    def mousePressEvent(self, event):
        if event.button() == Qt.LeftButton:
            self.mousePos = event.pos()

    def mouseMoveEvent(self, event):
        if (
            self.mousePos is None
            or event.buttons() != Qt.LeftButton
            or self.source is None
        ):
            return
        delta = event.pos() - self.mousePos

        self.moveCenter(-delta.x(), -delta.y())
        self.mousePos = event.pos()

    def mouseReleaseEvent(self, event):
        self.mousePos = None

    def paintEvent(self, event):
        if self.pano is None:
            return
        qp = QPainter(self)
        qp.drawPixmap(self.rect(), self.pano, self.sourceRect)

    def resizeEvent(self, event):
        if self.center is None or self.pano is None:
            return
        self.sourceRect.setSize(self.size())
        self.sourceRect.moveCenter(self.center.toPoint())

    def center_on_heading(self, heading):
        """Center the panoramic image on a given heading."""
        if self.source is None:
            return
        # Calculate the horizontal position corresponding to the heading
        center_x = compute_center_x_for_heading(
            self.source.width(), self.initial_heading, heading
        )
        self.center.setX(int(center_x))
        self.sourceRect.moveCenter(self.center.toPoint())
        self.update()
