#! python3  # noqa: E265

"""
    Main plugin module.
"""

# standard
from functools import partial
from pathlib import Path

# PyQGIS
from qgis.core import QgsApplication, QgsSettings
from qgis.gui import QgisInterface
from qgis.PyQt.QtCore import QCoreApplication, QLocale, QTranslator, QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QAction

# project
from panoramax.__about__ import (
    DIR_PLUGIN_ROOT,
    __icon_path__,
    __title__,
    __uri_homepage__,
)
from panoramax.gui.dlg_authentication import AuthenticationDialog
from panoramax.gui.dlg_settings import PlgOptionsFactory
from panoramax.gui.panoramax_map_tool import PanoramaxMapTool
from panoramax.processing import PluginPanoramaxProvider
from panoramax.toolbelt import PlgLogger

# ############################################################################
# ########## Classes ###############
# ##################################


class PluginPanoramaxPlugin:
    def __init__(self, iface: QgisInterface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class which \
        provides the hook by which you can manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.log = PlgLogger().log
        self.provider = None

        # translation
        # initialize the locale
        self.locale: str = QgsSettings().value("locale/userLocale", QLocale().name())[
            0:2
        ]
        locale_path: Path = (
            DIR_PLUGIN_ROOT / f"resources/i18n/{__title__.lower()}_{self.locale}.qm"
        )
        self.log(message=f"Translation: {self.locale}, {locale_path}", log_level=4)
        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path.resolve()))
            QCoreApplication.installTranslator(self.translator)

        # dialogs placeholders
        self.dlg_register_instance = None
        self.dlg_viewer = None
        self.map_tool = None

    def initGui(self):
        """Set up plugin UI elements."""

        # settings page within the QGIS preferences menu
        self.options_factory = PlgOptionsFactory()
        self.iface.registerOptionsWidgetFactory(self.options_factory)

        # -- Actions

        # Instance registration (authentication)
        self.dlg_register_instance = AuthenticationDialog(self.iface.mainWindow())
        self.dlg_register_instance.authentication_config_saved.connect(
            self.on_register_instance_changed
        )
        self.action_instance_new = QAction(
            QgsApplication.getThemeIcon("mActionAdd.svg"),
            self.tr("Register a new Panoramax instance"),
            self.iface.mainWindow(),
        )
        self.action_instance_new.triggered.connect(
            self.display_dlg_instance_registration
        )

        # Display geolocated picture (viewer)
        self.action_viewer = QAction(
            QIcon(str(__icon_path__)),
            self.tr("Geolocated picture viewer"),
            self.iface.mainWindow(),
        )
        self.action_viewer.setCheckable(True)
        self.action_viewer.toggled.connect(self.activate_map_tool)
        self.iface.addToolBarIcon(self.action_viewer)

        self.action_help = QAction(
            QgsApplication.getThemeIcon("mActionHelpContents.svg"),
            self.tr("Help"),
            self.iface.mainWindow(),
        )
        self.action_help.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.test = QAction(
            QIcon(str(__icon_path__)),
            self.tr("Test"),
            self.iface.mainWindow(),
        )

        self.action_settings = QAction(
            QgsApplication.getThemeIcon("console/iconSettingsConsole.svg"),
            self.tr("Settings"),
            self.iface.mainWindow(),
        )
        self.action_settings.triggered.connect(
            lambda: self.iface.showOptionsDialog(currentPage=f"mOptionsPage{__title__}")
        )

        # -- Menu
        self.iface.addPluginToMenu(__title__, self.action_instance_new)
        self.iface.addPluginToMenu(__title__, self.action_viewer)
        self.iface.addPluginToMenu(__title__, self.action_settings)
        self.iface.addPluginToMenu(__title__, self.action_help)

        # -- Toolbar
        self.iface.addToolBarIcon(self.action_viewer)

        # -- Processing
        self.initProcessing()

        # -- Help menu

        # documentation
        self.iface.pluginHelpMenu().addSeparator()
        self.action_help_plugin_menu_documentation = QAction(
            QIcon(str(__icon_path__)),
            f"{__title__} - Documentation",
            self.iface.mainWindow(),
        )
        self.action_help_plugin_menu_documentation.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.iface.pluginHelpMenu().addAction(
            self.action_help_plugin_menu_documentation
        )

    def on_register_instance_changed(self, instance_url: str):
        """Actions performed when the instance registration dialog is closed."""
        if self.map_tool is not None:
            self.map_tool.refresh_api_client(new_url=instance_url)

    def initProcessing(self):
        self.provider = PluginPanoramaxProvider()
        QgsApplication.processingRegistry().addProvider(self.provider)

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)

    def unload(self):
        """Cleans up when plugin is disabled/uninstalled."""
        # -- Clean up menu
        self.iface.removePluginMenu(__title__, self.action_instance_new)
        self.iface.removePluginMenu(__title__, self.action_viewer)
        self.iface.removePluginMenu(__title__, self.action_help)
        self.iface.removePluginMenu(__title__, self.action_settings)

        # -- Clean up toolbar
        self.iface.removeToolBarIcon(self.action_viewer)

        # -- Clean up preferences panel in QGIS settings
        self.iface.unregisterOptionsWidgetFactory(self.options_factory)

        # -- Unregister processing
        QgsApplication.processingRegistry().removeProvider(self.provider)

        # remove from QGIS help/extensions menu
        if self.action_help_plugin_menu_documentation:
            self.iface.pluginHelpMenu().removeAction(
                self.action_help_plugin_menu_documentation
            )

        # remove actions
        del self.action_settings
        del self.action_help

    def display_dlg_instance_registration(self) -> None:
        """Display instance registration dialog."""
        if self.dlg_register_instance is not None:
            self.dlg_register_instance.show()

    def display_dlg_viewer(self) -> None:
        """Display viewer dialog."""
        if self.dlg_viewer is not None:
            self.dlg_viewer.show()

    def activate_map_tool(self, checked):
        canvas = self.iface.mapCanvas()
        self.log(
            message=self.tr("Activate Panoramax map tool"),
            log_level=0,
        )
        if checked:
            self.log(
                message=self.tr("Activate Panoramax map tool : CHECKED"),
                log_level=0,
            )
            if self.map_tool is None:
                self.map_tool = PanoramaxMapTool(canvas, self.iface)
            canvas.setMapTool(self.map_tool)
            self.map_tool.activated.emit()
        else:
            canvas.unsetMapTool(self.map_tool)

    def deactivate_map_tool(self):
        """Deactivate the Panoramax map tool."""
        self.log(
            message=self.tr("Deactivate Panoramax map tool"),
            log_level=0,
        )
        if self.action_viewer.isChecked():
            self.action_viewer.setChecked(False)
        self.map_tool.deactivated.emit()
