#! python3  # noqa: E265

import math

from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsCoordinateTransformContext,
    QgsDistanceArea,
    QgsGeometry,
    QgsProject,
)
from qgis.gui import QgsMapCanvas, QgsMapTool, QgsVertexMarker
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QColor

from panoramax.gui.dlg_image_viewer import DlgImageViewer
from panoramax.service.api_client import ApiClient
from panoramax.toolbelt import PlgLogger, PlgOptionsManager


class PanoramaxMapTool(QgsMapTool):
    def __init__(self, canvas: QgsMapCanvas, iface):
        """
        Custom map control to display closest panoramax picture from clicked coordinates.
        :param canvas: map canvas
        :type canvas: QgsMapCanvas
        :param iface: qgis interface
        :type iface: QgsInterface
        """
        super().__init__(canvas)
        self.canvas = canvas
        self.iface = iface

        # toolbelt
        self.log = PlgLogger().log
        self.plg_settings_mngr = PlgOptionsManager()
        self.plg_settings = self.plg_settings_mngr.get_plg_settings()

        # API client
        self.api_client = None

        # map tool
        self.current_picture = None
        self.current_heading = None
        self.dlg_image_viewer = DlgImageViewer(iface.mainWindow())
        self.imageViewerDockArea = Qt.RightDockWidgetArea

        self.dlg_image_viewer.dockLocationChanged.connect(
            self.on_image_dock_location_changed
        )
        self.marker_triangle = QgsVertexMarker(self.canvas)
        self.marker_circle = QgsVertexMarker(self.canvas)
        self.dlg_image_viewer.pano_widget.orientationChanged.connect(
            self.update_marker_orientation
        )
        self.dlg_image_viewer.btn_right.clicked.connect(self.on_btn_right_clicked)
        self.dlg_image_viewer.btn_left.clicked.connect(self.on_btn_left_clicked)

    def on_image_dock_location_changed(self, area):
        self.imageViewerDockArea = area

    def refresh_api_client(self, new_url: str):
        if new_url:
            self.log(
                message=f"Refresh API client with url {new_url}",
                log_level=0,
                push=True,
                duration=2,
            )
            self.api_client = ApiClient(new_url)
        else:
            self.check_api_client()

    def update_marker_orientation(self, angle):
        """
        Update map marker orientation
        :param angle: angle between 0° and 360° from the north
        :type angle: float
        """
        self.current_heading = angle
        self.log(
            message=f"CURRENT HEADING : {self.current_heading}", push=False, log_level=0
        )
        self.marker_triangle.setRotation(angle)

    def transformed_geom(self, geometry, source_crs, target_crs) -> QgsGeometry:
        """
        Reproject and buffer geom
        :param geometry: Geometry point
        :type geometry: QgsGeometry
        :param source_crs: CRS code source
        :param source_crs: string
        :param target_crs: CRS code target
        :param target_crs: string
        :return: transformed geometry
        :rtype: QgsGeometry
        """
        pt_geom = QgsGeometry().fromWkt(geometry.asWkt())
        xform = QgsCoordinateTransform(source_crs, target_crs, QgsProject.instance())
        pt_geom.transform(xform)
        buffered_geom = pt_geom.buffer(1, 8)
        bb_geom = QgsGeometry.fromRect(buffered_geom.boundingBox())
        return bb_geom

    def canvasReleaseEvent(self, event):
        """On release mouse"""
        # Handle mouse release events
        point = self.toMapCoordinates(event.pos())
        point_geom = QgsGeometry.fromPointXY(point)

        # Convert the point to metric coordinates
        self.canvas_to_metric_coordinates(point_geom)

        # create buffer around the clicked point
        buffer_dist = 300  # in meters
        buffer_segments = 2
        search_buffer_geom = point_geom.buffer(buffer_dist, buffer_segments)

        # reproject the buffer geometry from the map canvas CRS to WGS84
        self.metric_to_geographic_coordinates(search_buffer_geom)

        self.log(
            message=self.tr(f"Buffer (geographic): {search_buffer_geom.asWkt()}"),
            log_level=0,
        )
        # search area as the buffer bbox
        search_area = search_buffer_geom.boundingBox().asWktCoordinates()
        search_area_param = search_area.replace(" ", ",").replace(",,", ",")

        # Get picture infos from API
        if not self.check_api_client():
            return
        try:
            self.log(
                message=self.tr(f"Searching picture with area: {search_area_param}"),
                log_level=0,
            )
            self.current_picture = self.api_client.search_picture(search_area_param)
        except Exception as e:
            self.log(
                message=self.tr(f"Error while searching picture: {str(e)}"),
                log_level=0,
                push=True,
                duration=5,
            )
            return
        # Get image from url info
        self.display_image()

    def check_api_client(self):
        if not self.api_client:
            if not self.plg_settings_mngr.get_plg_settings().instance_url:
                self.log(
                    message=self.tr(
                        "No instance URL defined. Please register a panoramax instance"
                    ),
                    log_level=2,
                    push=True,
                    duration=5,
                )
                return
            self.api_client = ApiClient(
                self.plg_settings_mngr.get_plg_settings().instance_url
            )
        self.log(
            message=self.tr("API client available"),
            log_level=0,
        )
        return True

    def canvas_to_metric_coordinates(self, geometry):
        """
        Transforms the given geometry from the current map canvas coordinate reference system to WGS84 (EPSG:4326) coordinates.
        :param sourceCrs: QgsCoordinateReferenceSystem, the source CRS (current map canvas CRS)
        :param destCrs: QgsCoordinateReferenceSystem, the destination CRS (WGS84)
        :param geometry: QgsGeometry, the geometry to be transformed
        :type geometry: QgsGeometry
        :return: None, The transformation is applied in-place to the provided geometry.
        :rtype: None
        """
        sourceCrs = QgsCoordinateReferenceSystem(
            self.canvas.mapSettings().destinationCrs().postgisSrid()
        )
        destCrs = QgsCoordinateReferenceSystem(3857)
        canvas_to_3857 = QgsCoordinateTransform(
            sourceCrs, destCrs, QgsProject.instance()
        )
        geometry.transform(canvas_to_3857)

    def metric_to_geographic_coordinates(self, geometry):
        sourceCrs = QgsCoordinateReferenceSystem(3857)
        destCrs = QgsCoordinateReferenceSystem(4326)
        canvas_to_wgs84 = QgsCoordinateTransform(
            sourceCrs, destCrs, QgsProject.instance()
        )
        geometry.transform(canvas_to_wgs84)

    def geographic_to_canvas_coordinates(self, geometry):
        """
        Transforms the given geometry from WGS84 (EPSG:4326) coordinates to the current map canvas coordinate reference system.
        :param sourceCrs: QgsCoordinateReferenceSystem, the source CRS (WGS84)
        :param destCrs: QgsCoordinateReferenceSystem, the destination CRS (current map canvas CRS)
        :type destCrs: QgsCoordinateReferenceSystem
        :param geometry: QgsGeometry, the geometry to be transformed
        :type geometry: QgsGeometry
        :return: None, The transformation is applied in-place to the provided geometry.
        :rtype: None
        """
        sourceCrs = QgsCoordinateReferenceSystem(4326)
        destCrs = QgsCoordinateReferenceSystem(
            self.canvas.mapSettings().destinationCrs().postgisSrid()
        )
        wgs84_to_canvas = QgsCoordinateTransform(
            sourceCrs, destCrs, QgsProject.instance()
        )
        geometry.transform(wgs84_to_canvas)

    def display_image(self):
        """
        Displays the current image in the image viewer dialog, updates sequence information,
        and creates map markers for the image location.
        """
        if not self.check_api_client():
            return
        try:
            image_pixmap = self.api_client.get_image_by_url(
                self.current_picture.get_url()
            )
        except Exception as e:
            self.log(
                message=self.tr(f"Error while getting image from API: {str(e)}"),
                log_level=2,
                push=True,
            )
            return

        # Get Sequence infos
        if not self.current_picture.get_sequence_id():
            self.log(
                message=self.tr("No sequence found for the picture"),
                log_level=2,
                push=True,
            )
            return
        try:
            sequence = self.api_client.get_sequence(
                self.current_picture.get_sequence_id()
            )
        except Exception as e:
            self.log(
                message=self.tr(f"Error while getting sequence from API: {str(e)}"),
                log_level=2,
                push=True,
            )
            return
        pictures_count = sequence.get_pictures_count()

        # Display image in the viewer dialog without given bearing
        self.dlg_image_viewer.set_pixmap(image_pixmap)
        self.dlg_image_viewer.pano_widget.initial_heading = (
            self.current_picture.get_heading()
        )

        # compute the road bearing in the direct way
        road_bearing = self.compute_bearing(
            self.current_picture.get_geom(),
            self.current_picture.get_next_geom(),
        )
        if road_bearing is None:
            road_bearing = self.current_picture.get_heading()
        self.dlg_image_viewer.initial_orientation = road_bearing
        self.log(
            message=f"display image - initial orientation : {self.dlg_image_viewer.initial_orientation}",
            log_level=0,
        )

        # Center image to the road bearing or the heading explicitely changed by user
        self.dlg_image_viewer.pano_widget.center_on_heading(
            road_bearing if self.current_heading is None else self.current_heading
        )
        self.iface.addDockWidget(self.imageViewerDockArea, self.dlg_image_viewer)
        self.dlg_image_viewer.set_content_infos(self.current_picture, pictures_count)

        # Create marker for the picture location
        marker_geom = QgsGeometry.fromPointXY(self.current_picture.get_geom())
        self.geographic_to_canvas_coordinates(marker_geom)
        icon_size = 30
        self.marker_triangle.setCenter(marker_geom.asPoint())
        self.marker_circle.setCenter(marker_geom.asPoint())
        self.marker_triangle.setIconSize(icon_size)
        self.marker_circle.setIconSize(20)
        self.marker_triangle.setColor(QColor(255, 255, 255))
        self.marker_circle.setColor(QColor(255, 255, 255))
        self.marker_triangle.setFillColor(QColor(26, 35, 126))
        self.marker_circle.setFillColor(QColor(30, 136, 229))
        self.marker_triangle.setIconType(
            QgsVertexMarker.IconType.ICON_INVERTED_TRIANGLE
        )
        self.marker_circle.setIconType(QgsVertexMarker.IconType.ICON_CIRCLE)

        # Set rotation of the map marker to the road bearing
        self.marker_triangle.setRotation(
            float(
                road_bearing if self.current_heading is None else self.current_heading
            )
        )
        self.marker_triangle.setPenWidth(3)
        self.marker_circle.setPenWidth(3)
        self.marker_triangle.setZValue(1000)
        self.marker_circle.setZValue(1001)

    def compute_bearing(self, point1, point2):
        """
        Computes the bearing between two points.
        :param point1: First point as QgsPointXY
        :param point2: Second point as QgsPointXY
        :return: Bearing in degrees
        """
        if point1 is None or point2 is None:
            return None
        distance_area = QgsDistanceArea()
        context = QgsCoordinateTransformContext()
        distance_area.setSourceCrs(QgsCoordinateReferenceSystem(4326), context)

        bearing = math.degrees(distance_area.bearing(point1, point2)) % 360

        self.log(
            message=self.tr(
                f"Computing bearing between {str(point1)} and {str(point2)} = {bearing}"
            ),
            log_level=0,
        )
        return bearing

    def activate(self):
        """Activate the custom map tool"""
        self.log(
            message=self.tr("Panoramax map tool activated"),
            log_level=0,
        )
        self.current_heading = None

    def deactivate(self):
        """Deactivate the custom map tool"""
        self.log(
            message=self.tr("Panoramax map tool deactivated"),
            log_level=0,
        )
        self.current_heading = None

    def on_btn_right_clicked(self):
        """Handle up button click."""
        nav_direction = self.dlg_image_viewer.btn_right.property("to")
        self.nav_to(nav_direction)

    def on_btn_left_clicked(self):
        """Handle down button click."""
        nav_direction = self.dlg_image_viewer.btn_left.property("to")
        self.nav_to(nav_direction)

    def nav_to(self, direction: str):
        """
        Navigate to the next or previous picture in the sequence.
        :param direction: "next" or "prev"
        :type direction: str
        :raises Exception: if no picture available in the sequence.
        :return: None
        :rtype: None
        """
        if direction == "next":
            if not self.current_picture.get_next():
                self.log(
                    message=self.tr("No next picture available in the sequence"),
                    log_level=0,
                    push=True,
                    duration=5,
                )
                return
            # Get next picture from API
            next_picture_id = self.current_picture.get_next()
            self.current_picture = self.api_client.get_picture(next_picture_id)
        elif direction == "prev":
            if not self.current_picture.get_previous():
                self.log(
                    message=self.tr("No previous picture available in the sequence"),
                    log_level=0,
                    push=True,
                    duration=5,
                )
                return
            # Get previous picture from API
            previous_picture_id = self.current_picture.get_previous()
            if not self.check_api_client():
                return
            try:
                self.current_picture = self.api_client.get_picture(previous_picture_id)
            except Exception as e:
                self.log(
                    message=self.tr(
                        f"Error while getting previous picture from API: {str(e)}"
                    ),
                    log_level=2,
                    push=True,
                    duration=5,
                )
                return
        else:
            self.log(
                message=self.tr(f"Unknown direction: {direction}"),
                log_level=2,
                push=True,
                duration=5,
            )
            return
        self.display_image()
