import json

from qgis.core import QgsBlockingNetworkRequest
from qgis.PyQt.QtCore import QByteArray, QCoreApplication, QUrl
from qgis.PyQt.QtGui import QPixmap
from qgis.PyQt.QtNetwork import QNetworkRequest

from panoramax.model.picture import Picture
from panoramax.model.sequence import Sequence
from panoramax.toolbelt.log_handler import PlgLogger


class ApiClient:
    def __init__(self, base_url: str):

        # toolbelt
        self.log = PlgLogger().log

        self.base_url = base_url

    def __get(self, url, params=None):
        if params is None:
            params = {}
        else:
            url += "?" + "&".join(f"{key}={value}" for key, value in params.items())
        request = QNetworkRequest(QUrl(url))
        request.setAttribute(QNetworkRequest.FollowRedirectsAttribute, True)
        request.setMaximumRedirectsAllowed(3)
        blocking_request = QgsBlockingNetworkRequest()
        try:
            blocking_request.get(request)
        except Exception as e:
            self.log(
                message=self.tr(f"Exception during GET request to {url}: {str(e)}"),
                log_level=2,
                push=False,
            )
            return

        reply = blocking_request.reply()
        return reply

    def get(self, endpoint, params=None):
        url = f"{self.base_url}/{endpoint}"
        self.log(
            message=self.tr(f"GET request to {url} with params {params}"),
            log_level=0,
        )
        reply = self.__get(url, params)
        if reply.error():
            raise Exception(
                f"API request error: {reply.errorString()}. \
                Please check the instance URL and your network connection."
            )
        result = reply.content().data().decode()
        if not result:
            raise Exception("Empty response from API")
        return json.loads(result)

    def get_picture(self, picture_id: str):
        picture = self.get(endpoint=f"api/pictures/{picture_id}")
        if not picture:
            raise Exception("Empty response from API")
        return Picture(picture)

    def get_sequence(self, sequence_id: str):
        sequence = self.get(endpoint=f"api/collections/{sequence_id}")
        if not sequence:
            raise Exception("Empty response from API")
        return Sequence(sequence)

    def get_image_by_url(self, url):
        self.log(
            message=self.tr(f"Downloading image from {url}"),
            log_level=0,
        )
        reply = self.__get(url)
        if reply.error():
            raise Exception(
                f"API request error: {reply.errorString()}. \
                Please check the instance URL and your network connection."
            )
        data = reply.content()
        if not data:
            raise Exception("Empty response from API")
        pixmap = QPixmap()
        pixmap.loadFromData(QByteArray(data))
        return pixmap

    def search_picture(self, bbox: str):
        """Search for pictures in a given area.

        :param coordinates_transformer: coordinates transformer.
        :type coordinates_transformer: QgsCoordinateTransform

        :return: first picture found in the area.
        :rtype: Picture or None if no picture found
        :raises Exception: if the API request fails or if no picture is found.
        :raises ValueError: if bbox is not a valid bounding box.
        """

        if not bbox:
            raise ValueError("Bounding box (bbox) must be provided")
        if not isinstance(bbox, str):
            raise ValueError(
                "Bounding box (bbox) must be a string in the format 'minx,miny,maxx,maxy'"
            )
        # Validate bbox format
        try:
            minx, miny, maxx, maxy = map(float, bbox.split(","))
        except ValueError:
            raise ValueError(
                "Bounding box (bbox) must be a string in the format 'minx,miny,maxx,maxy'"
            )

        # call picture search API
        search_endpoint = "api/search"
        search_params = {
            "bbox": bbox,
            "limit": 1,
        }
        pictures = self.get(endpoint=search_endpoint, params=search_params)
        if (
            not pictures
            or "features" not in pictures
            or not pictures["features"]
            or len(pictures["features"]) == 0
        ):
            self.log(
                message=self.tr("No pictures found in the area"),
                log_level=0,
                push=True,
            )
            return None
        return Picture(pictures["features"][0])

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)
