# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PanoramaViewerDialog
                                 A QGIS plugin
 A simple panorama viewer for QGIS. Based on QWebView with Pannellum js library
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2023-05-26
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Pavel Pereverzev
        email                : pasha004@yandex.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

# -*- coding: utf-8 -*-

import os
import requests

from http.server import HTTPServer, SimpleHTTPRequestHandler
from urllib.parse import urlparse

from qgis.utils import iface
from qgis._core import *


from PyQt5.QtWebKitWidgets import QWebView
from PyQt5 import *
from PyQt5 import QtCore
from PyQt5.QtWidgets import (
    QVBoxLayout,
    QWidget,
    QCheckBox,
    QHBoxLayout,
    QGridLayout,
    QPushButton,
    QMessageBox,
    QProgressBar,
    QDockWidget,
    QComboBox,
    QMainWindow,
    QSizePolicy,
    QStyle,
    QLabel,
)
from PyQt5.QtGui import *
from PyQt5.QtCore import *


base_folder = os.path.dirname(os.path.realpath(__file__))
# base_folder =  r'C:\Users\Pereverzev.PV\AppData\Roaming\QGIS\QGIS3\profiles\default\python\plugins\panorama_viewer'

HOST, PORT = "", 8030


def read_in_chunks(file_object, chunk_size=None):
    """Read file by chunk index """
    while True:
        data = file_object.read(chunk_size)
        if not data:
            break
        yield data


class GetPanorama(QWidget):
    """Get panorama image from web or local storage
    depending on method
    """

    def __init__(self, parent):
        super().__init__()
        self.main_app = parent

    def get_pano_file(self, file_url, method):
        """Selecting a method to get an image """
        if method == "download":
            result = self.download(file_url)
        else:
            result = self.copy_file(file_url)
        return result

    def download(self, url):
        """Web loader which also writes file chunk by chunk """
        self.main_app.pbar.setDisabled(False)
        r = requests.get(url, allow_redirects=True, stream=True)
        if r.status_code != 200:
            return False
        total_length = int(r.headers.get("content-length"))
        file_name = os.path.join(base_folder, "image.JPG")
        if total_length: 
            with open(file_name, "wb") as f:
                total_length = int(total_length)
                p_step = 100 / total_length * 4096
                counter = 0
                for data in r.iter_content(chunk_size=4096):
                    counter += p_step
                    f.write(data)
                    self.main_app.pbar.setValue(int(counter))
                    QtCore.QCoreApplication.processEvents()
        self.main_app.pbar.setValue(0)
        self.main_app.pbar.setDisabled(True)
        return True

    def copy_file(self, file_in):
        """File copier, also uses chunk writing method """
        self.main_app.pbar.setDisabled(False)
        file_out = os.path.join(base_folder, "image.JPG")
        file_stats = os.stat(file_in)
        size_b = file_stats.st_size
        ch_optimal_size = int(size_b / 100)
        p_step = 100 / size_b * ch_optimal_size
        counter = 0
        open(file_out, "w").close()
        with open(file_in, "rb") as f:
            for piece in read_in_chunks(f, ch_optimal_size):
                with open(file_out, "ab") as fout:
                    fout.write(piece)
                counter += p_step
                self.main_app.pbar.setValue(int(counter))
                QtCore.QCoreApplication.processEvents()
        self.main_app.pbar.setValue(0)
        self.main_app.pbar.setDisabled(True)
        return True


class QuietHandler(SimpleHTTPRequestHandler):
    """Web server handler, which will not write any message
    in python console. As a result, no need to open python console
    in QGIS in order to make plugin work
    """

    def log_message(self, format, *args):
        pass


class HttpDaemon(QtCore.QThread):
    """Simple web server class
    """

    def __init__(self, parent, path):
        super(QThread, self).__init__()
        self.server_path = path

    def run(self):
        os.chdir(self.server_path)
        self.server = HTTPServer((HOST, PORT), QuietHandler)
        self.server.serve_forever()

    def stop(self):
        self.server.shutdown()
        self.server.socket.close()


class PanoramaViewerDialog(QDockWidget):
    """Main widget is a dock widget located on the right bottom corner """

    def __init__(self, wrapper):
        QDockWidget.__init__(self)
        self.wrapper = wrapper
        self.resize(480, 480)
        self.setWindowTitle("PanoramaView")
        self.gv = PanoramaViewer(self)
        self.gv.setMinimumSize(QSize(480, 480))
        self.setWidget(self.gv)
        self.closeEvent = self.onDestroy
        self.wrapper.plugin_is_opened = True

    def onDestroy(self, e):
        """Widget close handler """
        self.wrapper.plugin_is_opened = False
        self.gv.reset_tr()
        iface.mapCanvas().selectionChanged.disconnect(self.gv.define_selection)


class PanoramaViewer(QMainWindow):
    """Widget interface """

    def __init__(self, parent):
        super().__init__(parent=None)
        
        # init settings
        self.wrapper = parent
        self.setWindowFlags(self.windowFlags() | QtCore.Qt.WindowStaysOnTopHint)
        self.setWindowTitle("PanoramaView")
        self.setGeometry(800, 650, 1200, 880)
        
        # custom attrs
        self.httpd = None
        self.current_layer = None

        # layout 
        centralWidget = QWidget()
        browser_layout = QHBoxLayout()
        centralLayout = QVBoxLayout()
        control_layout = QVBoxLayout()
        grid_layout = QGridLayout(self)
        grid_layout.setSpacing(10)

        # webview setup
        self.view = QWebView(self)
        sp = self.view.sizePolicy()
        sp.setVerticalPolicy(QSizePolicy.Expanding)
        self.view.setSizePolicy(sp)
        self.view.settings().setObjectCacheCapacities(0, 0, 0)

        # other widgets
        self.lbl_layers = QLabel("Layer:")
        self.cmb_layers = QComboBox()

        self.lbl_fields = QLabel("Field:")
        self.cmb_fields = QComboBox()
        
        self.update_layers = QPushButton()
        self.update_layers.setFixedWidth(30)
        self.update_layers.setIcon(self.style().standardIcon(QStyle.SP_BrowserReload))

        self.auto_update = QCheckBox("Auto-update view")

        self.btn_find_panorama = QPushButton("View panorama")

        self.pbar = QProgressBar()
        self.pbar.setDisabled(True)

        # layout setup
        # 1. browser widget
        browser_layout.addWidget(self.view)

        # 2. layer-field selectors
        grid_layout.addWidget(self.lbl_layers, 1, 1, 1, 2, alignment=Qt.AlignBottom)
        grid_layout.addWidget(self.lbl_fields, 1, 3, 1, 2, alignment=Qt.AlignBottom)
        grid_layout.addWidget(self.cmb_layers, 2, 1, 1, 2, alignment=Qt.AlignBottom)
        grid_layout.addWidget(self.cmb_fields, 2, 3, 1, 2, alignment=Qt.AlignBottom)
        grid_layout.addWidget(self.update_layers, 2, 6, 1, 1, alignment=Qt.AlignBottom)

        # 3. controls and progress bar
        control_layout.addWidget(self.auto_update)
        control_layout.addWidget(self.btn_find_panorama)
        control_layout.addWidget(self.pbar)

        # 4. adding all layouts 
        centralLayout.addLayout(browser_layout)
        centralLayout.addLayout(grid_layout)
        centralLayout.addLayout(control_layout)
        centralWidget.setLayout(centralLayout)
        self.setCentralWidget(centralWidget)

        # triggers
        self.btn_find_panorama.clicked.connect(self.get_pic)
        self.update_layers.clicked.connect(self.load_layers)
        self.auto_update.stateChanged.connect(self.auto_upd_check)
        self.cmb_layers.currentIndexChanged.connect(self.update_current_layer)
        iface.mapCanvas().selectionChanged.connect(self.define_selection)
        self.load_layers()

        # launching web server
        self.httpd = HttpDaemon(self, base_folder)
        self.httpd.start()

    def define_selection(self):
        """Get pic if auto-update view is checked"""
        if self.auto_update.isChecked():
            self.get_pic()

    def auto_upd_check(self):
        """Auto-update handler to make "View panorama" button enabled/disabled """
        if self.auto_update.isChecked():
            self.btn_find_panorama.setDisabled(True)
        else:
            self.btn_find_panorama.setDisabled(False)

    def load_layers(self):
        """Update layer combobox"""
        self.layer_data = {
            lyr.id(): lyr
            for lyr in QgsProject.instance().mapLayers().values()
            if type(lyr) == QgsVectorLayer and lyr.isValid()
        }
        self.cmb_layers.clear()
        i = 0
        for layer_id, layer in self.layer_data.items():
            self.cmb_layers.addItem(layer.name())
            self.cmb_layers.setItemData(i, layer_id, QtCore.Qt.ToolTipRole)
            self.cmb_layers.setItemData(i, layer)
            i += 1
        self.update_current_layer()

    def warning_message(self, err_text):
        """Custom warning message"""
        msg = QMessageBox()
        msg.warning(self, "Warning", err_text)
        return

    def update_current_layer(self):
        """Setting current layer and its fields from combobox value"""
        self.current_layer = self.cmb_layers.currentData()
        if self.current_layer:
            layer_fields = [
                f.name() for f in self.current_layer.fields() if f.type() == 10
            ]
            self.cmb_fields.clear()
            self.cmb_fields.addItems(layer_fields)

        self.btn_find_panorama.setChecked(False)
       

    def reset_tr(self):
        """Reset server"""
        self.btn_find_panorama.setChecked(False)
        if self.httpd:
            self.httpd.stop()

    def get_pic(self):
        """Get pic attrs from selected point and proceed to view panorama"""
        if self.cmb_fields.currentText():
            field_idx = (
                self.current_layer.fields().names().index(self.cmb_fields.currentText())
            )
            selected_features = self.current_layer.selectedFeatures()
            if not selected_features:
                if not self.auto_update.isChecked():
                    self.warning_message("No items selected")
                return
            current_feature = selected_features[0]
            cf_attr = current_feature.attributes()[field_idx]
            result = urlparse(cf_attr)
            img_get = False
            
            # panorama path check
            if cf_attr and os.path.isfile(cf_attr):
                img_get = GetPanorama(self).get_pano_file(cf_attr, "copy")
            elif all([result.scheme, result.netloc]):
                img_get = GetPanorama(self).get_pano_file(cf_attr, "download")
            else:
                pass
            
            # view panorama or not
            if img_get:
                self.view.load(QUrl("http://localhost:8030/index_local.html"))
            else:
                self.view.load(QUrl("http://localhost:8030/index_error.html"))
            return



# dw = PanoramaViewerDialog(None)
# iface.addDockWidget(Qt.RightDockWidgetArea, dw)
# dw.setFloating(True)
