# -*- coding: utf-8 -*-

"""
/***************************************************************************
 ProcessingModuleNameAlgorithm
                                 A QGIS plugin
 This is the plugin description.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-02-18
        copyright            : (C) 2025 by fdobad
        email                : fbadilla@ing.uchile.cl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "fdobad"
__date__ = "2025-02-18"
__copyright__ = "(C) 2025 by fdobad"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from pathlib import Path
from re import sub
from shutil import which

from osgeo_utils.gdal_calc import GDALDataTypeNames
from processing.algs.gdal.GdalAlgorithm import GdalAlgorithm
from processing.algs.gdal.GdalUtils import GdalUtils
from qgis.core import (QgsProcessing, QgsProcessingException, QgsProcessingParameterBoolean, QgsProcessingParameterEnum,
                       QgsProcessingParameterExtent, QgsProcessingParameterMultipleLayers, QgsProcessingParameterNumber,
                       QgsProcessingParameterRasterDestination, QgsProcessingParameterString)
from qgis.PyQt.QtCore import QCoreApplication

python = "python" if which("python") else "python3"

class ProcessingGdalCalcSumAlgorithm(GdalAlgorithm):
    INPUT = "INPUT"
    WEIGHTS = "WEIGHTS"
    EXTENT_OPT = "EXTENT_OPT"
    EXTENT_OPTIONS = ["ignore", "fail", "union", "intersect"]
    EXTENT = "PROJWIN"
    OUTPUT = "OUTPUT"
    NO_DATA = "NO_DATA"
    # OPTIONS = "OPTIONS"
    # EXTRA = "EXTRA"
    RTYPE = "RTYPE"
    HIDE = "HIDE_NO_DATA"

    TYPE = GDALDataTypeNames

    def __init__(self):
        super().__init__()

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterMultipleLayers(
                name=self.INPUT,
                description=self.tr("Input rasters"),
                layerType=QgsProcessing.TypeRaster,
                defaultValue=[QgsProcessing.TypeRaster],
                optional=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.WEIGHTS,
                self.tr(
                    "Additional (space separated) command-line parameters, such as a, b or threshold for bipiecewiselinear or stepup/down"
                ),
                defaultValue=None,
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                self.RTYPE,
                self.tr("Output raster <b>type</b>"),
                options=self.TYPE,
                defaultValue=self.TYPE.index("Float32"),
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.NO_DATA,
                self.tr("Set output <b>NoDataValue</b>"),
                type=QgsProcessingParameterNumber.Type.Double,
                defaultValue=None,
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.HIDE,
                self.tr("HideNoData: ignores the input NoData directive(s) and uses them 'as is' into the calculation"),
                defaultValue=True,
                optional=True,
            )
        )

        if GdalUtils.version() >= 3030000:
            extent_opt_param = QgsProcessingParameterEnum(
                self.EXTENT_OPT,
                self.tr("Handling of extent differences"),
                options=[o.title() for o in self.EXTENT_OPTIONS],
                defaultValue=0,
            )
            extent_opt_param.setHelp(self.tr("This option determines how to handle rasters with different extents"))
            self.addParameter(extent_opt_param)

        if GdalUtils.version() >= 3030000:
            extent_param = QgsProcessingParameterExtent(self.EXTENT, self.tr("Output extent"), optional=True)
            extent_param.setHelp(self.tr("Custom extent of the output raster"))
            self.addParameter(extent_param)

        # options_param = QgsProcessingParameterString(
        #     self.OPTIONS,
        #     self.tr("Additional creation options"),
        #     defaultValue="",
        #     optional=True,
        # )
        # options_param.setFlags(options_param.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced)
        # options_param.setMetadata({"widget_wrapper": {"widget_type": "rasteroptions"}})
        # self.addParameter(options_param)
        # extra_param = QgsProcessingParameterString(
        #     self.EXTRA,
        #     self.tr(
        #         "Override minimun and maximun values (else the values are calculated from the WHOLE input raster)."
        #     ),
        #     defaultValue=None,
        #     optional=True,
        # )
        # extra_param.setFlags(extra_param.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced)
        # self.addParameter(extra_param)

        self.addParameter(QgsProcessingParameterRasterDestination(self.OUTPUT, self.tr("Calculated")))

    def name(self):
        return "weightedsummator"

    def displayName(self):
        return self.tr("Weighted Rasters Sum")

    # def group(self):
    #     """
    #     Returns the name of the group this algorithm belongs to. This string
    #     should be localised.
    #     """
    #     return self.tr(self.groupId())

    # def groupId(self):
    #     """
    #     Returns the unique ID of the group this algorithm belongs to. This
    #     string should be fixed for the algorithm, and must not be localised.
    #     The group id should be unique within each provider. Group id should
    #     contain lowercase alphanumeric characters only and no spaces or other
    #     formatting characters.
    #     """
    #     return "The Algorithm Group"

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def commandName(self):
        # return "python " + str(Path(__file__).parent / "gdal_calc_sum.py")
        return "gdal_calc_sum.py"

    def getConsoleCommands(self, parameters, context, feedback, executing=True):

        out = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
        self.setOutputValue(self.OUTPUT, out)
        if self.NO_DATA in parameters and parameters[self.NO_DATA] is not None:
            noData = self.parameterAsDouble(parameters, self.NO_DATA, context)
        else:
            noData = "none"

        if self.HIDE in parameters and parameters[self.HIDE] is not None:
            hide = self.parameterAsBoolean(parameters, self.HIDE, context)
        else:
            hide = False

        arguments = [
            "--format",
            GdalUtils.getFormatShortNameFromFilename(out),
        ]

        rtype = self.parameterAsEnum(parameters, self.RTYPE, context)
        if self.TYPE[rtype] in ["CInt16", "CInt32", "CFloat32", "CFloat64"] and GdalUtils.version() < 3050300:
            raise QgsProcessingException(
                self.tr("Complex integer and float types are only supported by GDAL 3.5.3 and later")
            )
        if self.TYPE[rtype] == "Int8" and GdalUtils.version() < 3070000:
            raise QgsProcessingException(self.tr("Int8 type is only supported by GDAL 3.7.0 and later"))

        arguments.append("--type " + self.TYPE[rtype])

        if noData is not None:
            arguments.append(f"--NoDataValue {noData}")

        if hide:
            arguments.append("--hideNoData")

        # Check GDAL version for projwin and extent options (GDAL 3.3 is required)
        if GdalUtils.version() < 3030000 and self.EXTENT in parameters.keys():
            raise QgsProcessingException(self.tr("Custom extent is only supported by GDAL 3.3 and later"))
        if GdalUtils.version() < 3030000 and self.EXTENT_OPT in parameters.keys():
            raise QgsProcessingException(self.tr("Extent options are only supported by GDAL 3.3 and later"))
        # --projwin and --extent option are mutually exclusive
        if (self.EXTENT in parameters.keys() and parameters[self.EXTENT] is not None) and (
            self.EXTENT_OPT in parameters.keys() and parameters[self.EXTENT_OPT] != 0
        ):
            raise QgsProcessingException(
                self.tr("Custom extent and extent options are mutually exclusive. Please choose one.")
            )
        extent = self.EXTENT_OPTIONS[self.parameterAsEnum(parameters, self.EXTENT_OPT, context)]
        if extent != "ignore":
            arguments.append(f"--extent {extent}")

        def all_equal(iterator):
            iterator = iter(iterator)
            try:
                first = next(iterator)
            except StopIteration:
                return True
            return all(first == rest for rest in iterator)

        infiles = []
        pixel_size_X, pixel_size_Y, srs = [], [], []
        layers = self.parameterAsLayerList(parameters, self.INPUT, context)
        if not layers:
            raise QgsProcessingException(self.tr("No input rasters provided"))
        crs = None
        for layer in layers:
            if not Path(layer.publicSource()).is_file():
                raise QgsProcessingException(f"Raster {layer.name()} file not found: {layer.publicSource()}")
            layer_details = GdalUtils.gdal_connection_details_from_layer(layer)
            infiles += ['"' + layer_details.connection_string + '"']
            # arguments.append('"' + r.publicSource() + '"')
            pixel_size_X.append(layer.rasterUnitsPerPixelX())
            pixel_size_Y.append(layer.rasterUnitsPerPixelY())
            srs.append(layer.crs().authid())
            crs = layer.crs()
        if not (all_equal(pixel_size_X) and all_equal(pixel_size_Y) and all_equal(srs)):
            raise QgsProcessingException(
                self.tr(
                    f"For all output extent options, the pixel size (resolution) and SRS (Spatial Reference System) of all the input rasters must be the same \n{pixel_size_X=}\n{pixel_size_Y=}\n{srs=}"
                )
            )

        if crs:
            bbox = self.parameterAsExtent(parameters, self.EXTENT, context, crs)
            if not bbox.isNull():
                arguments.append(f"--projwin {bbox.xMinimum()} {bbox.yMaximum()} {bbox.xMaximum()} {bbox.yMinimum()}")

        arguments.append("--outfile")
        arguments.append('"' + out + '"')

        # options = self.parameterAsString(parameters, self.OPTIONS, context)
        # if options:
        #     arguments.append(f"--co {options}")
        # if self.EXTRA in parameters and parameters[self.EXTRA] not in (None, ""):
        #     arguments.append("--")
        #     arguments.append(self.parameterAsString(parameters, self.EXTRA, context))

        weights = self.parameterAsString(parameters, self.WEIGHTS, context)
        if weights:
            try:
                weights = [str(float(w)) for w in weights.split()]
            except ValueError:
                raise QgsProcessingException(self.tr("Weights must be a list of numbers separated by spaces"))
            if len(weights) != len(infiles):
                raise QgsProcessingException(
                    self.tr(f"Number of weights {len(weights)} must match the number of rasters {len(infiles)}")
                )
            arguments.append("--weights " + " ".join(weights))

        return [python, str(Path(__file__).parent / "gdal_calc_sum.py")] + arguments + ["--"] + infiles

    def helpUrl(self):
        return "https://gdal.org/programs/gdal_calc.html"

    def shortHelpString(self):
        import html

        from .gdal_calc_sum import __doc__ as docstring

        docstring = sub(r"<!-- BEGIN_ARGPARSE_DOCSTRING -->", "", docstring)
        docstring = sub(r"<!-- END_ARGPARSE_DOCSTRING -->", "", docstring)
        html_docstring = html.escape(docstring).replace("\n", "<br>")
        html_content = f"<pre><code>{html_docstring}</code></pre>"
        return self.tr(html_content)

    def helpString(self):
        return self.shortHelpString()
