# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Marraqueta
                                 A QGIS plugin
 Ponders different rasters with different utility functions
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-05-14
        git sha              : $Format:%H$
        copyright            : (C) 2024 by fdobad@github
        email                : fbadilla@ing.uchile.cl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
# fmt: off
# from qgis.PyQt.QtCore import pyqtRemoveInputHook
# pyqtRemoveInputHook()
# from IPython.terminal.embed import InteractiveShellEmbed
# InteractiveShellEmbed()()
# fmt: on
"""
import os.path
from functools import partial
from pathlib import Path
from tempfile import NamedTemporaryFile

from qgis.core import (Qgis, QgsApplication, QgsMessageLog, QgsProcessingAlgRunnerTask, QgsProcessingContext,
                       QgsProject, QgsRasterLayer, QgsTask)
from qgis.PyQt.QtCore import QCoreApplication, QSettings, QTranslator
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

from .constants import TAG
from .models.model import Model
# Initialize Qt resources from file resources.py
from .resources.resources import *
# Import the code for the dialog
from .views.view import Dialog


class Marraqueta:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(self.plugin_dir, "i18n", "Marraqueta_{}.qm".format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&Pan Europeo")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        self.tasks = []
        self.final_task = None
        self.context = None
        self.model = None
        self.dlg = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("Marraqueta", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ":/plugins/pan_batido/resources/icon.png"
        self.add_action(
            icon_path, text=self.tr("Load rasters before launching!"), callback=self.run, parent=self.iface.mainWindow()
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&Pan Europeo"), action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.context = QgsProcessingContext()
            self.context.setProject(QgsProject.instance())
            self.model = Model(iface=self.iface, context=self.context)
            self.dlg = Dialog(iface=self.iface, model=self.model)
            print("===Dialog created===")
        else:
            print("===Dialog already===")

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            print("===OK was pressed===")
            QgsMessageLog.logMessage("OK was pressed", tag=TAG, level=Qgis.Info)
            # self.model.print_current_params()
            # self.doit(self.model, self.dlg)

        else:
            print("===else than OK===")
            # self.dlg.model.print_all_params()

    def doit(self, model, view):
        """
        Run the calculations
        - get a temporary folder
        - get n+1 temporary files in that folder
        - use the processing algorithm "paneuropeo:normalizator" to normalize each raster
        - use the processing algorithm "paneuropeo:sumator" to sum all the normalized rasters with their weights
        - chain the sumator to run after all the normalizators
        """
        QgsMessageLog.logMessage(" ", tag=TAG, level=Qgis.Success)
        QgsMessageLog.logMessage("Starting calculations", tag=TAG, level=Qgis.Info)

        weights = []
        outfiles = []
        self.tasks = []
        self.final_task = None

        add2map = view.checkBox_load_normalized.isChecked()

        def task_finished(context, successful, results, force_name="Result", add2map=True):
            if not successful:
                QgsMessageLog.logMessage(f"Task finished unsuccessfully {results}", tag=TAG, level=Qgis.Warning)
            else:
                QgsMessageLog.logMessage(f"Task finished successfully {results}", tag=TAG, level=Qgis.Info)
                if add2map:
                    output_layer = context.getMapLayer(results["OUTPUT"])
                    if output_layer and output_layer.isValid():
                        QgsProject.instance().addMapLayer(context.takeResultLayer(output_layer.id()))
                        print("from context")
                    elif Path(results["OUTPUT"]).is_file():
                        layer = QgsRasterLayer(results["OUTPUT"], force_name)
                        QgsProject.instance().addMapLayer(layer)
                        print("from file")

        rasters = model.get_rasters()
        model.balance_weights()
        for raster_name, raster in rasters.items():
            if model.get_visibility(raster_name):
                weight = model.get_weight(raster_name) / 100
                weights += [weight]

                method = model.get_current_utility_function_name(raster_name)
                if method in ["minmax", "maxmin", "bipiecewiselinear_percent", "stepup_percent", "stepdown_percent"]:
                    minimum, maximum = model.get_minmax(raster_name)
                else:
                    minimum, maximum = None, None

                func_params = model.get_raster_params(raster_name, method)
                func_values_str = " ".join([str(param["value"]) for param in func_params.values()])

                outfile = NamedTemporaryFile(suffix=".tif", delete=False).name
                outfiles += [outfile]

                task = QgsProcessingAlgRunnerTask(
                    algorithm=QgsApplication.processingRegistry().algorithmById("paneuropeo:normalizator"),
                    parameters={
                        "EXTENT_OPT": 0,
                        "INPUT_A": raster,  # .publicSource(),
                        "MAX": maximum,
                        "METHOD": method,
                        "MIN": minimum,
                        "NO_DATA": None,
                        "OUTPUT": outfile,
                        "PARAMS": func_values_str,
                        "PROJWIN": None,
                        "RTYPE": 7,
                    },
                    context=self.context,
                )
                task.setDescription(f"Normalizing {raster_name}")
                task.executed.connect(
                    partial(
                        task_finished,
                        self.context,
                        force_name="norm_" + raster_name,
                        add2map=add2map,
                    )
                )
                self.tasks += [task]
                # report
                if func_values_str:
                    func_values_str = " params: {func_values_str}"
                QgsMessageLog.logMessage(
                    f"{raster_name=} {weight=} {method=}" + func_values_str,
                    tag=TAG,
                    level=Qgis.Info,
                )

        self.final_task = QgsProcessingAlgRunnerTask(
            algorithm=QgsApplication.processingRegistry().algorithmById("paneuropeo:weightedsummator"),
            parameters={
                "EXTENT_OPT": 0,
                "INPUT": outfiles,
                "NO_DATA": None,
                "OUTPUT": "TEMPORARY_OUTPUT",
                "PROJWIN": None,
                "RTYPE": 7,
                "WEIGHTS": " ".join(map(str, weights)),
            },
            context=self.context,
        )
        self.final_task.executed.connect(partial(task_finished, self.context, force_name="Result", add2map=True))

        # Add the final sum task as a subtask with dependencies on all normalization tasks
        for task in self.tasks:
            self.final_task.addSubTask(task, [], QgsTask.SubTaskDependency.ParentDependsOnSubTask)

        QgsApplication.taskManager().addTask(self.final_task)
