# -*- coding: utf-8 -*-

# Import standard libraries
import os,math

# Import the PyQt and QGIS libraries
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from qgis.gui import *

# Import the NumPy library
import numpy as np

# Initialize Qt resources from file resources.py
# import resources

# import related code 
# Set up current path.
currentPath = os.path.dirname( __file__ )

# Import custom libraries
from cad4qgis_core import *

from genericDlg import genericDlg
import matrixTransformation as trans2D
from pLine4qgisUi import pLine4qgisUi
from pLine4qgisItemsSelectDlg import pLine4qgisItemsSelectDlg
from pLine4qgisPlineNavDlg import pLine4qgisPlineNavDlg

"""
	Plugin for qGis to perform inserting/editing of points
	in OGR vector layers.
"""

class pLine4qgis:

	vers = '0.01'
	build_date = '2012-08-04'

	selectList = []	# lista dei selezionati
	markerList = []	# lista dei marcatori
	numPnts = 10		# number of segments of polylines 
	helpFlag = False
	"""
		goFlag management:
		cLayer != '' -> goFlag=1000
		capabilities=ok -> goFlag=1100
		geometryType == 0 -> goFlag=1110
		geometryType == 1 -> goFlag=1120
			inquiry functions allowed
		isEditable == True -> goFlag=1111/1121;
			inserting/editing functions allowed
	"""
	goFlag = 0
	"""
		clickTool.canvasClicked.disconnect() [QgsMapToolEmitPoint()] seems raise error
		when activated without connections; we register in isClickToolActivated when
		clickTool is connected to avoid improper disconnections
	"""
	isClickToolActivated = False

	def __init__(self, iface):
		# Save reference to the QGIS interface
		self.iface = iface
		# reference to map canvas
		self.canvas = self.iface.mapCanvas()
		# out click tool will emit a QgsPoint on every click
		self.clickTool = QgsMapToolEmitPoint(self.canvas)
		# stabilisce la tolleranza di ricerca
		self.eps = self.canvas.mapUnitsPerPixel() * 5
		# create our GUI dialog
		self.dlg = pLine4qgisUi()
		# controle status layer
		self.handleLayerChange()

	def initGui(self):
		# Create action that will start plugin configuration
		self.action = QAction(
			QIcon(":/plugins/cad4qgisEdges/cad4qgis.png"),
			"lineTools",
			self.iface.mainWindow()
		)
		# connect the action to the run method
		QObject.connect(self.action, SIGNAL("triggered()"), self.run)
		# Add toolbar button and menu item
		self.iface.addToolBarIcon(self.action)
		self.iface.addPluginToMenu("cad4qgis", self.action)

		# connect the button handler to signal   

		QObject.connect(self.dlg.sbNumPnts,SIGNAL("valueChanged(int)"),self.numPntsChanged)

		QObject.connect(self.dlg.pbInqLine,SIGNAL("clicked()"),self.pLineInqTool)
		QObject.connect(self.dlg.pbInqEdge,SIGNAL("clicked()"),self.pLineEdgeInqTool)
		QObject.connect(self.dlg.pbInqVertex,SIGNAL("clicked()"),self.pLineVertexInqTool)
		QObject.connect(self.dlg.pbLineLength,SIGNAL("clicked()"),self.pLineLengthTool)
		QObject.connect(self.dlg.pbVerticesDiag,SIGNAL("clicked()"),self.pLineVerticesDiagTool)
		QObject.connect(self.dlg.pbLineAdd,SIGNAL("clicked()"),self.pLineAdd)
		QObject.connect(self.dlg.pbLineOffset,SIGNAL("clicked()"),self.pLineOffsetTool)
		QObject.connect(self.dlg.pbLineRectSeries,SIGNAL("clicked()"),self.pLineRectSeriesTool)
		QObject.connect(self.dlg.pbLinePolarSeries,SIGNAL("clicked()"),self.pLinePolarSeriesTool)
		QObject.connect(self.dlg.pbLineAxialMirror,SIGNAL("clicked()"),self.pLineAxialMirrorTool)
		QObject.connect(self.dlg.pbLineRadialMirror,SIGNAL("clicked()"),self.pLineRadialMirrorTool)
		QObject.connect(self.dlg.pbLineDel,SIGNAL("clicked()"),self.pLineDelTool)
		QObject.connect(self.dlg.pbLineFillet,SIGNAL("clicked()"),self.pLineFilletTool)
		QObject.connect(self.dlg.pbLineChamfer,SIGNAL("clicked()"),self.pLineChamferTool)
		QObject.connect(self.dlg.pbLineRegPolyg,SIGNAL("clicked()"),self.pLineRegPolygTool)
		QObject.connect(self.dlg.pbLineCircle2P,SIGNAL("clicked()"),self.pLineCircle2PTool)
		QObject.connect(self.dlg.pbLineCircleInCbyP,SIGNAL("clicked()"),self.pLineCircleInCbyPTool)
		QObject.connect(self.dlg.pbLineCircle3P,SIGNAL("clicked()"),self.pLineCircle3PTool)
		QObject.connect(self.dlg.pbLineArc,SIGNAL("clicked()"),self.pLineArcTool)
		QObject.connect(self.dlg.pbLineArc3P,SIGNAL("clicked()"),self.pLineArc3PTool)
		QObject.connect(self.dlg.pbLineParabola1,SIGNAL("clicked()"),self.pLineParabola1Tool)
		QObject.connect(self.dlg.pbLineParabola2,SIGNAL("clicked()"),self.pLineParabola2Tool)
		QObject.connect(self.dlg.pbLineParabola3,SIGNAL("clicked()"),self.pLineParabola3Tool)
		QObject.connect(self.dlg.pbLineParabola4,SIGNAL("clicked()"),self.pLineParabola4Tool)
		QObject.connect(self.dlg.pbLineEllisse,SIGNAL("clicked()"),self.pLineEllisseTool)
		QObject.connect(self.dlg.pbLineEllisseByP,SIGNAL("clicked()"),self.pLineEllisseByPTool)
		QObject.connect(self.dlg.pbLineHermite,SIGNAL("clicked()"),self.pLineHermiteTool)
		QObject.connect(self.dlg.pbLineBezier,SIGNAL("clicked()"),self.pLineBezierTool)
		QObject.connect(self.dlg.pbLineUnRspline,SIGNAL("clicked()"),self.pLineUnRsplineTool)
		QObject.connect(self.dlg.pbLineOverhauser,SIGNAL("clicked()"),self.pLineOverhauserTool)
		QObject.connect(self.dlg.pbLineBetaSpline,SIGNAL("clicked()"),self.pLineBetaSplineTool)
		QObject.connect(self.dlg.pbLineInterpolation,SIGNAL("clicked()"),self.pLineInterpolationTool)
		QObject.connect(self.dlg.pbLineVrtsEditMouse,SIGNAL("clicked()"),self.pLineVrtsEditMouse)
		QObject.connect(self.dlg.pbLineVrtsEdit,SIGNAL("clicked()"),self.pLineVrtsEditTool)
		QObject.connect(self.dlg.pbLineOpenClose,SIGNAL("clicked()"),self.pLineOpenCloseTool)
		QObject.connect(self.dlg.pbLineIntersect,SIGNAL("clicked()"),self.pLineIntersectionTool)
		QObject.connect(self.dlg.pbLineUnion,SIGNAL("clicked()"),self.pLineUnionTool)
		QObject.connect(self.dlg.pbLineSplit,SIGNAL("clicked()"),self.pLineSplitTool)
		QObject.connect(self.dlg.pbLineTranslation,SIGNAL("clicked()"),self.pLineTranslationTool)
		QObject.connect(self.dlg.pbLineScaling,SIGNAL("clicked()"),self.pLineScalingTool)
		QObject.connect(self.dlg.pbLineRotation,SIGNAL("clicked()"),self.pLineRotationTool)
		QObject.connect(self.dlg.pbLineCollimation1P,SIGNAL("clicked()"),self.pLineCollimation1PTool)
		QObject.connect(self.dlg.pbLineCollimation2P,SIGNAL("clicked()"),self.pLineCollimation2PTool)
		QObject.connect(self.dlg.pbLineCollimationNP,SIGNAL("clicked()"),self.pLineCollimationNPTool)
		QObject.connect(self.dlg.pbAbout,SIGNAL("clicked()"),self.about)
		QObject.connect(self.dlg.pbInfo,SIGNAL("clicked()"),self.info)
		QObject.connect(self.dlg.pbHelpFlag,SIGNAL("clicked()"),self.helpFlagOnOff)
		# connect the layer changed handler to a signal that the TOC layer has changed
		QObject.connect(self.iface,SIGNAL("currentLayerChanged(QgsMapLayer *)"), self.handleLayerChange)

	# run method that performs all the real work
	def run(self):
		# make our clickTool the tool that we'll use for now
		self.canvas.setMapTool(self.clickTool)
		# show the dialog
		self.dlg.show()
		result = self.dlg.exec_()

	def unload(self):
		# try to disconnect all signals
		if self.isClickToolActivated:
			self.clickTool.canvasClicked.disconnect()
			self.isClickToolActivated = False
		# clean up
		cleanSelection(self)
		# Remove the plugin menu item and icon
		self.iface.currentLayerChanged.disconnect()
		self.iface.removePluginMenu("cad4qgis",self.action)
		self.iface.removeToolBarIcon(self.action)

	def about(self):
		about(self.iface.mainWindow())

	def info(self):
		info(self.iface.mainWindow())

	def helpFlagOnOff(self):
		"""
			(Dis)Attiva l'help contestuale
		"""
		if self.helpFlag:
			self.helpFlag = False
		else:
			self.helpFlag = True

	def handleLayerChange(self):
		"""
			Aggiorna il layer corrente, il provider e il crs e aggiorna il goFlag.
		"""
		self.goFlag = 0
		self.cLayer = ''
		layName = ''
		layEdit = ''
		prName = ''
		crsId = ''
		if self.canvas.currentLayer():
			self.cLayer = self.canvas.currentLayer()
			if self.cLayer:
				self.goFlag = 1000
				layName = self.cLayer.name()
				crs = self.cLayer.crs()
				self.canvas.mapRenderer().setDestinationCrs(crs)
				crsId = crs.authid()
				self.provider = self.cLayer.dataProvider()
				prName = self.provider.name()
				caps = self.provider.capabilities()
				if (caps & QgsVectorDataProvider.AddFeatures) and (caps & QgsVectorDataProvider.DeleteFeatures) and (caps & QgsVectorDataProvider.ChangeGeometries):
					self.goFlag = 1100
					if self.cLayer.geometryType() == QGis.Line:
						self.goFlag = 1120
						QObject.connect(
							self.cLayer,
							SIGNAL("editingStarted()"),
							self.handleLayerEditabilityStart
						)
						QObject.connect(
							self.cLayer,
							SIGNAL("editingStopped()"),
						self.handleLayerEditabilityStop
						)
						if self.cLayer.isEditable():
							layEdit = 'editable'
							self.goFlag = self.goFlag + 1
					else:
						QMessageBox.information(
							self.iface.mainWindow(),
							'handleLayerChange',
							"Sorry, I can't manipulate this type of layer (%d)" % (self.goFlag)
						)
#	qui bisognerebbe decidere, sull base del goFlag ottenuta, se connettere o
#  disconnettere, però forse andrebbe sconneso all'inizio il layer precedente
		# aggiorna il pannello di dialogo
		self.dlg.setLayer(layName,layEdit,prName,crsId)

	def handleLayerEditabilityStart(self):
		"""
			Controlla l'inizio della editabilità del layer
		"""
		self.goFlag = 1121
		self.dlg.setLayerEdit('editable')

	def handleLayerEditabilityStop(self):
		"""
			Controlla la chiusura della editabilità del layer
		"""
		self.goFlag = 1120
		self.dlg.setLayerEdit('')
		# try to disconnect all signals
		if self.isClickToolActivated:
			self.clickTool.canvasClicked.disconnect()
			self.isClickToolActivated = False

	def warning(self,target):
		"""
			Attenzione: forse è meglio scomporre il flag nei 4 componenti: valutare

			Visualizza l'errore;
			l'errore  segnalato solo nel caso < target, ad es. nel caso delle funzioni
			di inquiry il target è 110 (100 layer defined + 10 geometryType = POINT),
			pertanto la segnalazione circa la non editabilità del layer non verrà mai
			segnalata perchè goFlag == 110 eguaglia il target vnificando il test.
		"""
		if self.goFlag < target and self.goFlag < 1000:
			QMessageBox.information(
				self.iface.mainWindow(),
				"Warning",
				"The layer is not defined (%d)" % (self.goFlag)
			)
		elif self.goFlag < target and self.goFlag < 1100:
			QMessageBox.information(
				self.iface.mainWindow(),
				"Warning",
				"The provider has not this capability (%d)" % (self.goFlag)
			)
		elif self.goFlag < target and self.goFlag < 1110:
			QMessageBox.information(
				self.iface.mainWindow(),
				"Warning",
				"The layer is of incorrect type (%d)" % (self.goFlag)
			)
		elif self.goFlag != 1111 and self.goFlag != 1121:
				QMessageBox.information(
					self.iface.mainWindow(),
					"Warning",
					"The layer is not editable (%d)" % (self.goFlag)
				)

	def numPntsChanged(self):
		self.numPnts = self.dlg.getNumPnts()

# ------------------ dummy functions ------------------------

	def dummyFunction(self):
		QMessageBox.information(
			self.iface.mainWindow(),
			'addPntByMouse',
			"opzione non disponibile"
		)

	def prova(self):
		pass





# ------------------ inquiring functions ------------------------

	def pLineInqTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineInq',
				"Inquiry a pLine"
			)
			self.helpFlag = False
		if self.goFlag >= 1120:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineInq
			)
			self.isClickToolActivated = True
		else:
			self.warning(1120)

	def pLineInq(self,point):
		"""
			Inquiry an pLine
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			m = pLineHighlight(self.canvas,feat)
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			msg = "Feature id=%d #vertices=%d length=%f" % (feat.id(),len(line),geom.length())	
			# calcola gli estremi
			x1,y1,x2,y2 = pLineStartEnd(feat)
			if x1 == x2 and y1 == y2:
				msg = msg + ' area=%f' % (geom.area())
			msg = msg + ' start=(%f %f) end=(%f %f)' % (x1,y1,x2,y2)
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineInq',
				msg
			)
			self.canvas.scene().removeItem(m)

	def pLineEdgeInqTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineInq',
				"Inquiry an edge"
			)
			self.helpFlag = False
		if self.goFlag >= 1120:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineEdgeInq
			)
			self.isClickToolActivated = True
		else:
			self.warning(1120)

	def pLineEdgeInq(self,point):
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			pM = pLineHighlight(self.canvas,feat)
			# cerca il segmento
			tmp = searchPlineEdge(feat,point,self.eps)
			if len(tmp) == 2:
				edg,[p1,p2] = tmp
				eM = edgeHighlight(self.canvas,p1,p2)
				d,a = edgeLengthAndInclin(p1,p2)
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineInq',
					'edge start=(%f %f) end=(%f %f) length=%f  inclination=%f' % (p1.x(),p1.y(),p2.x(),p2.y(),d,a)
				)
				self.canvas.scene().removeItem(eM)
			self.canvas.scene().removeItem(pM)

	def pLineVertexInqTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineInq',
				"Inquiry a vertex of a pLine"
			)
			self.helpFlag = False
		if self.goFlag >= 1120:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineVertexInq
			)
			self.isClickToolActivated = True
		else:
			self.warning(1120)

	def pLineVertexInq(self,point):
		"""
			Inquiry an pLine
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			fM = pLineHighlight(self.canvas,feat)
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				pM = pntHighlight(self.canvas,p.x(),p.y())
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineInq',
					"Feature id=%d vertex=%d X=%f Y=%f" % (feat.id(),pid,p.x(),p.y())	
				)
				self.canvas.scene().removeItem(pM)
			self.canvas.scene().removeItem(fM)

	def pLineLengthTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineLength',
				"Partial and total length of a polyline; press the Cancel button to finish"
			)
			self.helpFlag = False
		if self.goFlag >= 1120:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineLength
			)
			self.isClickToolActivated = True
			# inizializza i registri
			self.lenTot = 0.0
			self.x1,self.y1 = 0.0,0.0
			self.Dx,self.Dy = 0.0,0.0
			self.counter = 0
			self.rubBnd = QgsRubberBand(self.canvas)
			self.markerList.append(self.rubBnd)
		else:
			self.warning(1120)

	def pLineLength(self,point):
		"""
			Calcola la lunghezza di una polyline
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				self.counter += 1
				pid,p = tmp
				x2,y2 = p.x(),p.y()
				pM = pntHighlight(self.canvas,x2,y2)
				self.markerList.append(pM)
				self.rubBnd.addPoint(QgsPoint(x2,y2))
				if self.counter > 1:
					dx,dy = x2-self.x1,y2-self.y1
					d = math.sqrt(dx**2+dy**2)
					self.lenTot = self.lenTot + d
					self.Dx = self.Dx + dx
					self.Dy = self.Dy + dy
					result = QMessageBox.question(
						self.iface.mainWindow(),
						"pLineLength",
						"Last length:%12.3f - Total length:%12.3f (dx=%12.3f dy=%12.3f)" % (d,self.lenTot,self.Dx,self.Dy),
						1,
						2
					)
					if result == 2:
						cleanSelection(self)
						# disconnect from click signal
						self.clickTool.canvasClicked.disconnect()
						self.isClickToolActivated = False
						return
				self.x1,self.y1 = x2,y2

	def pLineVerticesDiagTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineVerticesDiag',
				"Looking for duplicated vertices"
			)
			self.helpFlag = False
		if self.goFlag >= 1120:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineVerticesDiag
			)
			self.isClickToolActivated = True
		else:
			self.warning(1120)

	def pLineVerticesDiag(self,point):
		"""
			controlla eventuali vertici doppi;
			usa una soglia di tolleranza pari a self.eps/20
		"""
		tol = self.eps/20

		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			fM = pLineHighlight(self.canvas,feat)
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			# legge il primo vertice
			p1 = line.pop(0)
			x1,y1 = p1.x(),p1.y()
			count = 0
			idx = 0
			# esegue il ciclo
			for p2 in line:
				x2,y2 = p2.x(),p2.y()
				if abs(x2-x1) <= tol and abs(y2-y1) <= tol:
					count += 1
					pM = pntHighlight(self.canvas,x2,y2)
					QMessageBox.information(
						self.iface.mainWindow(),
						'pLineVerticesDiag',
						"Vertex %d (%f %f) duplicated" % (idx,x2,y2)
					)
					self.canvas.scene().removeItem(pM)
				x1,y1 = x2,y2
				idx += 1
			self.canvas.scene().removeItem(fM)
# segnala sempre per	dare riscontro all'utente del funzionmento della procedur	if count:
		QMessageBox.information(
			self.iface.mainWindow(),
			'pLineVerticesDiag',
			"%d duplicated vertices found" % (count)
		)

# ------------------ constructing pLines ------------------------

	def pLineAdd(self):
		"""
			Adding an pLine
		"""
		QMessageBox.information(
			self.iface.mainWindow(),
			'pLineAd',
			"For pLines adding please use the native functions of qGis"
		)

	def pLineDelTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineDel',
				"pLine deleting"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineDel
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineDel(self,point):
		"""
			delete an pLine
		"""
#		cleanSelection(self)
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			fM = pLineHighlight(self.canvas,feat)
			result = QMessageBox.question(
				self.iface.mainWindow(),
				'pLineDel',
				"Feature id=%d" % (feat.id()),
				1,
				2
			)
			if result == 1:
				self.cLayer.deleteFeature(feat.id())
				self.canvas.refresh()
			self.canvas.scene().removeItem(fM)

	def pLineOffsetTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineOffset',
				"Polyline offset"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			myDlg = genericDlg('pLineOffset',['d'])
			myDlg.setValues([0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.offset = float(tmp.pop())
				if self.offset != 0:
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.pLineOffset
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'pLineOffset',
						"The offset cannot be null"
					)
		else:
			self.warning(1121)

	def pLineOffset(self,point):
		"""
			Calcola l'offset di una polilinea; l'originale viene mantenuto.
			Attenzione: è indifeso rispetto al primo punto doppio.
			Questa routine è difficoltosa per la scelta del centro del fillet
			a seconda dell giacitura dei lati: VALUTARE BENE PRIMA DI APPORTARE
			MODIFICHE.
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# leggo la polilinea
			geom = feat.geometry()
			line = geom.asPolyline()
			# inizializzo la nuova plinea
			newLine = []
			# costruisco il nuovo
			p1 = line.pop(0)
			x1,y1 = p1.x(),p1.y()
			p2 = line.pop(0)
			x2,y2 = p2.x(),p2.y()
			ux,uy = x2-x1,y2-y1
			mu = math.sqrt(ux**2+uy**2)
			# creo punto 12
			x12,y12 = x1-self.offset*uy/mu,y1+self.offset*ux/mu
			pnt12 = QgsPoint(x12,y12)
			newLine.append(pnt12)
			# creo punto 21
			x21,y21 = x2-self.offset*uy/mu,y2+self.offset*ux/mu
			pnt21 = QgsPoint(x21,y21)
			# eseguo ciclo
			for p3 in line:
				# leggo vertice seguente
				x3,y3 = p3.x(),p3.y()
				ux,uy = x3-x2,y3-y2
				mu = math.sqrt(ux**2+uy**2)
				if mu:
					# creo punto 23
					x23,y23 = x2-self.offset*uy/mu,y2+self.offset*ux/mu
					pnt23 = QgsPoint(x23,y23)
					# creo punto 32
					x32,y32 = x3-self.offset*uy/mu,y3+self.offset*ux/mu
					pnt32 = QgsPoint(x32,y32)
					# controlliamo se si incontrano
					tmp = intersection1(pnt12,pnt21,pnt23,pnt32)
					if tmp != -1:
						x,y = tmp	# intersezione valida
						pnt23 = QgsPoint(x,y)
						newLine.append(pnt23)
						pnt21 = pnt23
					else:
						# creo arco p21-p23
						newList = arc(QgsPoint(x2,y2),pnt21,pnt23,self.numPnts)
						newLine.extend(newList)
					# aggiorno i vertici
					x1,y1 = x2,y2
					x2,y2 = x3,y3
					pnt12 = pnt23
					pnt21 = pnt32
			# completo la plinea	(QUESTO VA LLA FINE TRASLATO)
			x32,y32 = x3-self.offset*uy/mu,y3+self.offset*ux/mu
			pnt32 = QgsPoint(x32,y32)
			newLine.append(pnt32)
			# aggiungo la feature
			newFeat = QgsFeature()
#			newFeat.addAttribute(0,"hello")
			newFeat.setGeometry(QgsGeometry.fromPolyline(newLine))
			self.cLayer.addFeatures( [ newFeat ] )

# ------------------ rect & polar series ------------------------

	def pLineRectSeriesTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineRectSeries',
				"Creation of rectangular series of polyline"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			myDlg = genericDlg('pLineRectSeries',['# copies','Dx (m)','Dy (m)'])
			myDlg.setValues([0,0.0,0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.Dy = float(tmp.pop())
				self.Dx = float(tmp.pop())
				self.numRep = int(tmp.pop())
				if self.numRep <= 0:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineRectSeries",
						"Warning: the number of copies cannot be null"
					)
					return
				if self.Dx == 0.0 and self.Dy == 0.0:
					QMessageBox.information(
					self.iface.mainWindow(),
						"pLineRectSeries",
						"Warning: the vector cannot be null"
					)
					return
				# try to disconnect all signals
				if self.isClickToolActivated:
					self.clickTool.canvasClicked.disconnect()
				# connect to click signal
				QObject.connect(
					self.clickTool,
					SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
					self.pLineRectSeries
				)
				self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineRectSeries(self,point):
		"""
			Rectangular series of pLines
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			geom = feat.geometry()
			line = geom.asPolyline()
			for i in range(self.numRep):
				newLine = []
				for p in line:
					newPoint = QgsPoint(p.x()+(i+1)*self.Dx, p.y()+(i+1)*self.Dy)
					newLine.append(newPoint)
				newFeat = QgsFeature()
#				newFeat.addAttribute(0,"hello")
				newFeat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ newFeat ] )

	def pLinePolarSeriesTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLinePolarSeries',
				"Creation of polar series of polyline"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
#			selezione del centro di rotazione
			myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Series center')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = self.selectList.pop()	# preleva l'ultimo inserito
				if type(tmp) == QgsPoint:
					x0,y0 = tmp.x(),tmp.y()
					# dialogo per parametri
					myDlg = genericDlg('pLinePolarSeries',['# copies','angle (360)'])
					myDlg.setValues([0,0.0])
					myDlg.show()
					result = myDlg.exec_()
					if result:
						tmp = myDlg.getValues()
						angle = float(tmp.pop())
						self.numRep = int(tmp.pop())
						if self.numRep <= 0:
							QMessageBox.information(
								self.iface.mainWindow(),
								"pLinePolarSeries",
								"Warning: the number of copies cannot be null"
							)
							return
						if angle == 0.0:
							QMessageBox.information(
								self.iface.mainWindow(),
								"pLinePolarSeries",
								"Warning: the angle cannot be null"
							)
							return
						# prepara matrici di trasformazione
						da = trans2D.sessag2rad(angle/(self.numRep-1))
						mat1 = trans2D.translation(-x0,-y0)
						mat2 = trans2D.rotation(da)
						mat = np.dot(mat2,mat1)
						mat3 = trans2D.translation(x0,y0)
						self.mat = np.dot(mat3,mat)
						# try to disconnect all signals
						if self.isClickToolActivated:
							self.clickTool.canvasClicked.disconnect()
						# connect to click signal
						QObject.connect(
							self.clickTool,
							SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
							self.pLinePolarSeries
						)
						self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLinePolarSeries(self,point):
		"""
			Polar series of pLines
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			geom = feat.geometry()
			line = geom.asPolyline()
			# eseguo ciclo
			for i in range(self.numRep):
				# trasforma i vertici della polilinea
				for k,p in enumerate(line):
					x,y,w = np.dot(self.mat,[p.x(),p.y(),1])
					newPoint = QgsPoint(x,y)
					line[k] = newPoint	# deve fare l'aggiornamento sul posto
				newFeat = QgsFeature()
#				newFeat.addAttribute(0,"hello")
				newFeat.setGeometry(QgsGeometry.fromPolyline(line))
				self.cLayer.addFeatures( [ newFeat ] )

# ------------------ axial & radial mirror ------------------------

	def pLineAxialMirrorTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineAxialMirror',
				"Creation of axial mirroring of polyline"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			cleanSelection(self)
#			selezione dell'asse di simmetria
			myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Symmetry axys')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 2:
					# preleva l'asse di simmetria
					p2 = self.selectList.pop()
					p1 = self.selectList.pop()
					self.x1,self.y1 = p1.x(),p1.y()	# servono anche nell routine collegata
					x2,y2 = p2.x(),p2.y()
					# calcolo i coefficienti
					ux,uy = x2-self.x1,y2-self.y1
					mu = math.sqrt(ux**2+uy**2)
					# print 'ux,uy,mu',ux,uy,mu
					self.ux,self.uy = ux/mu,uy/mu
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.pLineAxialMirror
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineAxialMirror",
						"Axys invalid"
					)
		else:
			self.warning(1121)

	def pLineAxialMirror(self,point):
		"""
			Axial mirroring of pLines
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# leggo la polilinea
			geom = feat.geometry()
			line = geom.asPolyline()
			# costruisco la copia
			newLine = []
			for p in line:
				xi,yi = p.x(),p.y()
				tx,ty = xi-self.x1,yi-self.y1
				tu = tx*self.ux + ty*self.uy
				# print 'tx,ty,tu',tx,ty,tu
				PQx,PQy = 2*(tx-tu*self.ux),2*(ty-tu*self.uy)
				# print 'PQx,PQy',PQx,PQy
				Qx,Qy = xi-PQx,yi-PQy
				newPoint = QgsPoint(Qx,Qy)
				newLine.append(newPoint)
			newFeat = QgsFeature()
#			newFeat.addAttribute(0,"hello")
			newFeat.setGeometry(QgsGeometry.fromPolyline(newLine))
			self.cLayer.addFeatures( [ newFeat ] )

	def pLineRadialMirrorTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineRadialMirror',
				"Creation of radial mirroring of polyline"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			cleanSelection(self)
#			selezione dell'asse di simmetria
			myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Mirror center')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 1:
					# preleva l'asse di simmetria
					p = self.selectList.pop()
					self.x0,self.y0 = p.x(),p.y()	# servono anche nell routine collegata
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.pLineRadialMirror
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineRadialMirror",
						"Center invalid"
					)
		else:
			self.warning(1121)

	def pLineRadialMirror(self,point):
		"""
			Radial mirroring of pLines
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# leggo la polilinea
			geom = feat.geometry()
			line = geom.asPolyline()
			# costruisco la copia
			newLine = []
			for p in line:
				newPoint = QgsPoint(2*self.x0-p.x(),2*self.y0-p.y())
				newLine.append(newPoint)
			newFeat = QgsFeature()
#			newFeat.addAttribute(0,"hello")
			newFeat.setGeometry(QgsGeometry.fromPolyline(newLine))
			self.cLayer.addFeatures( [ newFeat ] )

# ------------------ fillet & chamfer ------------------------

	def pLineFilletTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineFillet',
				"Filleting a polyline (?!?!)"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			myDlg = genericDlg('pLineFillet',['radius'])
			myDlg.setValues([0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.rad = float(tmp[0])
				if self.rad > 0.0:
					cleanSelection(self)
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.pLineFillet
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineFillet",
						"Warning: the radius cannot be null"
					)
		else:
			self.warning(1121)

	def pLineFillet(self,point):
		"""
			pLine fillet;
			quando pronto decommentare l'ultima istuzione (cancellazione della primitiva originale)
		"""
		num = 10	# usiamo un numero fisso di segmenti per il raccordo

		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# leggo la polilinea
			geom = feat.geometry()
			line = geom.asPolyline()
			# costruisco il nuovo
			p1 = line.pop(0)
			x1,y1 = p1.x(),p1.y()
			p2 = line.pop(0)
			x2,y2 = p2.x(),p2.y()
			# inizializzo la nuova plinea
			newLine = []
			newPoint = QgsPoint(x1,y1)
			newLine.append(newPoint)
			for p3 in line:
				ux,uy = x2-x1,y2-y1
				mu = math.sqrt(ux**2+uy**2)
				a = math.atan2(uy,ux)
				# leggo vertice seguente
				x3,y3 = p3.x(),p3.y()
				vx,vy = x3-x2,y3-y2
				mv = math.sqrt(vx**2+vy**2)
				b = math.atan2(vy,vx)
				g = math.pi+a-b
				if g < math.pi:
					g = g/2
				else:
					g = (2*math.pi-g)/2
				d = self.rad/math.tan(g)
				# calcolo vertice p21
				x21,y21 = x2-d*ux/mu,y2-d*uy/mu
				newPoint = QgsPoint(x21,y21)
				# creo segmento p1-p21
				newLine.append(newPoint)
				# calcolo vertice p23
				x23,y23 = x2+d*vx/mv,y2+d*vy/mv
				# punto medio
				x9,y9 = (x21+x23)/2,(y21+y23)/2
				tx,ty = x9-x2,y9-y2
				mt = math.sqrt(tx**2+ty**2)
				# 	centro
				s = math.sqrt(self.rad**2+d**2)
				x,y = x2+s*tx/mt,y2+s*ty/mt
				# creo arco p21-p23
				newList = arc(QgsPoint(x,y),newPoint,QgsPoint(x23,y23),num)
				newList.pop(0)	# toglie il primo che è un doppione di newPoint
				newLine.extend(newList)
				# aggiorno i vertici
				x1,y1 = x23,y23
				x2,y2 = x3,y3
			# completo la plinea
			newPoint = QgsPoint(x3,y3)
			newLine.append(newPoint)
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolyline(newLine))
			self.canvas.refresh()

	def pLineChamferTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineChamfer',
				"Chamfering a polyline (?!?!)"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			myDlg = genericDlg('pLineChamfer',['dimension'])
			myDlg.setValues([0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.cFer = float(tmp[0])
				if self.cFer > 0.0:
					cleanSelection(self)
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.pLineChamfer
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineChamfer",
						"Warning: the chamfer cannot be null"
					)
		else:
			self.warning(1121)

	def pLineChamfer(self,point):
		"""
			pLine splitting
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# leggo la polilinea
			geom = feat.geometry()
			line = geom.asPolyline()
			# cancello l'originale
			self.cLayer.deleteFeature(feat.id())
			# costruisco il nuovo
			p1 = line.pop(0)
			x1,y1 = p1.x(),p1.y()
			p2 = line.pop(0)
			x2,y2 = p2.x(),p2.y()
			# inizializzo la nuova plinea
			newLine = []
			newPoint = QgsPoint(x1,y1)
			newLine.append(newPoint)
			for p3 in line:
				# calcolo vertice p21
				dx,dy = x2-x1,y2-y1
				d = math.sqrt(dx**2+dy**2)
				x21,y21 = x2-self.cFer*dx/d,y2-self.cFer*dy/d
				newPoint = QgsPoint(x21,y21)
				# creo segmento p1-p21
				newLine.append(newPoint)
				# calcolo vertice p23
				x3,y3 = p3.x(),p3.y()
				dx,dy = x3-x2,y3-y2
				d = math.sqrt(dx**2+dy**2)
				x23,y23 = x2+self.cFer*dx/d,y2+self.cFer*dy/d
				newPoint = QgsPoint(x23,y23)
				# creo lato p21-p23
				newLine.append(newPoint)
				# aggiorno i vertici
				x1,y1 = x23,y23
				x2,y2 = x3,y3
			# completo la plinea
			newPoint = QgsPoint(x3,y3)
			newLine.append(newPoint)
			# aggiungo la feature
			newFeat = QgsFeature()
#			newFeat.addAttribute(0,"hello")
			newFeat.setGeometry(QgsGeometry.fromPolyline(newLine))
			self.cLayer.addFeatures( [ newFeat ] )

# ------------ quadratic interpolation points (circle) ------------------

	def pLineRegPolygTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineRegPolyg',
				"Polygon of N vertices with center C."
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# definisce i parametri
			dlg = genericDlg('pLineRegPolyg',['radius (m)'])
			dlg.setValues([0.0])
			dlg.show()
			result = dlg.exec_()
			if result:
				tmp = dlg.getValues()
				self.rad = float(tmp.pop())
				if self.rad > 0:
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					cleanSelection(self)
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.pLineRegPolyg
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineRegPolyg",
						"The radius must be greater than zero"
					)
		else:
			self.warning(1121)

	def pLineRegPolyg(self,point):
		"""
			Calcola un poligono regolare di n lati, con centro p e raggio r.
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				newLine = cerchio(p,self.rad,self.numPnts)	
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )

	def pLineCircle2PTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntCircle2P',
				"Circle with diameter in N1 and N2"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineCircle2P
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineCircle2P(self,point):
		"""
			Disegna il cerchio per 2 punti
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono lmeno 2 punti
		if len(self.selectList) >= 2:
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = cerchio2P(p1,p2,self.numPnts)
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"Circle2P",
					"Oops, the circle cannot be calculated: let controle the points"
				)
			cleanSelection(self)

	def pLineCircleInCbyPTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineCircleInCbyP',
				"Circle with center C by P"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineCircleInCbyP
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineCircleInCbyP(self,point):
		"""
			Disegna il cerchio per con centro C passante per P
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono lmeno 2 punti
		if len(self.selectList) >= 2:
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = cerchioInCperP(p1,p2,self.numPnts)	
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"CircleInCbyP",
					"Oops, the circle cannot be calculated: let controle the points"
				)
			cleanSelection(self)

	def pLineCircle3PTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineCircle3P',
				"Circle by 3 points"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineCircle3P
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineCircle3P(self,point):
		"""
			Disegna il cerchio per con centro C passante per P
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono lmeno 2 punti
		if len(self.selectList) >= 3:
			p3 = self.selectList.pop()
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = cerchio3P(p1,p2,p3,self.numPnts)
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"Circle3P",
					"Oops, the circle cannot be calculated: let controle the points"
				)
			cleanSelection(self)

# ------------ quadratic interpolation points (arc) ------------------

	def pLineArcTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntArc',
				"Arc with center C, radius R between angles alpha and BETA"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# definisce i parametri
			dlg = genericDlg('pntArc',['radius (m)','from (360)','to (360)'])
			dlg.setValues([0.0,0.0,0.0,10])
			dlg.show()
			result = dlg.exec_()
			if result:
				tmp = dlg.getValues()
				self.beta = float(tmp.pop())
				self.alfa = float(tmp.pop())
				self.rad = float(tmp.pop())
				if self.alfa != self.beta:
					if self.rad > 0:
						# try to disconnect all signals
						if self.isClickToolActivated:
							self.clickTool.canvasClicked.disconnect()
						cleanSelection(self)
						# connect to click signal
						QObject.connect(
							self.clickTool,
							SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
							self.pLineArc
						)
						self.isClickToolActivated = True
					else:
						QMessageBox.information(
							self.iface.mainWindow(),
							"pLineArc",
							"The tarting and ending angles cannot coincide"
						)
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineArc",
						"The radius cannot be null"
					)
		else:
			self.warning(1121)

	def pLineArc(self,point):
		"""
			Disegna arco di centro c, raggio rad, compreso
			fra l'angolo alfa e l'angolo beta.
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,c = tmp
				pM = pntHighlight(self.canvas,c.x(),c.y())
				self.markerList.append(pM)	
				newLine = arc1(c,self.rad,self.alfa,self.beta,self.numPnts)	
				# è valido?
				if len(newLine) > 0:
					feat = QgsFeature()
#					feat.addAttribute(0,"hello")
					feat.setGeometry(QgsGeometry.fromPolyline(newLine))
					self.cLayer.addFeatures( [ feat ] )
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineArc",
						"Give me a vertex"
					)
			cleanSelection(self)

	def pLineArc3PTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntArc',
				"Arc from p1 to p3 by p2"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineArc3P
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineArc3P(self,point):
		"""
			Disegna un arco per 3 punti compresi fra 1 e 3 punto
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono almeno 2 punti
		if len(self.selectList) >= 3:
			p3 = self.selectList.pop()
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = arc2(p1,p2,p3,self.numPnts)
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
				self.iface.mainWindow(),
				"pntArc3P",
				"The points must be all distinct"
				)	
			cleanSelection(self)

# ------------ quadratic interpolation points (parabole) ------------------

	def pLineParabola1Tool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntParabola1',
				"Parabole with focus F and directrix in H (symmetry axys FH)"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# definisce i parametri
			dlg = genericDlg('Parabola1',['domain'])
			dlg.setValues([0.0])
			dlg.show()
			result = dlg.exec_()
			if result:
				tmp = dlg.getValues()
				self.domX = float(tmp.pop())
				# try to disconnect all signals
				if self.isClickToolActivated:
					self.clickTool.canvasClicked.disconnect()
				cleanSelection(self)
				# connect to click signal
				QObject.connect(
					self.clickTool,
					SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
					self.pLineParabola1
				)
		else:
			self.warning(1121)

	def pLineParabola1(self,point):
		"""
			Calcola la semiparabola destra con fuoco
			in n1 e intercetta in n2;
			la parabola si estende dal fuoco fino all'ascissa b
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono almeno 2 punti
		if len(self.selectList) >= 2:
			i = self.selectList.pop()
			f = self.selectList.pop()
			newLine = parabola1(f,i,self.domX,self.numPnts)	
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"pLineParabola1",
					"Oops, the parabole cannot be calculated: let controle the points"
				)
			cleanSelection(self)

	def pLineParabola2Tool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineParabola2',
				"Parabole with focus F and apex in C (symmetry axys FC)"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# definisce i parametri
			dlg = genericDlg('Parabola2',['domain'])
			dlg.setValues([0.0])
			dlg.show()
			result = dlg.exec_()
			if result:
				tmp = dlg.getValues()
				self.domX = float(tmp.pop())
				# try to disconnect all signals
				if self.isClickToolActivated:
					self.clickTool.canvasClicked.disconnect()
				cleanSelection(self)
				# connect to click signal
				QObject.connect(
					self.clickTool,
					SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
					self.pLineParabola2
				)
		else:
			self.warning(1121)

	def pLineParabola2(self,point):
		"""
			Calcola n punti della semiparabola destra con fuoco
			in n1 e apice in n2;
			la parabola si estende dal fuoco fino all'ascissa b
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono almeno 2 punti
		if len(self.selectList) >= 2:
			a = self.selectList.pop()
			f = self.selectList.pop()
			newLine = parabola2(f,a,self.domX,self.numPnts)	
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"pLineParabola2",
					"Oops, the parabole cannot be calculated: let controle the points"
				)
			cleanSelection(self)

	def pLineParabola3Tool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntParabola3',
				"Parabole with focus F, axys of symmetry FG, by P"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineParabola3
			)
		else:
			self.warning(1121)

	def pLineParabola3(self,point):
		"""
			Calcola n punti della semiparabola destra con asse f-g
			con fuoco in f passante per p;
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono almeno 3 punti
		if len(self.selectList) >= 3:
			p = self.selectList.pop()
			g = self.selectList.pop()
			f = self.selectList.pop()
			newLine = parabola3(f,g,p,self.numPnts)
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"pLineParabola3",
					"Oops, the parabole cannot be calculated: let controle the points"
				)
			cleanSelection(self)

	def pLineParabola4Tool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntParabola4',
				"Parabole with vertex C, axys of symmetry CH, by P"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineParabola4
			)
		else:
			self.warning(1121)

	def pLineParabola4(self,point):
		"""
			Calcola n punti della semiparabola destra con asse c-h
			con vertice in c passante per p;
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono almeno 3 punti
		if len(self.selectList) >= 3:
			p = self.selectList.pop()
			h = self.selectList.pop()
			a = self.selectList.pop()
			newLine = parabola4(a,h,p,self.numPnts)	
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"pLineParabola4",
					"Oops, the parabole cannot be calculated: let controle the points"
				)
			cleanSelection(self)

# ------------ quadratic interpolation points (ellipsys) ------------------

	def pLineEllisseTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineEllisse',
				"Ellipsis with foci F1 e F2 and sum of distances D"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# definisce i parametri
			dlg = genericDlg('pLineEllisse',['sum of distances'])
			dlg.setValues([0.0,10])
			dlg.show()
			result = dlg.exec_()
			if result:
				tmp = dlg.getValues()
				self.dist = float(tmp.pop())
				if self.dist > 0:
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					cleanSelection(self)
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.pLineEllisse
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineEllisse",
						"sum of distances must be greater than zero"
					)
		else:
			self.warning(1121)

	def pLineEllisse(self,point):
		"""
			Calcola l'ellisse con fuochi in n1,n2 e somma delle distanza pari a d
			per un migliore rappresentazione il dominio X è diviso in n-5 tratti
			e 4 punti sono sollocati a
				-d/2+dx/4
				-d/2+dx/3
				d/2-dx/3
				d/2-dx/4.
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono almeno 2 punti
		if len(self.selectList) >= 2:
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = ellisse(p1,p2,self.dist,self.numPnts)
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"pLineEllisse",
					"Oops, the ellipsys cannot be calculated: let controle the points"
				)
			cleanSelection(self)

	def pLineEllisseByPTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineEllisseByP',
				"Ellipsis with foci F1 e F2 and sum of distances D"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineEllisseByP
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineEllisseByP(self,point):
		"""
			Calcola l'ellisse con fuochi in f1,f2 passante per P.
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# cerca il vertice
			tmp = searchPlineVertex(feat,point,self.eps)
			if len(tmp) == 2:
				pid,p = tmp
				self.selectList.append(p)
				pM = pntHighlight(self.canvas,p.x(),p.y())
				self.markerList.append(pM)	
		# controlla se ci sono almeno 3 punti
		if len(self.selectList) >= 3:
			p3 = self.selectList.pop()
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = ellisseByP(p1,p2,p3,self.numPnts)
			if len(newLine) > 0:
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolyline(newLine))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"pLineEllisseByP",
					"Oops, the ellipsys cannot be calculated: let controle the points"
				)
			cleanSelection(self)

# --------------------cubic interpolation points ------------------------

	def pLineHermiteTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineHermite',
				"Cubic interpolation of Hermite (Foley-VanDam-alt pag. 485)."
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			self.pLineHermite()
		else:
			self.warning(1121)

	def pLineHermite(self):
		"""
			Calcola l'interpolazione cubica hermitiana
			nel piano XY.
			(Foley-VanDam-alt pag. 485).
		"""
		# selezione dei punti da interpolare
		myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Hermite interpolation')
		myDlg.show()
		result = myDlg.exec_()
		if result:
			if len(self.selectList) >= 4:
				p4 = self.selectList.pop()
				p3 = self.selectList.pop()
				p2 = self.selectList.pop()
				p1 = self.selectList.pop()
				list = Hermite(p1,p2,p3,p4,self.numPnts)
				if len(list):
					newLine = []
					for [x,y] in list:
						newPoint = QgsPoint(x,y)
						newLine.append(newPoint)
					feat = QgsFeature()
#					feat.addAttribute(0,"hello")
					feat.setGeometry(QgsGeometry.fromPolyline(newLine))
					self.cLayer.addFeatures( [ feat ] )
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'pLineHermite',
						"The interpolation cannot take place: let controle the points."
					)
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineHermite',
					"Please, give me four points."
				)
		cleanSelection(self)

	def pLineBezierTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineBezier',
				"Cubic interpolation of Bezier (Foley-VanDam-alt pag. 485)."
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			self.pLineBezier()
		else:
			self.warning(1121)

	def pLineBezier(self):
		"""
			Calcola l'interpolazione cubica di Bezier;
			(Foley-VanDam-Feiner-Hughes pag. 485).
		"""
		# selezione dei punti da interpolare
		myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Bezier interpolation')
		myDlg.show()
		result = myDlg.exec_()
		if result:
			if len(self.selectList) >= 4:
				p4 = self.selectList.pop()
				p3 = self.selectList.pop()
				p2 = self.selectList.pop()
				p1 = self.selectList.pop()
				list = Bezier(p1,p2,p3,p4,self.numPnts)
				if len(list):
					newLine = []
					for [x,y] in list:
						newPoint = QgsPoint(x,y)
						newLine.append(newPoint)
					feat = QgsFeature()
#					feat.addAttribute(0,"hello")
					feat.setGeometry(QgsGeometry.fromPolyline(newLine))
					self.cLayer.addFeatures( [ feat ] )
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'pLineBezier',
						"The interpolation cannot take place: let controle the points."
					)
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineBezier',
					"Please, give me four points."
				)
		cleanSelection(self)

	def pLineUnRsplineTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntUnRspline',
				"Uniform nonRational Bspline (Foley-VanDam-Feiner-Hughes pag. 485)."
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			self.pLineUnRspline()
		else:
			self.warning(1121)

	def pLineUnRspline(self):
		"""
			Calcola l'interpolazione cubica uNrBspline;
			(Foley-VanDam-Feiner-Hughes pag. 485).
		"""
		# selezione dei punti da interpolare
		myDlg = pLine4qgisItemsSelectDlg(self,'vertex','UnRspline interpolation')
		myDlg.show()
		result = myDlg.exec_()
		if result:
			if len(self.selectList) >= 4:
				p4 = self.selectList.pop()
				p3 = self.selectList.pop()
				p2 = self.selectList.pop()
				p1 = self.selectList.pop()
				list = UnRspline(p1,p2,p3,p4,self.numPnts)
				if len(list):
					newLine = []
					for [x,y] in list:
						newPoint = QgsPoint(x,y)
						newLine.append(newPoint)
					feat = QgsFeature()
#					feat.addAttribute(0,"hello")
					feat.setGeometry(QgsGeometry.fromPolyline(newLine))
					self.cLayer.addFeatures( [ feat ] )
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'pLineBezier',
						"The interpolation cannot take place: let controle the points."
					)
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineBezier',
					"Please, give me four points."
				)
		cleanSelection(self)

	def pLineOverhauserTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntOverhauser',
				"Overhauser spline (Foley-VanDam-Feiner-Hughes pag. 485)."
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			self.pLineOverhauser()
		else:
			self.warning(1121)

	def pLineOverhauser(self):
		"""
			Calcola l'interpolazione cubica Overhauser Bspline (Catmul-Rom);
			(Foley-VanDam-Feiner-Hughes pag. 505).
		"""
		myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Overhauser interpolation')
		myDlg.show()
		result = myDlg.exec_()
		if result:
			if len(self.selectList) >= 4:
				p4 = self.selectList.pop()
				p3 = self.selectList.pop()
				p2 = self.selectList.pop()
				p1 = self.selectList.pop()
				list = Overhauser(p1,p2,p3,p4,self.numPnts)
				if len(list):
					newLine = []
					for [x,y] in list:
						newPoint = QgsPoint(x,y)
						newLine.append(newPoint)
					feat = QgsFeature()
#					feat.addAttribute(0,"hello")
					feat.setGeometry(QgsGeometry.fromPolyline(newLine))
					self.cLayer.addFeatures( [ feat ] )
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'pLineOverhauser',
						"The interpolation cannot take place: let controle the points."
					)
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineOverhauser',
					"Please, give me four points."
				)
		cleanSelection(self)

	def pLineBetaSplineTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntBetaSpline',
				"BetaSpline spline (Foley-VanDam-Feiner-Hughes pag. 505)."
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# definisce il grado del polinomio
			myDlg = genericDlg('BetaSpline',['bias','tension'])
			myDlg.setValues([0.5,0.5])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				tension = float(tmp.pop())
				bias = float(tmp.pop())
				self.pLineBetaSpline(bias,tension)
		else:
			self.warning(1121)

	def pLineBetaSpline(self,bias,tension):
		"""
			Calcola l'interpolazione cubica BetaSpline;
			(Foley-VanDam-Feiner-Hughes pag. 505).
		"""
		myDlg = pLine4qgisItemsSelectDlg(self,'vertex','BetaSpline interpolation')
		myDlg.show()
		result = myDlg.exec_()
		if result:
			if len(self.selectList) >= 4:
				p4 = self.selectList.pop()
				p3 = self.selectList.pop()
				p2 = self.selectList.pop()
				p1 = self.selectList.pop()
				list = BetaSpline(p1,p2,p3,p4,bias,tension,self.numPnts)
				if len(list):
					newLine = []
					for [x,y] in list:
						newPoint = QgsPoint(x,y)
						newLine.append(newPoint)
					feat = QgsFeature()
#					feat.addAttribute(0,"hello")
					feat.setGeometry(QgsGeometry.fromPolyline(newLine))
					self.cLayer.addFeatures( [ feat ] )
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'pLineBetaSpline',
						"The interpolation cannot take place: let controle the points."
					)
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineBetaSpline',
					"Please, give me four points."
				)
		cleanSelection(self)

# ------------------ generic interpolation pLines ------------------------

	def pLineInterpolationTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineInterpolation',
				"Generic interpolation"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			cleanSelection(self)
			# definisce il grado del polinomio
			myDlg = genericDlg('pLineInterpolation',['deg'])
			myDlg.setValues([2,10])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.deg = int(tmp.pop())
				if self.deg > 0:
					# la routine potrebbe essere monolitica; così forse risulta più leggibile
					self.pLineInterpolation()
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineInterpolation",
						"Warning: the polynomial degree cannot be null"
					)
		else:
			self.warning(1121)

	def pLineInterpolation(self):
		"""
			Calcola il polinomio di grado deg
			che interpola i punti dati
			rispetto ad un asse (2 punti) generico.
			Viene usato il metodo LSM.
			Il polinomio opera nel piano XY.
		"""
		# selezione dei punti da interpolare
		myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Interpolation points')
		myDlg.show()
		result = myDlg.exec_()
		if result:
			intPnts = self.selectList
			cleanSelection(self)
			if len(intPnts) > self.deg:
				# definisce l'asse di riferimento (2 punti)
				myDlg = pLine4qgisItemsSelectDlg(self,'vertex','axyx vertices')
				myDlg.show()
				result = myDlg.exec_()
				if result:
					axysPnts = self.selectList
					cleanSelection(self)
					if len(axysPnts) >= 2:
						newPnts = interpolation(intPnts,axysPnts,self.deg,self.numPnts)
						newLine = []
						for p in newPnts:
							x,y = p
							newPoint = QgsPoint(x,y)
							newLine.append(newPoint)
						feat = QgsFeature()
						feat.setGeometry(QgsGeometry.fromPolyline(newLine))
						self.cLayer.addFeatures( [ feat ] )
					else:
						QMessageBox.information(
							self.iface.mainWindow(),
							"pLineInterpolation",
							"The axys must be identified by two points"
						)
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"pLineInterpolation",
					"The number of points must be greater than deg"
				)

# ------------------ editing pLines ------------------------

	def pLineVrtsEditMouse(self):
		QMessageBox.information(
			self.iface.mainWindow(),
			'pLineVrtsEditMouse',
			"For analogic editing of polyline vertices please use the native command of qGis"
		)

	def pLineVrtsEditTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineVrtsEdit',
				"Polyline vertices editing"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineVrtsEdit
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineVrtsEdit(self,point):
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			title = "pLine %d" % (feat.id())
			myDlg = pLine4qgisPlineNavDlg(self,title,line)
			myDlg.show()
			result = myDlg.exec_()	# serve a fermare la finestra di dialogo
			if result:
				line = myDlg.line
				self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolyline(line))
				self.canvas.refresh()
			cleanSelection(self)

# ------------------ open/close pLines ------------------------

	def pLineOpenCloseTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineOpenClose',
				"Open/Close polylines"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineOpenClose
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineOpenClose(self,point):
		"""
			Open/Close polylines
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			m = pLineHighlight(self.canvas,feat)
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			if pLineIsClosed(feat):
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineOpenClose',
					"polyline chiusa: la apro"
				)
				line.pop()
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineOpenClose',
					"polyline aperta: la chiudo"
				)
				pnt = line[0]
				line.append(pnt)
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolyline(line))
			self.canvas.refresh()
			self.canvas.scene().removeItem(m)

# ------------------ intersection ------------------------

	def pLineIntersectionTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineUnion',
				"Polylines merging"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineIntersection
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineIntersection(self,point):
		"""
			Intersection of lines;
			limited to single lines
		"""
		searchPline(self,point)
		feat = QgsFeature()
		while self.provider.nextFeature(feat):
#			self.dlg.appendSel(feat.id())
			self.selectList.append(feat)
			m = pLineHighlight(self.canvas,feat)
			self.markerList.append(m)
			# eseguo quando sono due
			if len(self.selectList) >= 2:
				pl2 = self.selectList.pop()
				pl1 = self.selectList.pop()
				id1 = pl1.id()
				id2 = pl2.id()
				if id1 != id2:
					# legge la geometria
					geom = pl1.geometry()
					line1 = geom.asPolyline()
					if len(line1) <3:
						geom = pl2.geometry()
						line2 = geom.asPolyline()
						if len(line2) < 3:
							n4 = line2.pop()
							n3 = line2.pop()
							n2 = line1.pop()
							n1 = line1.pop()
							tmp = intersection(n1,n2,n3,n4)
							if tmp != -1:
								x,y = tmp
								newPoint = QgsPoint(x,y)
								# sistemo linea 1
								x1,y1 = n1.x(),n1.y()
								x2,y2 = n2.x(),n2.y()
								d1 = math.sqrt((x1-x)**2+(y1-y)**2)
								d2 = math.sqrt((x2-x)**2+(y2-y)**2)
								if d1 > d2:
									self.cLayer.changeGeometry(id1,QgsGeometry.fromPolyline([n1,newPoint]))
								else:
									self.cLayer.changeGeometry(id1,QgsGeometry.fromPolyline([n2,newPoint]))
								# sistemo linea 2
								x1,y1 = n3.x(),n3.y()
								x2,y2 = n4.x(),n4.y()
								d1 = math.sqrt((x1-x)**2+(y1-y)**2)
								d2 = math.sqrt((x2-x)**2+(y2-y)**2)
								if d1 > d2:
									self.cLayer.changeGeometry(id2,QgsGeometry.fromPolyline([n3,newPoint]))
								else:
									self.cLayer.changeGeometry(id2,QgsGeometry.fromPolyline([n4,newPoint]))
								self.canvas.refresh()
							else:
								QMessageBox.information(
									self.iface.mainWindow(),
									'pLineIntersection',
									'non esiste intersezione'
								)
						else:
							QMessageBox.information(
								self.iface.mainWindow(),
								'pLineIntersection',
								'line %d is complex' % (id2)
							)
					else:
						QMessageBox.information(
							self.iface.mainWindow(),
							'pLineIntersection',
							'line %d is complex' % (id1)
						)
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'pLineIntersection',
						"The polyline cannot self-intersect"
					)
				cleanSelection(self)
				return

# ------------------ union & splitting ------------------------

	def pLineUnionTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineUnion',
				"Polylines merging"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineUnion
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineUnion(self,point):
		"""
			Merge two polylines with a common extremity
		"""
		searchPline(self,point)
		feat = QgsFeature()
		while self.provider.nextFeature(feat):
#			self.dlg.appendSel(feat.id())
			self.selectList.append(feat)
			m = pLineHighlight(self.canvas,feat)
			self.markerList.append(m)
			# eseguo quando sono due
			if len(self.selectList) >= 2:
				pl2 = self.selectList.pop()
				pl1 = self.selectList.pop()
				id1 = pl1.id()
				id2 = pl2.id()
				if id1 != id2:
					# controlla le estremità
					xs1,ys1,xf1,yf1 = pLineStartEnd(pl1)
					xs2,ys2,xf2,yf2 = pLineStartEnd(pl2)
					# legge le geometrie
					geom = pl1.geometry()
					line1 = geom.asPolyline()
					geom = pl2.geometry()
					line2 = geom.asPolyline()
					if xf1 == xs2 and yf1 == ys2:
						# mette la 2 in coda alla 1
						line2.pop(0)	# elimina il primo punto perchè doppio
						line1.extend(line2)
						self.cLayer.deleteFeature(id2)
						self.cLayer.changeGeometry(id1,QgsGeometry.fromPolyline(line1))
					elif xs1 == xf2 and ys1 == yf2:
						# mette la 1 in coda alla 2
						line1.pop(0)	# elimina il primo punto perchè doppio
						line2.extend(line1)
						self.cLayer.deleteFeature(id1)
						self.cLayer.changeGeometry(id2,QgsGeometry.fromPolyline(line2))
					elif xf1 == xf2 and yf1 == yf2:
						# mette la 2 invertita in coda alla 1
						line2.reverse()
						line2.pop(0)	# elimina il primo punto perchè doppio
						line1.extend(line2)
						self.cLayer.deleteFeature(id2)
						self.cLayer.changeGeometry(id1,QgsGeometry.fromPolyline(line1))
					elif xs1 == xs2 and ys1 == ys2:
						# mette la 2 in coda alla 1 invertita
						line1.reverse()
						line2.pop(0)	# elimina il primo punto perchè doppio
						line1.extend(line2)
						self.cLayer.deleteFeature(id2)
						self.cLayer.changeGeometry(id1,QgsGeometry.fromPolyline(line1))
					else:
						QMessageBox.information(
							self.iface.mainWindow(),
							'pLineUnion',
							'polylines not contiguous'
						)
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'pLineUnion',
						"The polyline cannot be self-joined"
					)
				cleanSelection(self)
				return

	def pLineSplitTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineSplit',
				"Splitting of a polyline"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pLineSplit
			)
			self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineSplit(self,point):
		"""
			pLine splitting
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# leggo la polilinea
			geom = feat.geometry()
			line = geom.asPolyline()
			# cancello l'originale
			self.cLayer.deleteFeature(feat.id())
			# costruisco i singoli lati
			p1 = line.pop(0)
			x1,y1 = p1.x(),p1.y()
			for p in line:
				x2,y2 = p.x(),p.y()
				newFeat = QgsFeature()
#				newFeat.addAttribute(0,"hello")
				newFeat.setGeometry(QgsGeometry.fromPolyline([QgsPoint(x1,y1),QgsPoint(x2,y2)]))
				self.cLayer.addFeatures( [ newFeat ] )
				x1,y1 = x2,y2

# ------------------ transformations ------------------------

	def pLineTranslationTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineTranslation',
				"pLine translation by a vector"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			myDlg = genericDlg('pLineTranslation',['Dx (m)','Dy (m)'])
			myDlg.setValues([0.0,0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.Dy = float(tmp.pop())
				self.Dx = float(tmp.pop())
				if self.Dx == 0.0 and self.Dy == 0.0:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineTranslation",
						"Warning: the vector cannot be null"
					)
				else:
					# disconnect from click signal
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
						self.isClickToolActivated = False
			 		cleanSelection(self)
					# attiva il MapClicktTool
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.pLineTranslation
					)
					self.isClickToolActivated = True
		else:
			self.warning(1121)

	def pLineTranslation(self,point):
		# legge il selezionato
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			m = pLineHighlight(self.canvas,feat)
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			# costruisce la nuova
			newLine = []
			for p in line:
				newPoint = QgsPoint(p.x()+self.Dx, p.y()+self.Dy)
				newLine.append(newPoint)
				# aggiorn la geometria
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolyline(newLine))
			self.canvas.refresh()
			self.canvas.scene().removeItem(m)

	def pLineScalingTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineScaling',
				"pLine anysotropic scaling"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# selezione del centro di rotazione
			myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Symmetry center')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = self.selectList.pop()
				if type(tmp) == QgsPoint:
					self.x0,self.y0 = tmp.x(),tmp.y()
					# dialogo per paramentri
					myDlg = genericDlg('pLineScaling',['Sx','Sy'])
					myDlg.setValues([0.0,0.0])
					myDlg.show()
					result = myDlg.exec_()
					if result:
						tmp = myDlg.getValues()
						self.Sy = float(tmp.pop())
						self.Sx = float(tmp.pop())
						if self.Sx != 0.0 and self.Sy != 0.0:
							# disconnect from click signal
							if self.isClickToolActivated:
								self.clickTool.canvasClicked.disconnect()
								self.isClickToolActivated = False
					 		cleanSelection(self)
							# attiva il MapClicktTool
							QObject.connect(
								self.clickTool,
								SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
								self.pLineScaling
							)
							self.isClickToolActivated = True
						else:
							QMessageBox.information(
							self.iface.mainWindow(),
								"pLineScaling",
								"Warning: the scaling factors cannot be null"
							)
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineScaling",
						"Center invalid"
					)
		else:
			self.warning(1121)

	def pLineScaling(self,point):
		"""
			pLine scaling
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			# costruisce la nuova
			newLine = []
			for p in line:
				x,y = self.x0+(p.x()-self.x0)*self.Sx, self.y0+(p.y()-self.y0)*self.Sy	# uso forma semplificata
				newPoint = QgsPoint(x,y)
				newLine.append(newPoint)
			# aggiorna la geometria
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolyline(newLine))
			self.canvas.refresh()

	def pLineRotationTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineRotation',
				"pLine rotation"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
			# selezione del centro di rotazione
			myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Rotation center')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = self.selectList.pop()
				if type(tmp) == QgsPoint:
					x0,y0 = tmp.x(),tmp.y()
					# dialogo per paramentri
					myDlg = genericDlg('pLineRotation',['angle'])
					myDlg.setValues([0.0])
					myDlg.show()
					result = myDlg.exec_()
					if result:
						tmp = myDlg.getValues()
						angle = float(tmp.pop())
						if angle != 0.0:
							# prepara matrici di trasformazione
							da = trans2D.sessag2rad(angle)	# converte in radianti
							mat1 = trans2D.translation(-x0,-y0)
							mat2 = trans2D.rotation(da)
							mat = np.dot(mat2,mat1)
							mat3 = trans2D.translation(x0,y0)
							self.mat = np.dot(mat3,mat)
							# disconnect from click signal
							if self.isClickToolActivated:
								self.clickTool.canvasClicked.disconnect()
								self.isClickToolActivated = False
					 		cleanSelection(self)
							# attiva il MapClicktTool
							QObject.connect(
								self.clickTool,
								SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
								self.pLineRotation
							)
							self.isClickToolActivated = True
						else:
							QMessageBox.information(
							self.iface.mainWindow(),
								"pLineRotation",
								"Warning: the rotation angle cannot be null"
							)
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineScaling",
						"Center invalid"
					)
		else:
			self.warning(1121)

	def pLineRotation(self,point):
		"""
			pLine rotation
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			# costruisce la nuova
			newLine = []
			for p in line:
				x,y,w = np.dot(self.mat,[p.x(),p.y(),1])
				newPoint = QgsPoint(x,y)
				newLine.append(newPoint)
			# aggiorna la geometria
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolyline(newLine))
			self.canvas.refresh()

# ------------------ collimations ------------------------

	def pLineCollimation1PTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineCollimation1P',
				"Aligns a number of lines by making a matching pair of points"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
#			selezione della coppia di collimazione
			myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Matching pair')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 2:
					# preleva la coppia di punti
					p2 = self.selectList.pop()
					p1 = self.selectList.pop()
					x1,y1 = p1.x(),p1.y()
					x2,y2 = p2.x(),p2.y()
					# calcola lo spostamento
					self.dx,self.dy = x2-x1,y2-y1
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.pLineCollimation1P
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineAxialMirror",
						"Axys invalid"
					)
		else:
			self.warning(1121)

	def pLineCollimation1P(self,point):
		"""
			Esegue la collimazione di 1 punto
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			# costruisce la nuova
			newLine = []
			for p in line:
				x,y = p.x(),p.y()
				x,y = x+self.dx,y+self.dy
				newLine.append(QgsPoint(x,y))
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolyline(newLine))
			self.canvas.refresh()
# serve?		cleanSelection(self)

	def pLineCollimation2PTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineCollimation2P',
				"Aligns a number of points by making two matching pairs"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
#			selezione delle coppie di collimazione
			myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Two matching pairs')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 4:
					# preleva le coppie di punti
					d2 = self.selectList.pop()
					d2x,d2y = d2.x(),d2.y()
					s2 = self.selectList.pop()
					s2x,s2y = s2.x(),s2.y()
					d1 = self.selectList.pop()
					d1x,d1y = d1.x(),d1.y()
					s1 = self.selectList.pop()
					s1x,s1y = s1.x(),s1.y()
					if s1 == d1 and s2 == d2:
						QMessageBox.information(
							self.iface.mainWindow(),
							'pLineCollimation1P',
							"It doesn't make any sense to have coinciding points to correspond."
						)
					else:
						# calcola matrice di trsformazione
						# trasla n1 nell'origine
						mat = trans2D.translation(-s1x,-s1y)
						# ruota la linea n1-n2 sull'orizzontale
						a1 = math.atan2(s2y-s1y,s2x-s1x)
						# ruota per allinearla a P1-P2
						a2 = math.atan2(d2y-d1y,d2x-d1x)
						# applica la rotazione complessiva
						mat1 = trans2D.rotation(a2-a1)
						mat = np.dot(mat1,mat)
						# trasla nel punto P1
						mat1 = trans2D.translation(d1x,d1y)
						self.mat = np.dot(mat1,mat)
						# try to disconnect all signals
						if self.isClickToolActivated:
							self.clickTool.canvasClicked.disconnect()
						# connect to click signal
						QObject.connect(
							self.clickTool,
							SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
							self.pLineCollimation2P
						)
						self.isClickToolActivated = True
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					'pLineCollimation2P',
					"Please insert four points, two source and two destination."
				)
		else:
			self.warning(1121)

	def pLineCollimation2P(self,point):
		"""
			Collimazione con coppia di punti
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			# costruisce la nuova
			newLine = []
			for p in line:
				x,y = p.x(),p.y()
				x,y,w = np.dot(self.mat,[x,y,1])
				newLine.append(QgsPoint(x,y))
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolyline(newLine))
			self.canvas.refresh()

	def pLineCollimationNPTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineCollimationNP',
				"Aligns a number of points by making three or more matching pairs"
			)
			self.helpFlag = False
		if self.goFlag >= 1121:
#			selezione delle coppie di collimazione
			myDlg = pLine4qgisItemsSelectDlg(self,'vertex','Three or more matching pairs')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				numCoppie = len(self.selectList)/2
				if len(self.selectList)-2*numCoppie == 0:
					if numCoppie >= 3:
						# prepara le liste dei punti di collimazione
						oldCdsList = []
						newCdsList = []
						# preleva le coppie (A RITROSO!)
						for i in range(numCoppie):
							p = self.selectList.pop()
							x,y = p.x(),p.y()
							newCdsList.append([x,y])
							p = self.selectList.pop()
							x,y = p.x(),p.y()
							oldCdsList.append([x,y,1])	# qui va aggiunta la coordinata omogenea
# mettere il rbBand fra i punti della coppia?

# meglio controllare che siano distinti
						# calcola i parametri di trasformazione mediante LSM
						# probabilmente si potrebbe fre tutto in una passata
						# ma in questa prima implementazione con numpy preferisco essere cauto
						X = []
						Y = []
						for p in newCdsList:
							X.append(p[0])
							Y.append(p[1])
						sol1 = np.linalg.lstsq(oldCdsList,X)
						# print sol1
						sol2 = np.linalg.lstsq(oldCdsList,Y)
						# print sol2
						# costruisce la matrice di trasformazione (2x3)
						self.mat = np.matrix([[sol1[0][0],sol1[0][1],sol1[0][2]],[sol2[0][0],sol2[0][1],sol2[0][2]]])
						# try to disconnect all signals
						if self.isClickToolActivated:
							self.clickTool.canvasClicked.disconnect()
						cleanSelection(self)
						# connect to click signal
						QObject.connect(
							self.clickTool,
							SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
							self.pLineCollimationNP
						)
						self.isClickToolActivated = True
						cleanSelection(self)
					else:
						QMessageBox.information(
							self.iface.mainWindow(),
							"pLineCollimationNP",
							"Warning: are needed three or more couples of points"
						)
						cleanSelection(self)
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineCollimationNP",
						"Error: the number of points must be even"
					)
		else:
			self.warning(1111)

	def pLineCollimationNP(self,point):
		"""
			Esegue la collimazione dei punti contenuti in plist
			in modo che assumano i punti dell srcList
			vdno  collimare con quelli contenuti in destList.
			Usa il LSM.
			
			ATTENZIONE: la routine è molto simile a Collimation2P, però
			la 2P opera con una matrice 3x3, la NP con matrice 2x3 e poi la prima dà
			direttamente le coordinate x,y, la seconda attraverso la lista cds (altrimenti
			dà errore); se si allineano possono esserefuse in una sola routine.
		"""
		searchPline(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			line = geom.asPolyline()
			# costruisce la nuova
			newLine = []
			for p in line:
				cds = np.dot(self.mat,[p.x(),p.y(),1])
				newLine.append(QgsPoint(cds[0,0],cds[0,1]))
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolyline(newLine))
			self.canvas.refresh()



