# -*- coding: utf-8 -*-

"""
	vers. 000 2012-08-05
"""

# Import standard libraries
import math

# Import the PyQt and QGIS libraries
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from qgis.gui import *

# Import the NumPy library
import numpy as np

# Import custom libraries
import matrixTransformation as trans2D

def about(mw):
	"""
		Visualizza info sulla procedura
	"""
	QMessageBox.about(	#QtGui.
		mw,
		'About',
		"""
CAD tools for qGis
----------------------------
version:    012
build_date: 11 agosto 2012
author:     giuliano curti
email:      giulianc51 at gmail dot com
copyright:  2012-2017 giuliano curti
license:    GPL v2 (http://www.gnu.org/licenses/gpl-2.0.html)
		"""
	)

def info(mw):
	"""
		First advice to users
	"""
	QMessageBox.about(	#QtGui.
		mw,
		'Info',
		"""
(some) CAD tools for qGis (012).
--------------------------------------------------------
The procedure allows inserting/
editing points & lines with different
methods from the analytic geometry;

preselecting the HELP button give You
some info on the chosen function;

the procedure manipulate only vector
layer of type:
- 0 (points) and
- 1 (linestring);
is left to the user the selection of
the rigth layer type :-)

This procedure is EXPERIMENTAL; it
might contains many bugs, few
duplications and some mistakes, only
in part known to the author;

please inform me
(giulianc51 at gmail dot com)
about any encountered problems.
		"""
	)

def pntHighlight(canvas,x,y):
	"""
		attiva il marcatore del punto
	"""
	marker = QgsVertexMarker(canvas)
	marker.setIconType(QgsVertexMarker.ICON_CROSS)
	marker.setColor(QColor(0,255,0))
	marker.setIconSize(15)
	marker.setCenter(QgsPoint(x,y))
	marker.show()
	return marker

def edgeHighlight(canvas,p1,p2):
	"""
		marca il segmento p1-p2
	"""
	marker = QgsRubberBand(canvas)
	marker.setWidth(2)
	marker.setColor(QColor(0,0,255))
	marker.show()
	marker.addPoint(p1)
	marker.addPoint(p2)
	return marker

def pLineHighlight(canvas,feat):
	"""
		marca la polyline
	"""
	geom = feat.geometry()
	pline = geom.asPolyline()
	marker = QgsRubberBand(canvas)
	marker.setWidth(1)
	marker.setColor(QColor(255,0,0))
	marker.show()
	for p in pline:
		marker.addPoint(p)
	return marker

def cleanSelection(parent):
	"""
		Pulisce lo stack dei selezionati e toglie i marker;
		presume che il parent abbia tutti questi oggetti
	"""
	# azzera la selection list
	parent.selectList = []
#	parent.dlg.setSel('')
	# elimina i marcatori
	for m in parent.markerList:
		parent.canvas.scene().removeItem(m)
	parent.markerList = []

def edgeLengthAndInclin(p1,p2):
	"""
		Length nd inclination of an edge from P1 to P2
	"""
	dx,dy = p2.x()-p1.x(),p2.y()-p1.y()
	d = math.sqrt(dx**2+dy**2)
	a = trans2D.rad2sessag(math.atan2(dy,dx))
	return d,a

def pLineStartEnd(feat):
	"""
		Start and ending points of a polyline
	"""
	# legge la geometria
	geom = feat.geometry()
	line = geom.asPolyline()
	# scandisce i vertici
	start = line[0]
	xStart,yStart = start.x(),start.y()
	end = line[-1]
	xEnd,yEnd = end.x(),end.y()
	return (xStart,yStart,xEnd,yEnd)

def searchPline(parent,point):
	"""
		Cerca le polilinee individuate con il mouse;
		viene usato da pLine4qgisEdgeSelectDlg()
	"""
	# setup the provider select to filter results based on a rectangle
	pntGeom = QgsGeometry.fromPoint(point)  
	# scale-dependent buffer of 2 pixels-worth of map units
	pntBuff = pntGeom.buffer(parent.eps,0) 
	rect = pntBuff.boundingBox()
	# create the select statement
	parent.cLayer.select([],rect,True,True)	# prende quelli che intersecano

def searchPlineEdge(feat,point,eps):
	"""
		individua il (primo) segmento della polilinea;
		se l'utente clicca un vertice, verrà selezionato il primo dei segmenti incidenti
	"""
	# legge la geometria
	geom = feat.geometry()
	line = geom.asPolyline()
	# scandisce i vertici
	edg = 0
	x0,y0 = point.x(),point.y()
	p1 = line.pop(0)
	for p2 in line:
		x1,y1 = p1.x(),p1.y()
		x2,y2 = p2.x(),p2.y()
		# bounding box test
		xMin,xMax = min(x1,x2),max(x1,x2)
		yMin,yMax = min(y1,y2),max(y1,y2)
		if x0 >= xMin and x0 <= xMax and y0 >= yMin and y0 <= yMax:
			# calcolo piede perpendicolare dal punto con metodo vettoriale
			# questo probbilmente non è l'algoritmo più efficiente
			dux = x0 - x1
			duy = y0 - y1
			dx = x2 - x1
			dy = y2 - y1
			mw = math.sqrt(dx**2+dy**2)
			a = (dux*dx + duy*dy) / mw
			x = x1 + a*dx/mw
			y = y1 + a*dy/mw
			# calcola distanza
			dx,dy = x-x0,y-y0
			d = math.sqrt(dx**2+dy**2)
			if d <= eps:
				return [edg,[p1,p2]]
		edg += 1
		p1 = p2
	return []

def searchPlineVertex(feat,point,eps):
	"""
		individua il vertice della polilinea;
		restituisce  id's della linea e del vertice
	"""
	# legge la geometria
	geom = feat.geometry()
	line = geom.asPolyline()
	# scandisce i vertici
	idx = 0
	for p in line:
		if abs(point.x()-p.x()) <= eps and abs(point.y()-p.y()) <= eps:
			return [idx,p]
		idx += 1
	return []

def pLineIsClosed(feat):
	"""
		Controlla se la polilinea è chiusa
	"""
	# calcola gli estremi
	x1,y1,x2,y2 = pLineStartEnd(feat)
	if x1 == x2 and y1 == y2:
		return 1
	return 0

def interpolation(intPnts,axysPnts,deg,numPnt):
	"""
		Calcola l'interpolazione fra i punti dati in intPnts riferita all'asse axysPnts
	"""
	p2 = axysPnts.pop()
	p1 = axysPnts.pop()
	d,angle = edgeLengthAndInclin(p1,p2)
	# prepara la lista delle vecchie coordinate
	xList = []
	yList = []
	for p in intPnts:
		x,y = p.x(),p.y()
		xList.append(x)
		yList.append(y)
	# calcola il vettore di traslazione
	xMin = min(xList)
	yMin = min(yList)
	# trasla e ruota
	matT = trans2D.translation(-xMin,-yMin)
	alfa = trans2D.sessag2rad(angle)
	matR = trans2D.rotation(-alfa)
	mat = np.dot(matR,matT)
	# trasforma le coordinate
	for i in range(len(xList)):
		x,y = xList[i],yList[i]
		x,y,w = np.dot(mat,[x,y,1])
		xList[i],yList[i] = x,y
	# esegue calcolo
	sol = np.polynomial.polynomial.Polynomial.fit(xList,yList,deg)
	# dominio trasformato
	xMax = max(xList)
	# matrice inversa
	mat = np.linalg.inv(mat)
	# costruisce i punti interpolanti
	newList = []
	for x in np.linspace(0,xMax,numPnt):
		y = sol(x)
		# ritrasforma le coordinate
		x,y,w = np.dot(mat,[x,y,1])
		newList.append([x,y])
	return newList

def intersection(n1,n2,n3,n4):
	"""
		Intersezione fra i lati (n1-n2) e (n3-n4)
	"""
	# legge le coordinate
	x1,y1 = n1.x(),n1.y()
	x2,y2 = n2.x(),n2.y()
	x3,y3 = n3.x(),n3.y()
	x4,y4 = n4.x(),n4.y()
	# costruisce i vettori
	ux,uy = x2-x1,y2-y1
	vx,vy = x4-x3,y4-y3
	# print "1 vettore: ux=%5.2f uy=%5.2f" % (ux,uy)
	# print "2 vettore: vx=%5.2f vy=%5.2f" % (vx,vy)
	if ux == 0:
		if vx == 0:
			# print "1 non esiste intersezione"
			return -1
		else:
			x = x1
			b = (x1-x3)/vx
			y = y3+b*vy
			# print '2 b,x,y',b,x,y
	elif uy == 0:
		if vy == 0:
			# print "3 non esiste intersezione"
			return -1
		else:
			y = y1
			b = (y1-y3)/vy
			x = x3+b*vx
			# print '4 b,x,y',b,x,y
	else:
		if vy == 0:
			y = y3
			a = (y3-y1)/uy
			x = x1+a*ux
			# print '5 a,x,y',a,x,y
		elif vx == 0:
			x = x3
			a = (x3-x1)/ux
			y = y1+a*uy
			# print '6 a,x,y',a,x,y
		else:
			b = ((y1-y3)*ux+(x3-x1)*uy)/(ux*vy-uy*vx)
			x = x3+b*vx
			y = y3+b*vy
			# print '7 b,x,y',b,x,y
	return x,y

def intersection1(n1,n2,n3,n4):
	"""
		Intersezione fra i lati (n1-n2) e (n3-n4);
		controlla che il punto sia interno ai segmenti
	"""
	tmp = intersection(n1,n2,n3,n4)
	if tmp != -1:
		x,y = tmp	# intersezione valida
		# legge le coordinate
		x1,y1 = n1.x(),n1.y()
		x2,y2 = n2.x(),n2.y()
		x3,y3 = n3.x(),n3.y()
		x4,y4 = n4.x(),n4.y()
		if x >= min(x1,x2) and x <= max(x1,x2) and y >= min(y1,y2) and y <= max(y1,y2):
			if x >= min(x3,x4) and x <= max(x3,x4) and y >= min(y3,y4) and y <= max(y3,y4):
				return x,y
		return -1

def cerchio(c,rad,nPnts):
	"""
		Calcola un cercio di centro c e raggio rad
	"""
	list = []
	x0,y0 = c.x(),c.y()
	da = 2*math.pi/nPnts
	for i in range(0,nPnts+1):
		a = (i+1)*da
		x,y = x0+rad*math.cos(a),y0+rad*math.sin(a)
		list.append(QgsPoint(x,y))
	return list

def cerchio2P(p1,p2,nPnts):
	"""
		Cerchio per 2 punti
	"""
	list = []
	if p1 != p2:
		x1,y1 = p1.x(),p1.y()
		x2,y2 = p2.x(),p2.y()
		# centro
		x0,y0 = (x1+x2)/2,(y1+y2)/2
		# raggio
		rad = math.sqrt((x1-x0)**2+(y1-y0)**2)
		# esegue calcolo
		c = QgsPoint(x0,y0)
		list = cerchio(c,rad,nPnts)
	return list

def cerchioInCperP(c,p,nPnts):
	"""
		Disegna il cerchio per con centro C passante per P
	"""
	list = []
	if c != p:
		x0,y0 = c.x(),c.y()
		x1,y1 = p.x(),p.y()
		# raggio
		r = math.sqrt((x1-x0)**2+(y1-y0)**2)
		c = QgsPoint(x0,y0)
		list = cerchio(c,r,nPnts)
	return list

def cerchio3P(p1,p2,p3,nPnts):
	"""
		Cerchio per 3 punti
	"""
	list = []
	if p1 != p2 and p1 != p3 and p2 != p3:
		x1,y1 = p1.x(),p1.y()
		x2,y2 = p2.x(),p2.y()
		x3,y3 = p3.x(),p3.y()
		# coefficienti
		dx12,dy12 = x2-x1,y2-y1
		dx23,dy23 = x3-x2,y3-y2
		dx31,dy31 = x1-x3,y1-y3
		c = 2*dx12*dy23 - 2*dx23*dy12	# denominatore
		a = (-dy23*dy31 - dx23*dx31) / c
		# centro
		x0,y0 = (x1+x2)/2-a*dy12,(y1+y2)/2+a*dx12
		# raggio
		r = math.sqrt((x1-x0)**2+(y1-y0)**2)
		# esegue calcolo
		c = QgsPoint(x0,y0)
		list = cerchio(c,r,nPnts)
	return list

def arc(c,p1,p2,num):
	"""
		Disegna l'arco di num punti, con centro c, da p1 a p2.
	"""
	x0,y0 = c.x(),c.y()
	x1,y1 = p1.x(),p1.y()
	x2,y2 = p2.x(),p2.y()
	rad = (math.sqrt((x1-x0)**2+(y1-y0)**2)+math.sqrt((x2-x0)**2+(y2-y0)**2))/2
	alfa = math.atan2(y1-y0,x1-x0)
	beta = math.atan2(y2-y0,x2-x0)
	da = (beta-alfa) / (num-1)
	list = []
	for i in range (0,num):
		x2,y2 = x0+rad*math.cos(alfa+i*da),y0+rad*math.sin(alfa+i*da)
		list.append(QgsPoint(x2,y2))
	return list

def arc1(c,rad,alfa,beta,num):
	"""
		Disegna l'arco di num punti, con centro c, raggio rad,
		dall'angolo alfa a beta.
	"""
	alfa = trans2D.sessag2rad(alfa)
	beta = trans2D.sessag2rad(beta)
	da = (beta-alfa)/num
	x0,y0 = c.x(),c.y()
	list = []
	for i in range (num+1):
		c = alfa + i*da
		x = x0 + rad*math.cos(c)
		y = y0 + rad*math.sin(c)
		list.append(QgsPoint(x,y))
	return list

"""
eventuale versione alternativa

def arc1(c,r,a,b,nPnts):

#		Arco di centro c, raggio r, dall'angolo a all'angolo b

	x0,y0 = p.x(),p.y()
	# esegue calcolo
	alfa = self.sessag2rad(a)
	beta = self.sessag2rad(b)
	p1 = QgsPoint(x0 + rad*math.cos(alfa),y0 + rad*math.sin(alfa))
	p2 = QgsPoint(x0 + rad*math.cos(beta),y0 + rad*math.sin(beta))
	list = arc(c,p1,p2,nPnts)
	return list

"""

def arc2(p1,p2,p3,num):
	"""
		Arco per 3 punti
	"""
	# controlla i punti coincidenti
	list = []
	if p1 != p2 and p1 != p3 and p2 != p3:
		x3,y3 = p3.x(),p3.y()
		x2,y2 = p2.x(),p2.y()
		x1,y1 = p1.x(),p1.y()
		# coefficienti
		dx12,dy12 = x2-x1,y2-y1
		dx23,dy23 = x3-x2,y3-y2
		dx31,dy31 = x1-x3,y1-y3
		c = 2*dx12*dy23 - 2*dx23*dy12	# denominatore
		a = (-dy23*dy31 - dx23*dx31) / c
		# centro
		x0,y0 = (x1+x2)/2-a*dy12, (y1+y2)/2+a*dx12
		# raggio
		rad = math.sqrt((x1-x0)**2+(y1-y0)**2)
		list = arc(QgsPoint(x0,y0),p1,p3,num)
	return list

def parabola1(f,i,b,nPnts):
	"""
		Semiparabola destra con fuoco in f e intercetta in i sul dominio b
	"""
	list = []
	# controlla i punti coincidenti
	if f != i:
		# fuoco
		x1,y1 = f.x(),f.y()
		# intercetta
		x2,y2 = i.x(),i.y()
		# calcola i vettori normalizzati parallelo e ortogonale agli assi principali
		ux,uy = x1-x2,y1-y2
		d = math.sqrt(ux**2+uy**2)
		# controlla il fuoco non sia sulla retta
		if d:
			ux,uy = ux/d,uy/d
			vx,vy = -uy,ux
			cx,cy = (x1+x2)/2,(y1+y2)/2
			c = d/2
			a = 1/(4*c)
			# print ux,uy,uz,"\n",vx,vy,vz
			# inizio calcolo
			for i in range(nPnts+1):
				x = i*b/nPnts
				y = a*x**2
				px = cx - x*vx + y*ux
				py = cy - x*vy + y*uy
				list.append(QgsPoint(px,py))
	return list

def parabola2(f,a,b,nPnts):
	"""
		Semiparabola destra con fuoco in f e apice in a sul dominio b
	"""
	# fuoco
	x1,y1 = f.x(),f.y()
	# intercetta
	x2,y2 = a.x(),a.y()
	x,y = 2*x2-x1,2*y2-y1
	list = parabola1(f,QgsPoint(x,y),b,nPnts)	
	return list

def parabola3(f,g,p,nPnts):
	"""
		Semiparabola destra con asse f-g,  fuoco in f passante per p;
	"""
	list = []
	# controlla i punti coincidenti
	if f != g and f != p and g != p:
		fx,fy = f.x(),f.y()
		gx,gy = g.x(),g.y()
		px,py = p.x(),p.y()
		# calcola i vettori normalizzati parallelo e ortogonale agli assi principali
		ux,uy = gx-fx,gy-fy
		d = math.sqrt(ux**2+uy**2)
		# controlla il punto non coincida con il fuoco
		if d:
			ux,uy = ux/d,uy/d
			vx,vy = -uy,ux
			d = math.sqrt((px-fx)**2+(py-fy)**2)
			b = (px-fx)*ux+(py-fy)*uy
			e = (px-fx)*vx+(py-fy)*vy
			c = (d-b)/2
			cx,cy = fx-c*ux,fy-c*uy
			a = 1/(4*c)
			# inizio calcolo
			for i in range(nPnts+1):
				x = i*e/nPnts
				y = a*x**2
				px = cx + x*vx + y*ux
				py = cy + x*vy + y*uy
				list.append(QgsPoint(px,py))
	return list

def parabola4(c,h,p,nPnts):
	"""
		Semiparabola destra con asse c-h, apice in c passante per p;
	"""
	list = []
	# preleva i punti
	cx,cy = c.x(),c.y()
	hx,hy = h.x(),h.y()
	px,py = p.x(),p.y()
	# calcola i vettori normalizzati parallelo e ortogonale agli assi principali
	vx,vy = hx-cx,hy-cy
	mv = math.sqrt(vx**2+vy**2)
	# controlla il punto non coincida con il fuoco
	if mv:
		vx,vy = vx/mv,vy/mv
		ux,uy = vy,-vx
		d = math.sqrt((px-cx)**2+(py-cy)**2)
		dy = np.vdot([px-cx,py-cy],[vx,vy])
		dx = math.sqrt(d**2-dy**2)
		a = dy/dx**2
		# inizio calcolo
		for i in range(nPnts+1):
			x = i*dx/nPnts
			y = a*x**2
			px = cx + x*ux + y*vx
			py = cy + x*uy + y*vy
			list.append(QgsPoint(px,py))
	return list

def ellisse(f1,f2,d,nPnts):
	"""
		Calcola l'ellisse con fuochi in f1,f2 e somma delle distanza pari a d;
		per migliorare l'pprossimazione calcola il doppio dei punti utente.
	"""
	list = []
	if f1 != f2:
		x1,y1 = f1.x(),f1.y()
		x2,y2 = f2.x(),f2.y()
		# calcola il centro dell'ellisse
		cx,cy = (x1+x2)/2,(y1+y2)/2
		# calcola i vettori normalizzati parallelo e ortogonale agli assi principali
		ux,uy = x2-x1,y2-y1
		mu = math.sqrt(ux**2+uy**2)
		# controlla i dati
		if d > mu:
			ux,uy = ux/mu,uy/mu
			vx,vy = -uy,ux
			# calcola alcune costanti
			a = d/2
			f2 = (x1-cx)**2+(y1-cy)**2
			b2 = a**2-f2
			for i in range(2*nPnts+1):
				x = (-1+float(i)/nPnts) * a
				y = math.sqrt(b2*(a**2-x**2)/a**2)
				# poi la trasforma (traslazione e rotazione) correttamente
				px,py = cx+x*ux+y*vx,cy+x*uy+y*vy
				list.append(QgsPoint(px,py))
	return list

def ellisseByP(f1,f2,p,nPnts):
	"""
		Calcola l'ellisse con fuochi in f1,f2 passante per p
	"""
	x1,y1 = f1.x(),f1.y()
	x2,y2 = f2.x(),f2.y()
	x,y = p.x(),p.y()
	d = math.sqrt((x-x1)**2+(y-y1)**2) + math.sqrt((x-x2)**2+(y-y2)**2)
	list = ellisse(f1,f2,d,nPnts)
	return list

def Hermite(p1,p2,p3,p4,nPnts):
	"""
		interpolazione di Hermite
	"""
	list = []
	# controlla i punti
	if p1 != p2 and p2 != p3 and p3 != p4:
		# preleva cds dei punti
		p1x,p1y = p1.x(),p1.y()
		p2x,p2y = p2.x(),p2.y()
		p3x,p3y = p3.x(),p3.y()
		p4x,p4y = p4.x(),p4.y()
		# esegue calcolo
		r1x,r1y = p2x-p1x,p2y-p1y
		r4x,r4y = p4x-p3x,p4y-p3y
		for i in range (nPnts+1):
			t = float(i)/nPnts
			# blending functions
			bf1 = 2*t**3 - 3*t**2 + 1
			bf2 = t**2 * (3-2*t)
			bf3 = t * (1-t)**2
			bf4 = t**2 * (t-1)
			x = bf1*p1x + bf2*p4x + bf3*r1x + bf4*r4x
			y = bf1*p1y + bf2*p4y + bf3*r1y + bf4*r4y
			list.append([x,y])
	return list

def Bezier(p1,p2,p3,p4,nPnts):
	"""
		interpolazione di Bezier
	"""
	list = []
	# controlla i punti
	if p1 != p2 and p2 != p3 and p3 != p4:
		# preleva cds dei punti
		p1x,p1y = p1.x(),p1.y()
		p2x,p2y = p2.x(),p2.y()
		p3x,p3y = p3.x(),p3.y()
		p4x,p4y = p4.x(),p4.y()
		# esegue calcolo
		for i in range (nPnts+1):
			t = float(i)/nPnts
			# blending functions
			bf1 = (1-t)**3
			bf2 = 3*t * (1-t)**2
			bf3 = 3*t**2 * (1-t)
			bf4 = t**3
			x = bf1*p1x + bf2*p2x + bf3*p3x + bf4*p4x
			y = bf1*p1y + bf2*p2y + bf3*p3y + bf4*p4y
			list.append([x,y])
	return list

def UnRspline(p1,p2,p3,p4,nPnts):
	"""
		interpolazione UnRspline
	"""
	list = []
	# controlla i punti
	if p1 != p2 and p2 != p3 and p3 != p4:
		# preleva cds dei punti
		p1x,p1y = p1.x(),p1.y()
		p2x,p2y = p2.x(),p2.y()
		p3x,p3y = p3.x(),p3.y()
		p4x,p4y = p4.x(),p4.y()
		# esegue calcolo
		r1x,r1y = p2x-p1x,p2y-p1y
		r4x,r4y = p4x-p3x,p4y-p3y
		for i in range (nPnts+1):
			t = float(i)/nPnts
			# blending functions
			bf1 = (1-t)**3/6
			bf2 = (3*t**3 - 6*t**2 + 4)/6
			bf3 = (-3*t**3 + 3*t**2 + 3*t +1)/6
			bf4 = t**3/6
			x = bf1*p1x + bf2*p2x + bf3*p3x + bf4*p4x
			y = bf1*p1y + bf2*p2y + bf3*p3y + bf4*p4y
			list.append([x,y])
	return list

def Overhauser(p1,p2,p3,p4,nPnts):
	"""
		interpolazione Overhauser
	"""
	list = []
	# controlla i punti
	if p1 != p2 and p2 != p3 and p3 != p4:
		# preleva cds dei punti
		p1x,p1y = p1.x(),p1.y()
		p2x,p2y = p2.x(),p2.y()
		p3x,p3y = p3.x(),p3.y()
		p4x,p4y = p4.x(),p4.y()
		# esegue calcolo
		r1x,r1y = p2x-p1x,p2y-p1y
		r4x,r4y = p4x-p3x,p4y-p3y
		for i in range (nPnts+1):
			t = float(i)/nPnts
			# blending functions
			bf1 = -t*(t-1)**2/2
			bf2 = (3*t**3-5*t**2+2)/2
			bf3 = t*(-3*t**2+4*t+1)/2
			bf4 = t**2*(t-1)/2
			x = bf1*p1x + bf2*p2x + bf3*p3x + bf4*p4x
			y = bf1*p1y + bf2*p2y + bf3*p3y + bf4*p4y
			list.append([x,y])
	return list

def BetaSpline(p1,p2,p3,p4,bias,tensione,nPnts):
	"""
		interpolazione BetaSpline
	"""
	list = []
	# controlla i punti
	if p1 != p2 and p2 != p3 and p3 != p4:
		# preleva cds dei punti
		p1x,p1y = p1.x(),p1.y()
		p2x,p2y = p2.x(),p2.y()
		p3x,p3y = p3.x(),p3.y()
		p4x,p4y = p4.x(),p4.y()
		# esegue calcolo
		d = tensione+2*bias**3+4*bias**2+4*bias+2
		for i in range (nPnts+1):
			t = float(i)/nPnts
			# blending functions
			bf1 = (-2*bias**3*t**3 + 6*bias**3*t**2 -6*bias**3*t + 2*bias**3)/d
			bf2 = (2*(tensione+bias**3+bias**2+bias)*t**3 - 3*(tensione+2*bias**3+2*bias**2)*t**2 + 6*(bias**3-bias)*t + (tensione+4*(bias**2+bias)))/d
			bf3 = (-2*(tensione+bias**2+bias+1)*t**3 + 3*(tensione+2*bias**2)*t**2 + 6*bias*t + 2)/d
			bf4 = 2*t**3/d
			x = bf1*p1x + bf2*p2x + bf3*p3x + bf4*p4x
			y = bf1*p1y + bf2*p2y + bf3*p3y + bf4*p4y
			list.append([x,y])
	return list

