# -*- coding: utf-8 -*-
"""
/***************************************************************************
P190Import
A QGIS plugin
Read UKOOA P1/90 files
Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
							-------------------
		begin                : 2019-05-17
		git sha              : $Format:%H$
		copyright            : (C) 2019 by Aaltronav s.r.o.
		email                : devel@aaltronav.eu
***************************************************************************/

/***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .p190_import_dialog import P190ImportDialog
import os.path

import qgis

from . import p190, p190_layer
from . import p190_tasks

# Anything bigger than this is considered a big file
BIG_FILESIZE = 2*1024*1024 # 2 MiB

class P190Import:
	"""QGIS Plugin Implementation."""

	def __init__(self, iface):
		"""Constructor.

		:param iface: An interface instance that will be passed to this class
			which provides the hook by which you can manipulate the QGIS
			application at run time.
		:type iface: QgsInterface
		"""
		# Save reference to the QGIS interface
		self.iface = iface
		# initialize plugin directory
		self.plugin_dir = os.path.dirname(__file__)
		# initialize locale
		locale = QSettings().value('locale/userLocale')[0:2]
		locale_path = os.path.join(
			self.plugin_dir,
			'i18n',
			'P190Import_{}.qm'.format(locale))

		if os.path.exists(locale_path):
			self.translator = QTranslator()
			self.translator.load(locale_path)

			if qVersion() > '4.3.3':
				QCoreApplication.installTranslator(self.translator)

		# Declare instance attributes
		self.actions = []
		self.menu = self.tr(u'&Seismic')

		# Check if plugin was started the first time in current QGIS session
		# Must be set in initGui() to survive plugin reloads
		self.first_start = None

		self.import_task = None
		self.filename = None

	# noinspection PyMethodMayBeStatic
	def tr(self, message):
		"""Get the translation for a string using Qt translation API.

		We implement this ourselves since we do not inherit QObject.

		:param message: String for translation.
		:type message: str, QString

		:returns: Translated version of message.
		:rtype: QString
		"""
		# noinspection PyTypeChecker,PyArgumentList,PyCallByClass
		return QCoreApplication.translate('P190Import', message)


	def add_action(
		self,
		icon_path,
		text,
		callback,
		enabled_flag=True,
		add_to_menu=True,
		add_to_toolbar=True,
		status_tip=None,
		whats_this=None,
		parent=None):
		"""Add a toolbar icon to the toolbar.

		:param icon_path: Path to the icon for this action. Can be a resource
			path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
		:type icon_path: str

		:param text: Text that should be shown in menu items for this action.
		:type text: str

		:param callback: Function to be called when the action is triggered.
		:type callback: function

		:param enabled_flag: A flag indicating if the action should be enabled
			by default. Defaults to True.
		:type enabled_flag: bool

		:param add_to_menu: Flag indicating whether the action should also
			be added to the menu. Defaults to True.
		:type add_to_menu: bool

		:param add_to_toolbar: Flag indicating whether the action should also
			be added to the toolbar. Defaults to True.
		:type add_to_toolbar: bool

		:param status_tip: Optional text to show in a popup when mouse pointer
			hovers over the action.
		:type status_tip: str

		:param parent: Parent widget for the new action. Defaults None.
		:type parent: QWidget

		:param whats_this: Optional text to show in the status bar when the
			mouse pointer hovers over the action.

		:returns: The action that was created. Note that the action is also
			added to self.actions list.
		:rtype: QAction
		"""

		icon = QIcon(icon_path)
		action = QAction(icon, text, parent)
		action.triggered.connect(callback)
		action.setEnabled(enabled_flag)

		if status_tip is not None:
			action.setStatusTip(status_tip)

		if whats_this is not None:
			action.setWhatsThis(whats_this)

		if add_to_toolbar:
			# Adds plugin icon to Plugins toolbar
			self.iface.addToolBarIcon(action)

		if add_to_menu:
			self.iface.addPluginToMenu(
				self.menu,
				action)

		self.actions.append(action)

		return action

	def initGui(self):
		"""Create the menu entries and toolbar icons inside the QGIS GUI."""

		icon_path = ':/plugins/p190_import/icon.png'
		self.add_action(
			icon_path,
			text=self.tr(u'Import P1&90'),
			callback=self.run,
			parent=self.iface.mainWindow())

		# will be set False in run()
		self.first_start = True


	def unload(self):
		"""Removes the plugin menu item and icon from QGIS GUI."""
		for action in self.actions:
			self.iface.removePluginMenu(
				self.tr(u'&Seismic'),
				action)
			self.iface.removeToolBarIcon(action)


	def run(self):
		"""
		If the dialog was accepted, run the import (possibly as a background
		task) and create the layers as per the dialog options.
		"""

		# Create the dialog with elements (after translation) and keep reference
		# Only create GUI ONCE in callback, so that it will only load when the plugin is started
		if self.first_start == True:
			self.first_start = False
			self.dlg = P190ImportDialog()

		# show the dialog
		self.dlg.show()
		# Run the dialog event loop
		result = self.dlg.exec_()
		# See if OK was pressed
		if result:
			# Don't let the user interact with the dialog
			# while we're processing the request
			self.dlg.setEnabled(False)

			rt = self.dlg.record_types
			sr = None
			if self.dlg.partialImport.isChecked():
				sr = ( self.dlg.shotpoint0.value(), self.dlg.shotpoint1.value() )

			self.filename = self.dlg.fileEdit.text()
			if self.filename:
				# Big files get processed in the background. Small ones are
				# dealt with in the main thread.
				if os.path.getsize(self.filename) > BIG_FILESIZE:
					task_description = "P1/90 import: "+os.path.basename(self.filename)
					self.import_task = p190_tasks.P190ImportTask(task_description, self.filename, only_records=rt, shot_range=sr, iface=self.iface)
					self.import_task.taskCompleted.connect(self.create_layers)
					qgis.core.QgsApplication.taskManager().addTask(self.import_task)

				else:
					records = p190.from_file(self.filename, only_records=rt, shot_range=sr)
					self.create_layers(records)

		# Re-enable the dialog
		self.dlg.setEnabled(True)

	def create_layers(self, records = None):

		if self.import_task:
			records = self.import_task.records
			self.import_task.taskCompleted.disconnect(self.create_layers)
			self.import_task = None

		if not records:
			self.iface.messageBar().pushMessage("No P1/90 records found in {}".format(self.filename), level=qgis.core.Qgis.Warning)
			return

		qgis.core.QgsMessageLog.logMessage("Read {} records from {}".format(len(records), self.filename), "P190", qgis.core.Qgis.Info, True)

		self.iface.messageBar().pushMessage("Read {} records from {}".format(len(records), self.filename))

		conflict = self.dlg.conflictComboBox.currentIndex()
		if self.dlg.gridCrs.isChecked() and self.dlg.crs:
			use_grid = True
			crs = self.dlg.crs
		else:
			use_grid = False
			crs = qgis.core.QgsCoordinateReferenceSystem(4326)

		if self.dlg.createLayerGroup.isChecked():
			root_node = p190_layer.group(os.path.basename(self.filename))
		else:
			root_node = None


		if self.dlg.shotpointsAsIndividualLayers.isChecked():
			layer_names = { r["line_name"]+" SP"+r["point_number"] for r in records if "line_name" in r.keys() and "point_number" in r.keys() }
		else:
			layer_names = { r["line_name"] for r in records if "line_name" in r.keys() }

		layer_names = [name for name in layer_names]
		layer_names.sort()

		for name in layer_names:
			if self.dlg.shotpointsAsIndividualLayers.isChecked():
				layer_records = [ r for r in records if "line_name" in r.keys() and "point_number" in r.keys() and (r["line_name"]+" SP"+r["point_number"]) == name ]
			else:
				layer_records = [ r for r in records if "line_name" in r.keys() and r["line_name"] == name ]

			if len(layer_records) == 0:
				print("Refusing to add empty layer")
				continue

			if conflict != 0: # Anything other than add again
				conflict_layers = qgis.core.QgsProject.instance().mapLayersByName(name)
				if len(conflict_layers):
					if conflict == 1: # Keep old
						continue
					elif conflict == 2: # Replace
						for layer in conflict_layers:
							qgis.core.QgsProject.instance().removeMapLayer(layer)
						# Then carry on and add the new layer

			p190_layer.add(p190_layer.create(name, layer_records, use_grid=use_grid, crs=crs), root_node)
