# -*- coding: utf-8 -*-
"""
/***************************************************************************
 OtsusMethodDialog
                                 A QGIS plugin
 This plugin applies Otsu's method for automated thresholding and segmentation of raster data.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2024-11-12
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Florian J. Ellsäßer
        email                : f.j.ellsaesser@utwente.nl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt import uic, QtWidgets
from qgis.PyQt.QtWidgets import QFileDialog
from qgis.core import QgsProject, Qgis, QgsRasterLayer
import os
import numpy as np
from osgeo import gdal


FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'otsusmethod_dialog_base.ui'))

class OtsusMethodDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor: initializes UI and connects signals for file selection and processing."""
        super(OtsusMethodDialog, self).__init__(parent)
        self.setupUi(self)

        # Connect buttons for input and output file selection
        self.search_input_button.clicked.connect(self.select_input_file)
        self.search_output_raster_button.clicked.connect(self.select_output_raster)
        self.search_output_button.clicked.connect(self.select_output_report)
        
        # Set up checkbox and combobox interaction
        self.checkBox_all_bands.stateChanged.connect(self.toggle_band_selection)
        self.comboBox_band_selection.clear()

    def select_input_file(self):
        """Browse for the input raster file and load bands."""
        file_name, _ = QFileDialog.getOpenFileName(self, "Select Raster File", "", "Raster files (*.tif *.img *.vrt)")
        if file_name:
            self.input_name.setText(file_name)
            self.load_bands(file_name)

    def select_output_raster(self):
        """Browse for output raster location."""
        file_name, _ = QFileDialog.getSaveFileName(self, "Select Output Raster File", "", "TIFF files (*.tif)")
        if file_name:
            self.output_raster_name.setText(file_name)

    def select_output_report(self):
        """Browse for output report location."""
        file_name, _ = QFileDialog.getSaveFileName(self, "Select Output Report File", "", "Text files (*.txt)")
        if file_name:
            self.output_name.setText(file_name)

    def load_bands(self, raster_path):
        """Load bands from the selected raster into the combobox and select Band 1 by default, allowing other selections."""
        dataset = gdal.Open(raster_path, gdal.GA_ReadOnly)
        if dataset:
            # Clear the combobox and add each band
            self.comboBox_band_selection.clear()
            for i in range(1, dataset.RasterCount + 1):
                self.comboBox_band_selection.addItem(f"Band {i}")
    
            # Set Band 1 as the default selection and ensure combobox remains enabled
            if dataset.RasterCount > 0:
                self.comboBox_band_selection.setCurrentIndex(0)  # Select Band 1 by default
                self.comboBox_band_selection.setEnabled(True)    # Ensure the combobox remains enabled for selection

    def toggle_band_selection(self):
        """Toggle between single band selection and all bands selection."""
        if self.checkBox_all_bands.isChecked():
            self.comboBox_band_selection.setEnabled(False)
            self.comboBox_band_selection.setCurrentIndex(-1)
        else:
            self.comboBox_band_selection.setEnabled(True)

    def apply_otsu_thresholding(self):
        """Perform Otsu's thresholding on the selected raster band(s)."""
        input_file = self.input_name.text()
        #output_file = self.output_raster_name.text()
        all_bands = self.checkBox_all_bands.isChecked()

        #data = self.read_img(input_file)
        data_gdal = gdal.Open(input_file,gdal.GA_ReadOnly)
        if all_bands:
            bands_to_use = [data_gdal.GetRasterBand(i + 1) for i in range(data_gdal.RasterCount)]
        else:
            # Get the specific band based on the selected index
            selected_band_index = self.comboBox_band_selection.currentIndex() + 1  # Band indices in GDAL are 1-based
            bands_to_use = [data_gdal.GetRasterBand(selected_band_index)]
        
        prj = data_gdal.GetProjection()
        geo = data_gdal.GetGeoTransform()
        # Get the geotransform from the GDAL dataset
        geo_transform = data_gdal.GetGeoTransform()

        lon = float(geo_transform[0])
        lat = float(geo_transform[3])

        # Generate grayscale
        greyscale_image = self.make_greyscale_image(bands_to_use)
        
        # Apply Otsu's method
        binary_image, threshold, valid_pixel_values = self.otsu_threshold(greyscale_image)
        self.output_image = binary_image

        # write output image
        self.write_output_images(binary_image,prj,geo,lon,lat,threshold)
        
        # Display results
        #plt.imshow(binary_image, cmap='gray')
        #plt.title(f"Otsu's Binary Image (Threshold = {threshold})")
        #plt.colorbar()
        #plt.show()

    def write_output_images(self, binary_image, projection, geo_transform, longitude, latitude, threshold):
        """
        This function writes the binary output data into a GeoTIFF and saves it to the specified location.
        Additionally, it adds the resulting layer to QGIS and writes a report with threshold details.
        """
        
        # Get the dimensions of the binary image
        rows, cols = binary_image.shape
        output_raster_path = self.output_raster_name.text()
        report_path = self.output_name.text()
    
        # Check output paths
        if not output_raster_path:
            raise ValueError("Output raster path is not defined.")
        if not report_path:
            raise ValueError("Output report path is not defined.")
        
        # Initialize GDAL GeoTIFF driver
        driver = gdal.GetDriverByName('GTiff')
        if driver is None:
            raise RuntimeError("GeoTIFF driver not available.")
    
        # Create output raster file
        try:
            out_raster = driver.Create(output_raster_path, cols, rows, 1, gdal.GDT_Float32)
            if out_raster is None:
                raise RuntimeError("Failed to create output raster file.")
    
            # Set geotransformation and projection
            out_raster.SetGeoTransform(geo_transform)
            out_raster.SetProjection(projection)
    
            # Write the binary image data to band 1
            out_band = out_raster.GetRasterBand(1)
            out_band.SetNoDataValue(0)  # Set NoData value for clear distinction
            out_band.WriteArray(binary_image)
            out_band.FlushCache()
    
            # Close the output raster to save changes
            out_raster.FlushCache()
        finally:
            # Ensuring resources are released
            out_band = None
            out_raster = None
    
        # Add the raster layer to QGIS for visualization
        qgis_raster_layer = QgsRasterLayer(output_raster_path, "Otsu's Method Output")
        if not qgis_raster_layer.isValid():
            raise RuntimeError("Failed to load the raster layer into QGIS.")
        QgsProject.instance().addMapLayer(qgis_raster_layer)
    
        # Write a report file with threshold details
        try:
            with open(report_path, 'w') as report_file:
                report_file.write(f"Otsu's Threshold: {threshold}\n")
        except IOError:
            raise IOError("Failed to write the threshold report.")
    
        def read_img(self, in_file, na_val=None):
            """Read a raster file and load its bands."""
            dataset = gdal.Open(in_file, gdal.GA_ReadOnly)
            if dataset is None:
                raise ValueError("Could not open the file.")
            bands = [dataset.GetRasterBand(i).ReadAsArray().astype(float) for i in range(1, dataset.RasterCount + 1)]
            return {"in_bands": bands}

    def make_greyscale_image(self,bands):
        # Convert each band to a numpy array
        band_arrays = []
    
        for band in bands:
            # Get the raster band as a numpy array
            band_data = band.ReadAsArray()  # Converts the band to a 2D numpy array
            band_arrays.append(band_data)
    
        # Stack the bands into a 3D array (if you have multiple bands)
        if len(band_arrays) > 1:
            stacked_bands = np.stack(band_arrays, axis=-1)  # Stack along the third axis (color channels)
            # Calculate the mean of the bands (greyscale image)
            greyscale_image = np.mean(stacked_bands, axis=-1)  # Mean across the color channels
        else:
            # If there's only one band, just return it
            greyscale_image = band_arrays[0]
    
        return greyscale_image

    def otsu_threshold(self, image):
        """Perform Otsu's thresholding on the grayscale image."""
        # Convert the image to integer and set positive values to 1
        image_int = np.nan_to_num(image, nan=0).astype(np.int32)
        image_int[image_int > 0] = 2
    
        # Step 1: Remove zero and NaN values from valid_pixel_values
        valid_pixel_values = image[~np.isnan(image) & (image != 0)].ravel()
    
        # Otsu's method: Calculate histogram and probabilities
        histogram, _ = np.histogram(valid_pixel_values, bins=256, range=(0, 256))
        total_pixels = valid_pixel_values.size
        probability = histogram / total_pixels
        cumulative_sum = np.cumsum(probability)
        cumulative_mean = np.cumsum(probability * np.arange(256))
        total_mean = cumulative_mean[-1]
        
        # Calculate between-class variance and find the threshold
        between_class_variance = ((total_mean * cumulative_sum - cumulative_mean) ** 2) / (cumulative_sum * (1 - cumulative_sum))
        between_class_variance = np.nan_to_num(between_class_variance)
        threshold = np.argmax(between_class_variance)
        
        # Step 2: Create binary image with 1 for values above the threshold
        binary_image = image > threshold
        
        # Step 3: Use image_int to set all values below the threshold to zero while keeping NaNs
        image_int[binary_image] = 1  # Set values above threshold to 1
        #image_int[~binary_image] = 0  # Set values below threshold to 0
        
        # Step 4: Preserve NaN values from the original image
        #image_int[np.isnan(image)] = np.nan
        
        # Step 5: Set all values equal to 2 in image_int to 0
        #image_int[image_int == 2] = 0
        
        # Return the binary image, threshold, and valid pixel values
        return image_int, threshold, valid_pixel_values