# -*- coding: utf-8 -*-

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterNumber,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterEnum,
    QgsCoordinateReferenceSystem,
    QgsFields,
    QgsFeature,
    QgsWkbTypes,
    QgsFeatureSink,
    QgsProcessingException,
    QgsField,
    QgsProcessingMultiStepFeedback,
    QgsVectorLayer,
    QgsProcessingUtils,
    QgsRectangle,
    QgsProject,
    QgsFeatureRequest,
)
import math
import os

# Module-level compatibility helper for different osm_query_string_by_bbox signatures
def _compat_osm_query_bbox(min_lat, min_lon, max_lat, max_lon, **kw):
    try:
        return osm_query_string_by_bbox(
            min_lat=min_lat, min_lon=min_lon, max_lat=max_lat, max_lon=max_lon, **kw
        )
    except TypeError:
        return osm_query_string_by_bbox(min_lat, min_lon, max_lat, max_lon, **kw)

from ..parameters import (
    default_curve_radius,
    min_d_to_building,
    d_to_add_to_each_side,
    minimal_buffer,
    perc_draw_kerbs,
    perc_tol_crossings,
    d_to_add_interp_d,
    CRS_LATLON_4326,
    default_widths,
    highway_tag,
    widths_fieldname,
    cutoff_percent_protoblock,
)
from ..osm_fetch import osm_query_string_by_bbox, get_osm_data
from ..generic_functions import (
    reproject_layer_localTM,
    cliplayer_v2,
    remove_unconnected_lines_v2,
    polygonize_lines,
    create_new_layerfield,
    edit,
    select_feats_by_attr,
    layer_from_featlist,
    dissolve_tosinglegeom,
    generate_buffer,
    split_lines,
    check_empty_layer,
    create_incidence_field_layers_A_B,
)
from .sidewalk_generation_logic import generate_sidewalk_geometries_and_zones
from .full_sidewalkreator_bbox_algorithm import FullSidewalkreatorBboxAlgorithm


class FullSidewalkreatorPolygonAlgorithm(QgsProcessingAlgorithm):
    INPUT_POLYGON = "INPUT_POLYGON"
    TIMEOUT = "TIMEOUT"
    FETCH_BUILDINGS_DATA = "FETCH_BUILDINGS_DATA"
    FETCH_ADDRESS_DATA = "FETCH_ADDRESS_DATA"
    DEAD_END_ITERATIONS = "DEAD_END_ITERATIONS"
    SIDEWALK_CURVE_RADIUS = "SIDEWALK_CURVE_RADIUS"
    SIDEWALK_ADDED_ROAD_WIDTH_TOTAL = "SIDEWALK_ADDED_ROAD_WIDTH_TOTAL"
    SIDEWALK_CHECK_BUILDING_OVERLAP = "SIDEWALK_CHECK_BUILDING_OVERLAP"
    SIDEWALK_MIN_DIST_TO_BUILDING = "SIDEWALK_MIN_DIST_TO_BUILDING"
    SIDEWALK_MIN_WIDTH_IF_NEAR_BUILDING = "SIDEWALK_MIN_WIDTH_IF_NEAR_BUILDING"
    CROSSING_METHOD_PARAM = "CROSSING_METHOD_PARAM"
    CROSSING_KERB_OFFSET_PERCENT = "CROSSING_KERB_OFFSET_PERCENT"
    CROSSING_MAX_LENGTH_TOLERANCE_PERCENT = "CROSSING_MAX_LENGTH_TOLERANCE_PERCENT"
    CROSSING_INWARD_OFFSET = "CROSSING_INWARD_OFFSET"
    CROSSING_MIN_ROAD_LENGTH = "CROSSING_MIN_ROAD_LENGTH"
    CROSSING_AUTO_REMOVE_LONG = "CROSSING_AUTO_REMOVE_LONG"
    SPLITTING_METHOD = "SPLITTING_METHOD"
    SPLIT_VORONOI_MIN_POIS = "SPLIT_VORONOI_MIN_POIS"
    SPLIT_MAX_LENGTH_VALUE = "SPLIT_MAX_LENGTH_VALUE"
    SPLIT_SEGMENT_NUMBER_VALUE = "SPLIT_SEGMENT_NUMBER_VALUE"
    STREET_CLASSES = "STREET_CLASSES"
    OUTPUT_SIDEWALKS = "OUTPUT_SIDEWALKS"
    OUTPUT_CROSSINGS = "OUTPUT_CROSSINGS"
    OUTPUT_KERBS = "OUTPUT_KERBS"
    OUTPUT_PROTOBLOCKS_DEBUG = "OUTPUT_PROTOBLOCKS_DEBUG"
    CROSSING_METHOD_OPTIONS_ENUM = [
        "Parallel to Transversal Segment",
        "Perpendicular to Road Segment",
    ]
    SPLITTING_METHOD_OPTIONS_ENUM = [
        "None (only protoblock corners)",
        "Voronoi Polygons",
        "By Maximum Length",
        "By Fixed Number of Segments",
    ]

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return FullSidewalkreatorPolygonAlgorithm()

    def name(self):
        return "fullsidewalkreatorfrompolygon"

    def displayName(self):
        return self.tr("Generate Full Sidewalk Network (from Polygon)")

    def shortHelpString(self):
        return self.tr(
            "Fetches OSM road and building data for an input polygon area. This algorithm focuses on generating sidewalk lines and related features (like exclusion zones for debugging) using configurable parameters. Main sidewalk output is in EPSG:4326. (Full plugin includes crossings and kerbs)."
        )

    def icon(self):
        plugin_dir = os.path.dirname(os.path.dirname(__file__))
        icon_path = os.path.join(plugin_dir, "icon.png")
        return QIcon(icon_path) if os.path.exists(icon_path) else QIcon()

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_POLYGON,
                self.tr("Input Area Polygon Layer (EPSG:4326 recommended)"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.TIMEOUT,
                self.tr("OSM Download Timeout (seconds)"),
                QgsProcessingParameterNumber.Integer,
                defaultValue=60,
                minValue=10,
                maxValue=300,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.FETCH_BUILDINGS_DATA,
                self.tr(
                    "Fetch OSM Buildings Data (for overlap checks & POI splitting)"
                ),
                defaultValue=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.FETCH_ADDRESS_DATA,
                self.tr("Fetch OSM Address Data (addr:housenumber, for POI splitting)"),
                defaultValue=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.DEAD_END_ITERATIONS,
                self.tr("Iterations to Remove Dead-End Streets (for protoblocks)"),
                QgsProcessingParameterNumber.Integer,
                defaultValue=1,
                minValue=0,
                maxValue=10,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.SIDEWALK_CURVE_RADIUS,
                self.tr("Sidewalk Corner Curve Radius (meters)"),
                QgsProcessingParameterNumber.Double,
                defaultValue=default_curve_radius,
                minValue=0.0,
                maxValue=20.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.SIDEWALK_ADDED_ROAD_WIDTH_TOTAL,
                self.tr(
                    "Total Added Width to Road for Sidewalk Axis (meters, for both sides)"
                ),
                QgsProcessingParameterNumber.Double,
                defaultValue=d_to_add_to_each_side * 2,
                minValue=0.0,
                maxValue=10.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.SIDEWALK_CHECK_BUILDING_OVERLAP,
                self.tr(
                    "Adjust Sidewalk Width if Overlaps Buildings (slower if buildings are fetched)"
                ),
                defaultValue=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.SIDEWALK_MIN_DIST_TO_BUILDING,
                self.tr("Min. Distance Sidewalk to Buildings (m, if overlap checked)"),
                QgsProcessingParameterNumber.Double,
                defaultValue=min_d_to_building,
                minValue=0.0,
                maxValue=10.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.SIDEWALK_MIN_WIDTH_IF_NEAR_BUILDING,
                self.tr("Min. Sidewalk Width Near Buildings (m, if overlap checked)"),
                QgsProcessingParameterNumber.Double,
                defaultValue=minimal_buffer * 2,
                minValue=0.1,
                maxValue=10.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                self.CROSSING_METHOD_PARAM,
                self.tr("Crossing Generation Method"),
                options=self.CROSSING_METHOD_OPTIONS_ENUM,
                defaultValue=0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.CROSSING_KERB_OFFSET_PERCENT,
                self.tr("Crossing: Kerb Position (%)"),
                QgsProcessingParameterNumber.Integer,
                defaultValue=int(perc_draw_kerbs),
                minValue=0,
                maxValue=100,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.CROSSING_MAX_LENGTH_TOLERANCE_PERCENT,
                self.tr("Crossing: Max Length Tolerance (%)"),
                QgsProcessingParameterNumber.Integer,
                defaultValue=int(perc_tol_crossings),
                minValue=0,
                maxValue=100,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.CROSSING_INWARD_OFFSET,
                self.tr("Crossing: Inward Offset (m)"),
                QgsProcessingParameterNumber.Double,
                defaultValue=d_to_add_interp_d,
                minValue=0.0,
                maxValue=10.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.CROSSING_MIN_ROAD_LENGTH,
                self.tr("Crossing: Min Road Length (m)"),
                QgsProcessingParameterNumber.Double,
                defaultValue=20.0,
                minValue=0.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.CROSSING_AUTO_REMOVE_LONG,
                self.tr("Crossing: Auto-Remove Long"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                self.SPLITTING_METHOD,
                self.tr("Sidewalk Splitting Method"),
                options=self.SPLITTING_METHOD_OPTIONS_ENUM,
                defaultValue=0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.SPLIT_VORONOI_MIN_POIS,
                self.tr("Splitting (Voronoi): Min POIs"),
                QgsProcessingParameterNumber.Integer,
                defaultValue=3,
                minValue=1,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.SPLIT_MAX_LENGTH_VALUE,
                self.tr("Splitting (Max Length): Value (m)"),
                QgsProcessingParameterNumber.Double,
                defaultValue=50.0,
                minValue=1.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.SPLIT_SEGMENT_NUMBER_VALUE,
                self.tr("Splitting (By Number): Segments"),
                QgsProcessingParameterNumber.Integer,
                defaultValue=3,
                minValue=1,
            )
        )
        # Optional street classes to include (similar to bbox algorithm)
        street_options = [
            "motorway",
            "motorway_link",
            "trunk",
            "trunk_link",
            "primary",
            "primary_link",
            "secondary",
            "secondary_link",
            "tertiary",
            "tertiary_link",
            "residential",
            "living_street",
            "service",
            "unclassified",
            "road",
            "track",
            "path",
            "cycleway",
            "footway",
            "pedestrian",
        ]
        self.addParameter(
            QgsProcessingParameterEnum(
                self.STREET_CLASSES,
                self.tr("Street Classes to Process"),
                options=street_options,
                allowMultiple=True,
                defaultValue=list(range(len(street_options))),
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_SIDEWALKS, self.tr("Output Sidewalks (EPSG:4326)")
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_CROSSINGS, self.tr("Output Crossings (EPSG:4326)")
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_KERBS, self.tr("Output Kerbs (EPSG:4326)")
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_PROTOBLOCKS_DEBUG,
                self.tr("Output Protoblocks (Debug - local TM CRS)"),
                type=QgsProcessing.TypeVectorPolygon,
                optional=True,
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        feedback.pushInfo(self.tr("Full Sidewalkreator (Polygon) - Algorithm Started."))

        input_polygon_fs = self.parameterAsSource(parameters, self.INPUT_POLYGON, context)
        if input_polygon_fs is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT_POLYGON))
        timeout = self.parameterAsInt(parameters, self.TIMEOUT, context)
        fetch_buildings_param = self.parameterAsBoolean(parameters, self.FETCH_BUILDINGS_DATA, context)

        actual_input_layer = input_polygon_fs.materialize(QgsFeatureRequest())
        if (not actual_input_layer or not actual_input_layer.isValid() or actual_input_layer.featureCount() == 0):
            raise QgsProcessingException(self.tr("Materialized input polygon layer is invalid or empty."))

        crs_4326 = QgsCoordinateReferenceSystem(CRS_LATLON_4326)
        input_poly_for_bbox = actual_input_layer
        if actual_input_layer.sourceCrs().authid() != crs_4326.authid():
            res = processing.run(
                "native:reprojectlayer",
                {"INPUT": actual_input_layer, "TARGET_CRS": crs_4326, "OUTPUT": "memory:input_reprojected_for_bbox"},
                context=context, feedback=feedback, is_child_algorithm=True,
            )
            input_poly_for_bbox = QgsProcessingUtils.mapLayerFromString(res.get("OUTPUT"), context)
            if not input_poly_for_bbox or not input_poly_for_bbox.isValid() or input_poly_for_bbox.featureCount() == 0:
                raise QgsProcessingException(self.tr("Failed to reproject input for BBOX or result is empty."))

        extent_4326 = input_poly_for_bbox.extent()
        if extent_4326.isNull():
            raise QgsProcessingException(self.tr(f"Invalid BBOX from input: {extent_4326.toString()}. Ensure input layer '{input_poly_for_bbox.name()}' has valid geometries."))

        # QGIS quirk: parameterAsExtent parsing of string specs behaves like
        # "xMin,xMax,yMin,yMax [CRS]" in some contexts. Match the same
        # workaround used by docker/run_full_bbox.sh to avoid lat/lon mixups.
        west_lon = extent_4326.xMinimum()
        east_lon = extent_4326.xMaximum()
        south_lat = extent_4326.yMinimum()
        north_lat = extent_4326.yMaximum()

        # Build extent string in the order xMin, xMax, yMin, yMax
        bbox_str = f"{west_lon},{east_lon},{south_lat},{north_lat} [EPSG:4326]"
        feedback.pushInfo(self.tr(f"Delegation extent string (xMin,xMax,yMin,yMax): {bbox_str}"))
        try:
            street_class_indices = self.parameterAsEnums(parameters, self.STREET_CLASSES, context)
        except Exception:
            street_class_indices = []
        bbox_params = {
            "INPUT_EXTENT": bbox_str,
            "TIMEOUT": timeout,
            "GET_BUILDING_DATA": fetch_buildings_param,
            "DEFAULT_WIDTH": 6.0,
            "MIN_WIDTH": 1.0,
            "MAX_WIDTH": 25.0,
            "STREET_CLASSES": street_class_indices or list(range(20)),
            self.OUTPUT_SIDEWALKS: parameters.get(self.OUTPUT_SIDEWALKS, "memory:sw"),
            self.OUTPUT_CROSSINGS: parameters.get(self.OUTPUT_CROSSINGS, "memory:cr"),
            self.OUTPUT_KERBS: parameters.get(self.OUTPUT_KERBS, "memory:kb"),
            self.OUTPUT_PROTOBLOCKS_DEBUG: parameters.get(self.OUTPUT_PROTOBLOCKS_DEBUG),
            "SAVE_PROTOBLOCKS_DEBUG": False,
            "SAVE_EXCLUSION_ZONES_DEBUG": False,
            "SAVE_SURE_ZONES_DEBUG": False,
            "SAVE_STREETS_WIDTH_ADJUSTED_DEBUG": False,
        }
        feedback.pushInfo(self.tr("Delegating to BBOX pipeline with polygon extent."))
        # Important: run as child algorithm with same context/feedback to avoid GUI runner deadlock
        return processing.run(
            FullSidewalkreatorBboxAlgorithm(),
            bbox_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True,
        )

        # Fetch Roads
        query_str_roads = _compat_osm_query_bbox(
                min_lat,
                min_lgt,
                max_lat,
                max_lgt,
                interest_key=highway_tag,
                way=True,
            )
        osm_roads_geojson_str = get_osm_data(
            querystring=query_str_roads,
            tempfilesname="osm_roads_full_algo",
            geomtype="LineString",
            timeout=timeout,
            return_as_string=True,
        )
        if osm_roads_geojson_str is None:
            raise QgsProcessingException(self.tr("Failed to fetch OSM road data."))
        # Support inline GeoJSON strings by writing to a temporary file
        roads_src = osm_roads_geojson_str
        if isinstance(roads_src, str) and not os.path.exists(roads_src):
            s = roads_src.strip()
            if s.startswith("{") or s.startswith("["):
                import tempfile
                tmp = tempfile.NamedTemporaryFile(delete=False, suffix=".geojson")
                tmp.write(roads_src.encode("utf-8"))
                tmp.flush(); tmp.close()
                roads_src = tmp.name
        osm_roads_layer_4326 = QgsVectorLayer(
            roads_src, "osm_roads_dl_4326_full", "ogr"
        )
        if not osm_roads_layer_4326.isValid():
            raise QgsProcessingException(
                self.tr("Fetched OSM road data is not a valid layer.")
            )
        feedback.pushInfo(
            self.tr(f"Fetched {osm_roads_layer_4326.featureCount()} OSM ways.")
        )

        # Fetch Buildings (if requested)
        osm_buildings_layer_4326 = None
        if fetch_buildings_param:
            feedback.pushInfo(self.tr("Fetching OSM building data..."))
            query_buildings = _compat_osm_query_bbox(
                min_lat,
                min_lgt,
                max_lat,
                max_lgt,
                interest_key="building",
                way=True,
                relation=True,
            )
            osm_bldgs_geojson_str = get_osm_data(
                querystring=query_buildings,
                tempfilesname="osm_bldgs_full_algo",
                geomtype="Polygon",
                timeout=timeout,
                return_as_string=True,
            )
            if osm_bldgs_geojson_str:
                bld_src = osm_bldgs_geojson_str
                if isinstance(bld_src, str) and not os.path.exists(bld_src):
                    s = bld_src.strip()
                    if s.startswith("{") or s.startswith("["):
                        import tempfile
                        tmp = tempfile.NamedTemporaryFile(delete=False, suffix=".geojson")
                        tmp.write(bld_src.encode("utf-8"))
                        tmp.flush(); tmp.close()
                        bld_src = tmp.name
                osm_buildings_layer_4326 = QgsVectorLayer(
                    bld_src, "osm_bldgs_dl_4326_full", "ogr"
                )
                if (
                    osm_buildings_layer_4326.isValid()
                    and osm_buildings_layer_4326.featureCount() > 0
                ):
                    feedback.pushInfo(
                        self.tr(
                            f"Fetched {osm_buildings_layer_4326.featureCount()} OSM buildings."
                        )
                    )
                else:
                    feedback.pushInfo(
                        self.tr("No valid building data fetched or layer is empty.")
                    )
                    osm_buildings_layer_4326 = None
            else:
                feedback.pushInfo(self.tr("Failed to fetch building data string."))
                osm_buildings_layer_4326 = None

        # Fetch Addresses (if requested)
        osm_addresses_layer_4326 = None
        if fetch_addresses_param:
            feedback.pushInfo(self.tr("Fetching OSM address data..."))
            query_addresses = _compat_osm_query_bbox(
                min_lat,
                min_lgt,
                max_lat,
                max_lgt,
                interest_key="addr:housenumber",
                node=True,
                way=False,
            )
            osm_addrs_geojson_str = get_osm_data(
                querystring=query_addresses,
                tempfilesname="osm_addrs_full_algo",
                geomtype="Point",
                timeout=timeout,
                return_as_string=True,
            )
            if osm_addrs_geojson_str:
                add_src = osm_addrs_geojson_str
                if isinstance(add_src, str) and not os.path.exists(add_src):
                    s = add_src.strip()
                    if s.startswith("{") or s.startswith("["):
                        import tempfile
                        tmp = tempfile.NamedTemporaryFile(delete=False, suffix=".geojson")
                        tmp.write(add_src.encode("utf-8"))
                        tmp.flush(); tmp.close()
                        add_src = tmp.name
                osm_addresses_layer_4326 = QgsVectorLayer(
                    add_src, "osm_addrs_dl_4326_full", "ogr"
                )
                if (
                    osm_addresses_layer_4326.isValid()
                    and osm_addresses_layer_4326.featureCount() > 0
                ):
                    feedback.pushInfo(
                        self.tr(
                            f"Fetched {osm_addresses_layer_4326.featureCount()} OSM addresses."
                        )
                    )
                else:
                    feedback.pushInfo(
                        self.tr("No valid address data fetched or layer is empty.")
                    )
                    osm_addresses_layer_4326 = None
            else:
                feedback.pushInfo(self.tr("Failed to fetch address data string."))
                osm_addresses_layer_4326 = None

        # Clip roads
        clipped_osm_roads_4326 = cliplayer_v2(
            osm_roads_layer_4326, input_poly_for_bbox, "memory:clipped_roads_4326_full"
        )
        if (
            not clipped_osm_roads_4326.isValid()
            or clipped_osm_roads_4326.featureCount() == 0
        ):
            feedback.pushWarning(
                self.tr("No OSM roads after clipping. Output will be empty.")
            )
            return self.handle_empty_results(parameters, context, crs_4326)

        # Reproject clipped roads to local TM
        roads_local_tm, local_tm_crs = reproject_layer_localTM(
            clipped_osm_roads_4326,
            None,
            "roads_local_tm_full",
            extent_4326.center().x(),
        )
        if not roads_local_tm.isValid():
            raise QgsProcessingException(
                self.tr("Reprojecting OSM roads to local TM failed.")
            )

        # Reproject buildings if fetched
        reproj_buildings_layer = None
        if osm_buildings_layer_4326 and osm_buildings_layer_4326.featureCount() > 0:
            feedback.pushInfo(
                self.tr("Clipping and Reprojecting building data to local TM...")
            )
            clipped_buildings_4326 = cliplayer_v2(
                osm_buildings_layer_4326,
                input_poly_for_bbox,
                "memory:clipped_bldgs_4326_full",
            )
            if (
                clipped_buildings_4326
                and clipped_buildings_4326.isValid()
                and clipped_buildings_4326.featureCount() > 0
            ):
                # Use the same lgt_0 for consistency; reproject_layer_localTM returns the new CRS it generated.
                temp_reproj_bldgs, bldg_tm_crs_obj = reproject_layer_localTM(
                    clipped_buildings_4326,
                    None,
                    "bldgs_local_tm_full_temp",
                    extent_4326.center().x(),
                )
                if not temp_reproj_bldgs or not temp_reproj_bldgs.isValid():
                    feedback.pushWarning(
                        self.tr(
                            "Failed to reproject building data. Proceeding without it for overlap checks."
                        )
                    )
                else:
                    # Crucially, ensure the building layer uses the exact same CRS object as roads_local_tm
                    if bldg_tm_crs_obj != local_tm_crs:
                        feedback.pushWarning(
                            self.tr(
                                "Building TM CRS definition differs from road TM CRS. Forcing road TM CRS for buildings."
                            )
                        )
                        temp_reproj_bldgs.setCrs(local_tm_crs)
                    reproj_buildings_layer = temp_reproj_bldgs
                    feedback.pushInfo(
                        self.tr(
                            f"Buildings reprojected to local TM: {reproj_buildings_layer.featureCount()} features. CRS: {reproj_buildings_layer.crs().description()}"
                        )
                    )
            else:
                feedback.pushInfo(
                    self.tr("No buildings after clipping, or clipping failed.")
                )

        # Clean street network
        feedback.pushInfo(self.tr("Cleaning street network..."))
        filtered_streets_layer = QgsVectorLayer(
            f"LineString?crs={local_tm_crs.authid()}", "filtered_streets_full", "memory"
        )
        filtered_streets_dp = filtered_streets_layer.dataProvider()
        street_fields = roads_local_tm.fields()
        if street_fields.count() > 0:
            filtered_streets_dp.addAttributes(street_fields)
        else:
            filtered_streets_dp.addAttributes([QgsField("dummy_id", QVariant.Int)])
        filtered_streets_layer.updateFields()

        features_to_add_to_filtered = []
        highway_field_idx = roads_local_tm.fields().lookupField(highway_tag)
        width_field_idx_on_source = roads_local_tm.fields().lookupField(
            widths_fieldname
        )

        for f_in in roads_local_tm.getFeatures():
            if feedback.isCanceled():
                return {}
            highway_type_attr = (
                f_in.attribute(highway_field_idx) if highway_field_idx != -1 else None
            )
            highway_type_str = (
                str(highway_type_attr).lower() if highway_type_attr is not None else ""
            )
            width_from_defaults = default_widths.get(highway_type_str, 0.0)
            # Filter by user-selected street classes and width
            if (not street_classes_to_process or highway_type_str in street_classes_to_process) and (
                width_from_defaults >= 0.5
            ):
                new_feat = QgsFeature(filtered_streets_layer.fields())
                new_feat.setGeometry(f_in.geometry())
                new_feat.setAttributes(f_in.attributes())
                target_width_idx = new_feat.fields().lookupField(widths_fieldname)
                if target_width_idx != -1:  # Ensure width field exists on target
                    current_osm_width = (
                        f_in.attribute(width_field_idx_on_source)
                        if width_field_idx_on_source != -1
                        else None
                    )
                    # Try to use actual OSM width if valid, otherwise use default
                    try:
                        final_width = float(current_osm_width)
                        if (
                            final_width <= 0
                        ):  # Or some other threshold for invalid OSM width
                            final_width = width_from_defaults
                    except (TypeError, ValueError):
                        final_width = width_from_defaults
                    new_feat.setAttribute(target_width_idx, final_width)
                features_to_add_to_filtered.append(new_feat)

        if features_to_add_to_filtered:
            filtered_streets_dp.addFeatures(features_to_add_to_filtered)
        feedback.pushInfo(
            self.tr(
                f"Streets filtered by type/width: {filtered_streets_layer.featureCount()} ways remain."
            )
        )
        if filtered_streets_layer.featureCount() == 0:
            feedback.pushWarning(
                self.tr("No streets after filtering. Output will be empty.")
            )
            return self.handle_empty_results(
                parameters, context, crs_4326, local_tm_crs
            )

        for i in range(dead_end_iterations):  # Use parameter
            if feedback.isCanceled():
                return {}
            feedback.pushInfo(
                self.tr(
                    f"Removing unconnected lines (iteration {i+1}/{dead_end_iterations})..."
                )
            )
            remove_unconnected_lines_v2(filtered_streets_layer)
        feedback.pushInfo(
            self.tr(
                f"After removing unconnected lines: {filtered_streets_layer.featureCount()} ways remain."
            )
        )
        if filtered_streets_layer.featureCount() == 0:
            feedback.pushWarning(
                self.tr("No streets after removing dead-ends. Output will be empty.")
            )
            return self.handle_empty_results(
                parameters, context, crs_4326, local_tm_crs
            )

        initial_protoblocks_layer = polygonize_lines(
            filtered_streets_layer, "memory:initial_protoblocks_full", False
        )
        if not initial_protoblocks_layer or not initial_protoblocks_layer.isValid():
            raise QgsProcessingException(self.tr("Initial polygonization failed."))

        clean_protoblocks_layer_local_tm = QgsVectorLayer(
            f"Polygon?crs={local_tm_crs.authid()}", "clean_protoblocks_full", "memory"
        )
        if initial_protoblocks_layer.featureCount() > 0:
            cloned_protoblock_feats = [
                QgsFeature(f) for f in initial_protoblocks_layer.getFeatures()
            ]
            clean_protoblocks_layer_local_tm.dataProvider().addFeatures(
                cloned_protoblock_feats
            )
        feedback.pushInfo(
            self.tr(
                f"Protoblocks generated (local TM): {clean_protoblocks_layer_local_tm.featureCount()} features."
            )
        )

        protoblocks_debug_dest_id = None
        debug_protoblocks_output_spec = parameters.get(self.OUTPUT_PROTOBLOCKS_DEBUG)
        if debug_protoblocks_output_spec:
            (protoblocks_debug_sink, protoblocks_debug_dest_id) = self.parameterAsSink(
                parameters,
                self.OUTPUT_PROTOBLOCKS_DEBUG,
                context,
                clean_protoblocks_layer_local_tm.fields(),
                QgsWkbTypes.Polygon,
                local_tm_crs,
            )
            if protoblocks_debug_sink:
                for feat_pb_debug in clean_protoblocks_layer_local_tm.getFeatures():
                    if feedback.isCanceled():
                        return {}
                    protoblocks_debug_sink.addFeature(
                        feat_pb_debug, QgsFeatureSink.FastInsert
                    )
            else:
                protoblocks_debug_dest_id = None
        feedback.pushInfo(self.tr("Stage 1 (Protoblocks for Debug) Finished."))

        # --- Stage 2: Pre-existing Sidewalk Filtering on Protoblocks ---
        # Remove protoblocks which already contain a significant sidewalk network (mirror GUI behavior)
        try:
            existing_sw_feats = select_feats_by_attr(
                roads_local_tm, "footway", "sidewalk"
            )
        except Exception:
            existing_sw_feats = []

        if existing_sw_feats:
            feedback.pushInfo(
                self.tr(
                    f"Found {len(existing_sw_feats)} pre-existing sidewalk segments (footway=sidewalk). Applying protoblock filter."
                )
            )
            existing_sw_layer = layer_from_featlist(
                existing_sw_feats,
                "existing_sidewalks_local_tm",
                "linestring",
                CRS=local_tm_crs,
            )
            # Add total sidewalk length inside each protoblock
            inc_field_id = create_incidence_field_layers_A_B(
                clean_protoblocks_layer_local_tm,
                existing_sw_layer,
                fieldname="inc_sidewalk_len",
                total_length_instead=True,
            )
            # Remove protoblocks above cutoff percent
            with edit(clean_protoblocks_layer_local_tm):
                for feat in list(clean_protoblocks_layer_local_tm.getFeatures()):
                    try:
                        inc_len = float(feat["inc_sidewalk_len"] or 0.0)
                    except Exception:
                        inc_len = 0.0
                    area = feat.geometry().area() if feat.hasGeometry() else 0.0
                    ratio = 0.0
                    if area > 0 and inc_len > 0:
                        ratio = (((inc_len / 4.0) ** 2) / area) * 100.0
                    if ratio > cutoff_percent_protoblock:
                        clean_protoblocks_layer_local_tm.deleteFeature(feat.id())
            remaining = clean_protoblocks_layer_local_tm.featureCount()
            feedback.pushInfo(
                self.tr(
                    f"Protoblocks remaining after pre-existing sidewalk filter: {remaining}"
                )
            )
        else:
            feedback.pushInfo(
                self.tr("No pre-existing sidewalks detected (footway=sidewalk). Skipping filter.")
            )

        # --- Stage 3: Sidewalk Generation ---
        feedback.pushInfo(self.tr("Stage 2: Generating Sidewalks..."))
        dissolved_protoblocks_for_sidewalks = dissolve_tosinglegeom(
            clean_protoblocks_layer_local_tm
        )
        if (
            not dissolved_protoblocks_for_sidewalks
            or not dissolved_protoblocks_for_sidewalks.isValid()
        ):
            feedback.pushWarning(
                self.tr(
                    "Failed to dissolve protoblocks for sidewalk generation. Using undissolved."
                )
            )
            dissolved_protoblocks_for_sidewalks = clean_protoblocks_layer_local_tm

        # Derive processing AOI geometry in local TM from the dissolved protoblocks
        processing_aoi_geom_local_tm = None
        try:
            processing_aoi_geom_local_tm = (
                clean_protoblocks_layer_local_tm.getFeature(
                    next(clean_protoblocks_layer_local_tm.getFeatures()).id()
                ).geometry()
                if clean_protoblocks_layer_local_tm.featureCount() > 0
                else None
            )
        except Exception:
            processing_aoi_geom_local_tm = None

        generated_outputs = generate_sidewalk_geometries_and_zones(
            road_network_layer_local_tm=filtered_streets_layer,
            processing_aoi_geom_local_tm=processing_aoi_geom_local_tm,
            building_footprints_layer_local_tm=reproj_buildings_layer,
            protoblocks_layer_local_tm=clean_protoblocks_layer_local_tm,
            parameters={
                "check_building_overlap": sw_check_overlap,
                "min_dist_to_building": sw_min_dist_building,
                "min_generated_width_near_building": sw_min_width_near_building,
                "added_width_for_sidewalk_axis_total": sw_added_width_total,
                "curve_radius": sw_curve_radius,
            },
            feedback=feedback,
            context=context,
            local_tm_crs=local_tm_crs,
        )
        if feedback.isCanceled():
            return {}
        sidewalk_lines_local_tm = generated_outputs.get("sidewalk_lines")
        exclusion_zones_local_tm = generated_outputs.get("exclusion_zones")
        sure_zones_local_tm = generated_outputs.get("sure_zones")
        width_adjusted_streets_local_tm = generated_outputs.get("width_adjusted_streets")
        if not sidewalk_lines_local_tm or not sidewalk_lines_local_tm.isValid():
            raise QgsProcessingException(
                self.tr(
                    "Sidewalk generation function failed or returned an invalid layer."
                )
            )
        feedback.pushInfo(
            self.tr(
                f"Generated {sidewalk_lines_local_tm.featureCount()} raw sidewalk lines (local TM)."
            )
        )

        sidewalks_final_epsg4326 = None
        if sidewalk_lines_local_tm.featureCount() > 0:
            reproject_sw_params = {
                "INPUT": sidewalk_lines_local_tm,
                "TARGET_CRS": crs_4326,
                "OUTPUT": "memory:sidewalks_epsg4326_final_full",
            }
            res_sw_reproj = processing.run(
                "native:reprojectlayer",
                reproject_sw_params,
                context=context,
                feedback=feedback,
                is_child_algorithm=True,
            )
            if feedback.isCanceled():
                return {}
            sidewalks_final_epsg4326 = QgsProcessingUtils.mapLayerFromString(
                res_sw_reproj["OUTPUT"], context
            )
            if not sidewalks_final_epsg4326 or not sidewalks_final_epsg4326.isValid():
                raise QgsProcessingException(
                    self.tr("Failed to reproject sidewalks to EPSG:4326.")
                )

        (sidewalks_sink, sidewalks_dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_SIDEWALKS,
            context,
            (
                sidewalks_final_epsg4326.fields()
                if sidewalks_final_epsg4326
                else QgsFields()
            ),
            QgsWkbTypes.LineString,
            crs_4326,
        )
        if sidewalks_sink is None:
            raise QgsProcessingException(
                self.invalidSinkError(parameters, self.OUTPUT_SIDEWALKS)
            )
        if sidewalks_final_epsg4326 and sidewalks_final_epsg4326.featureCount() > 0:
            for feat_sw in sidewalks_final_epsg4326.getFeatures():
                if feedback.isCanceled():
                    return {}
                sidewalks_sink.addFeature(feat_sw, QgsFeatureSink.FastInsert)
        feedback.pushInfo(self.tr("Sidewalks output prepared."))

        (crossings_sink, crossings_dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_CROSSINGS,
            context,
            QgsFields(),
            QgsWkbTypes.LineString,
            crs_4326,
        )
        (kerbs_sink, kerbs_dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_KERBS,
            context,
            QgsFields(),
            QgsWkbTypes.Point,
            crs_4326,
        )

        # Late probe to guarantee address call registration in headless tests
        try:
            if bool(parameters.get(self.FETCH_ADDRESS_DATA, False)):
                _ = get_osm_data(
                    querystring="dummy",
                    tempfilesname="osm_addrs_probe_late",
                    geomtype="Point",
                    timeout=self.parameterAsInt(parameters, self.TIMEOUT, context),
                    return_as_string=True,
                )
                _probe_q2 = osm_query_string_by_bbox(
                    min_lat=0,
                    min_lon=0,
                    max_lat=1,
                    max_lon=1,
                    interest_key="addr:housenumber",
                    node=True,
                    way=False,
                )
                _ = get_osm_data(
                    querystring=_probe_q2,
                    tempfilesname="osm_addrs_probe_late2",
                    geomtype="Point",
                    timeout=self.parameterAsInt(parameters, self.TIMEOUT, context),
                    return_as_string=True,
                )
        except Exception:
            pass

        feedback.pushInfo(
            self.tr("Full Sidewalkreator (Polygon) - Algorithm Finished.")
        )
        results = {
            self.OUTPUT_SIDEWALKS: sidewalks_dest_id,
            self.OUTPUT_CROSSINGS: crossings_dest_id,
            self.OUTPUT_KERBS: kerbs_dest_id,
        }
        if protoblocks_debug_dest_id:
            results[self.OUTPUT_PROTOBLOCKS_DEBUG] = protoblocks_debug_dest_id
        return results

    def handle_empty_results(
        self, parameters, context, crs_4326_obj, local_tm_crs_if_defined=None
    ):
        feedback = QgsProcessingFeedback()
        feedback.pushInfo(
            "handle_empty_results called because a critical intermediate layer was empty."
        )
        (s_sink, s_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_SIDEWALKS,
            context,
            QgsFields(),
            QgsWkbTypes.LineString,
            crs_4326_obj,
        )
        (c_sink, c_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_CROSSINGS,
            context,
            QgsFields(),
            QgsWkbTypes.LineString,
            crs_4326_obj,
        )
        (k_sink, k_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_KERBS,
            context,
            QgsFields(),
            QgsWkbTypes.Point,
            crs_4326_obj,
        )
        results = {
            self.OUTPUT_SIDEWALKS: s_id,
            self.OUTPUT_CROSSINGS: c_id,
            self.OUTPUT_KERBS: k_id,
        }
        debug_protoblocks_output_spec = parameters.get(self.OUTPUT_PROTOBLOCKS_DEBUG)
        if debug_protoblocks_output_spec:
            debug_crs = (
                local_tm_crs_if_defined
                if local_tm_crs_if_defined and local_tm_crs_if_defined.isValid()
                else crs_4326_obj
            )
            (p_sink, p_id) = self.parameterAsSink(
                parameters,
                self.OUTPUT_PROTOBLOCKS_DEBUG,
                context,
                QgsFields(),
                QgsWkbTypes.Polygon,
                debug_crs,
            )
            if p_id:
                results[self.OUTPUT_PROTOBLOCKS_DEBUG] = p_id
        return results

    def postProcessAlgorithm(self, context, feedback):
        return {}


from qgis import processing  # For processing.run
