# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Osm Map Matching
                                 A QGIS plugin
 Plugin che fa qualche cosa
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-07-26
        copyright            : (C) 2023 by G.Rossi
        email                : grossi@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'G.Rossi'
__date__ = '2023-07-26'
__copyright__ = '(C) 2023 by G.Rossi'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from PyQt5.QtGui import QIcon
from qgis.core import (QgsProcessing,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterNumber,
                       QgsCoordinateTransform,
                       QgsCoordinateReferenceSystem,
                       QgsProcessingParameterFileDestination,
                       QgsVectorFileWriter,
                       QgsProject)

from geopy.point import Point


from .ta import main as ta
from . import output

class OsmMapMatchingAlgorithm(QgsProcessingAlgorithm):
    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    VPOINTS = 'VECTOR_POINTS'
    MAXDIST = 'MAXDIST'
    MINLOOPSIZE = "MINLOOPSIZE"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.VPOINTS,
                self.tr('Vector Points'),
                [QgsProcessing.TypeVectorAnyGeometry]
            )
        )

        
        self.addParameter(
            QgsProcessingParameterNumber(
                self.MAXDIST,
                self.tr('Max distance from map point (meters)'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=30.0
            )
        )
        

        self.addParameter(
            QgsProcessingParameterNumber(
                self.MINLOOPSIZE,
                self.tr('Minimum loop size (meters)'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=15.0
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
              QgsProcessingParameterFileDestination(
                  self.OUTPUT,
                  self.tr('Output File'),
                  'ESRI Shapefiles (*.shp)',
              )
        )


    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        
        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.VPOINTS, context)
        out_shapefile = self.parameterAsFileOutput(parameters, self.OUTPUT, context)
        max_dist = self.parameterAsDouble(parameters, self.MAXDIST, context)
        min_loop_size = self.parameterAsDouble(parameters, self.MINLOOPSIZE, context)
        

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / source.featureCount() if source.featureCount() else 0
        features = source.getFeatures()        
        
        # the point must be reprojevted in EPSG:4326
        source_crs = QgsCoordinateReferenceSystem(source.sourceCrs().authid())
        target_crs = QgsCoordinateReferenceSystem("EPSG:4326")

        feedback.pushInfo('Vector reading')
        
        points_list = []

        for current, f in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            p = f.geometry().asPoint()
            
            geom = f.geometry()
            geom.transform(QgsCoordinateTransform(source_crs, target_crs, QgsProject.instance()))
            p = geom.asPoint()
             
            points_list.append(Point(latitude=p.y(), longitude=p.x()))

            feedback.setProgress(int(current * total))
            
        feedback.pushInfo('Analizing')
        G, path = ta.analyze(points_list, max_dist, min_loop_size, feedback)
        
        if path != None:
            out_df = ta.make_out_dataframe(G, path, log=feedback)
            feedback.pushInfo('Vector output creation')
            olayer = output.make_vector(out_df)
            QgsProject.instance().addMapLayer(olayer)
            
            # Write to an ESRI Shapefile format dataset using UTF-8 text encoding
            save_options = QgsVectorFileWriter.SaveVectorOptions()
            save_options.driverName = "ESRI Shapefile"
            save_options.fileEncoding = "UTF-8"
            transform_context = QgsProject.instance().transformContext()
            error = QgsVectorFileWriter.writeAsVectorFormatV3(olayer,
                                                              out_shapefile,
                                                              transform_context,
                                                              save_options)
            if error[0] == QgsVectorFileWriter.NoError:
                feedback.pushInfo("Output file created")
            else:
                feedback.pushInfo(error[0])
        else:
            out_shapefile = None
            

        return {self.OUTPUT: out_shapefile}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Analyze'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return '' # 

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return OsmMapMatchingAlgorithm()

    def icon(self):
        import os
        iconname = os.path.join(os.path.dirname(__file__), 'icon.png')
        #iconname = ''
        return QIcon(iconname)

    def shortHelpString(self):
        # TODO
        return '''This algorithm takes a vector layer of points that defines a route and creates a line layer that aligns this route with roads, streets, or paths from OpenStreetMap.
        
        The fields in the original OpenStreetMap data are applied to the lines in the output layer.
        
        For more information https://github.com/glucatv/osm_map_matching
        '''
