#! python3  # noqa: E265

"""
    Perform network request.
"""

# ############################################################################
# ########## Imports ###############
# ##################################

# standard library
from pathlib import Path
from typing import Optional, Union

# PyQGIS
from qgis.core import QgsApplication, QgsBlockingNetworkRequest, QgsFileDownloader
from qgis.PyQt.QtCore import QByteArray, QCoreApplication, QEventLoop, QUrl
from qgis.PyQt.QtNetwork import QNetworkReply, QNetworkRequest

# project
from oslandia.__about__ import __title__, __version__
from oslandia.toolbelt.log_handler import PlgLogger
from oslandia.toolbelt.preferences import PlgOptionsManager

# ############################################################################
# ########## Classes ###############
# ##################################


class NetworkRequestsManager:
    """Helper on network operations."""

    def __init__(self):
        """Initialization."""
        self.log = PlgLogger().log
        self.ntwk_requester = QgsBlockingNetworkRequest()

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)

    def build_request(
        self,
        url: QUrl = None,
        http_content_type: str = "application/json",
        http_user_agent: str = f"{__title__}/{__version__}",
    ) -> QNetworkRequest:
        """Build request object using plugin settings.

        :param url: request url, defaults to None
        :type url: QUrl, optional
        :param http_content_type: content type, defaults to "application/json"
        :type http_content_type: str, optional
        :param http_user_agent: http user agent, defaults to f"{__title__}/{__version__}"
        :type http_user_agent: str, optional
        :return: network request object.
        :rtype: QNetworkRequest
        """
        # create network object
        qreq = QNetworkRequest(url=url)

        # headers
        headers = {
            b"Accept": bytes(http_content_type, "utf8"),
            b"User-Agent": bytes(http_user_agent, "utf8"),
        }
        for k, v in headers.items():
            qreq.setRawHeader(k, v)

        return qreq

    def get_url(self, url: QUrl, config_id) -> QByteArray:
        """Send a get method.
        :raises ConnectionError: if any problem occurs during feed fetching.
        :raises TypeError: if response mime-type is not valid

        :return: feed response in bytes
        :rtype: QByteArray

        :example:

        .. code-block:: python

            import json
            response_as_dict = json.loads(str(response, "UTF8"))
        """

        auth_manager = QgsApplication.authManager()
        req = QNetworkRequest(url)
        auth_manager.updateNetworkRequest(req, config_id)

        # send request
        try:
            req_status = self.ntwk_requester.get(
                request=req,
                forceRefresh=True,
            )

            # check if request is fine
            if req_status != QgsBlockingNetworkRequest.NoError:
                self.log(
                    message=self.ntwk_requester.errorMessage(), log_level=2, push=True
                )
                raise ConnectionError(self.ntwk_requester.errorMessage())

            req_reply = self.ntwk_requester.reply()

            if req_reply.error() != QNetworkReply.NetworkError.NoError:
                self.log(message=req_reply.errorString(), log_level=2, push=1)
                raise ConnectionError(req_reply.errorString())

            if PlgOptionsManager.get_plg_settings().debug_mode:
                self.log(
                    message=self.tr("GET response: {}").format(
                        req_reply.content().data().decode("utf-8")
                    ),
                    log_level=4,
                    push=False,
                )

            return req_reply.content()

        except Exception as err:

            err_msg = self.tr("Houston, we've got a problem: {}".format(err))
            self.log(message=err_msg, log_level=2, push=True)
            return QByteArray()

    def post_url(
        self,
        url: QUrl,
        data: QByteArray = None,
        config_id: Optional[str] = None,
        content_type_header: str = "",
    ) -> QByteArray:
        """Send a post method with data option.
        :raises ConnectionError: if any problem occurs during feed fetching.
        :raises TypeError: if response mime-type is not valid

        :param url: url
        :type url: QUrl
        :param data: data for post, defaults to None
        :type data: QByteArray, optional
        :param config_id: QGIS auth config ID, defaults to None
        :type config_id: str, optional
        :param content_type_header: content type header for request, defaults to ""
        :type content_type_header: str, optional
        :return: feed response in bytes
        :rtype: QByteArray
        """
        auth_manager = QgsApplication.authManager()
        req = QNetworkRequest(url)
        if content_type_header:
            req.setHeader(
                QNetworkRequest.KnownHeaders.ContentTypeHeader, content_type_header
            )
        auth_manager.updateNetworkRequest(req, config_id)

        # send request
        try:
            req_status = self.ntwk_requester.post(request=req, data=data)

            # check if request is fine
            if req_status != QgsBlockingNetworkRequest.NoError:
                self.log(
                    message=self.ntwk_requester.errorMessage(), log_level=2, push=True
                )
                raise ConnectionError(self.ntwk_requester.errorMessage())

            req_reply = self.ntwk_requester.reply()

            if req_reply.error() != QNetworkReply.NetworkError.NoError:
                self.log(message=req_reply.errorString(), log_level=2, push=True)
                raise ConnectionError(req_reply.errorString())

            if PlgOptionsManager.get_plg_settings().debug_mode:
                self.log(
                    message=self.tr("POST response: {}").format(
                        req_reply.content().data().decode("utf-8")
                    ),
                    log_level=4,
                    push=False,
                )

            return req_reply.content()

        except Exception as err:
            err_msg = self.tr("Houston, we've got a problem: {}".format(err))
            self.log(message=err_msg, log_level=2, push=True)

    def put_url(
        self,
        url: QUrl,
        data: QByteArray = None,
        config_id: Optional[str] = None,
        content_type_header: str = "",
    ) -> QByteArray:
        """Send a put method with data option.
        :raises ConnectionError: if any problem occurs during feed fetching.
        :raises TypeError: if response mime-type is not valid

        :param url: url
        :type url: QUrl
        :param data: data for put, defaults to None
        :type data: QByteArray, optional
        :param config_id: QGIS auth config ID, defaults to None
        :type config_id: str, optional
        :param content_type_header: content type header for request, defaults to ""
        :type content_type_header: str, optional
        :return: feed response in bytes
        :rtype: QByteArray
        """
        auth_manager = QgsApplication.authManager()
        req = QNetworkRequest(url)
        if content_type_header:
            req.setHeader(
                QNetworkRequest.KnownHeaders.ContentTypeHeader, content_type_header
            )
        auth_manager.updateNetworkRequest(req, config_id)

        # send request
        try:
            req_status = self.ntwk_requester.put(request=req, data=data)

            # check if request is fine
            if req_status != QgsBlockingNetworkRequest.NoError:
                self.log(
                    message=self.ntwk_requester.errorMessage(), log_level=2, push=True
                )
                raise ConnectionError(self.ntwk_requester.errorMessage())

            req_reply = self.ntwk_requester.reply()

            if req_reply.error() != QNetworkReply.NetworkError.NoError:
                self.log(message=req_reply.errorString(), log_level=2, push=True)
                raise ConnectionError(req_reply.errorString())

            if PlgOptionsManager.get_plg_settings().debug_mode:
                self.log(
                    message=self.tr("PUT response: {}").format(
                        req_reply.content().data().decode("utf-8")
                    ),
                    log_level=4,
                    push=False,
                )

            return req_reply.content()

        except Exception as err:
            err_msg = self.tr("Houston, we've got a problem: {}".format(err))
            self.log(message=err_msg, log_level=2, push=True)

    def download_file_to(self, remote_url: str, local_path: Union[Path, str]) -> str:
        """Download a file from a remote web server accessible through HTTP.

        :param remote_url: remote URL
        :type remote_url: str
        :param local_path: path to the local file
        :type local_path: str
        :return: output path
        :rtype: str
        """
        # check if destination path is a str and if parent folder exists
        if isinstance(local_path, Path):
            local_path.parent.mkdir(parents=True, exist_ok=True)
            local_path = f"{local_path.resolve()}"
        elif isinstance(local_path, str):
            Path(local_path).parent.mkdir(parents=True, exist_ok=True)

        self.log(
            message=f"Downloading file from {remote_url} to {local_path}", log_level=4
        )
        # download it
        loop = QEventLoop()
        file_downloader = QgsFileDownloader(
            url=QUrl(remote_url), outputFileName=local_path, delayStart=True
        )
        file_downloader.downloadExited.connect(loop.quit)
        file_downloader.startDownload()
        loop.exec()

        self.log(
            message=f"Download of {remote_url} to {local_path} succeedeed", log_level=3
        )
        return local_path
