#! python3  # noqa: E265

"""
    Authentication dialog logic.
"""

# standard
from pathlib import Path

from qgis.core import QgsApplication
from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import QDialog, QMessageBox

from oslandia.gitlab_api.custom_exceptions import UnavailableUserException

# PyQGIS
from oslandia.gitlab_api.user import UserRequestManager
from oslandia.gui.dlg_authentication import AuthenticationDialog

# Plugin
from oslandia.toolbelt import PlgLogger, PlgOptionsManager


class UserDialog(QDialog):
    def __init__(self, parent=None):
        """Dialog to display current user and change user if needed.

        :param parent: parent widget, defaults to None
        :type parent: QObject, optional
        """
        # init module and ui
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        # toolbelt
        self.log = PlgLogger().log
        self.plg_settings_mngr = PlgOptionsManager()
        self.plg_settings = self.plg_settings_mngr.get_plg_settings()

        self.btn_change_user.setIcon(QgsApplication.getThemeIcon("user.svg"))
        self.btn_change_user.clicked.connect(self._change_user)
        self.update_user_widget()

        self.buttonBox.accepted.connect(self.accept)

    def update_user_widget(self) -> None:
        """Update user widget with current user"""
        try:
            manager = UserRequestManager()
            user = manager.get_current_user()
            self.wdg_user.set_user(user)
        except UnavailableUserException as exc:
            QMessageBox.information(
                self,
                self.tr("Error"),
                self.tr(f"Can't define current user : {exc}"),
            )

    def _change_user(self) -> None:
        """Display AuthenticationDialog to change connected user"""
        dlg = AuthenticationDialog(self)
        if dlg.exec():
            self.update_user_widget()
