import shutil
from pathlib import Path

import oslandia.toolbelt.log_handler as log_hdlr
from oslandia.toolbelt.application_folder import get_app_dir


class CacheManager:
    """Class for local cache management."""

    def __init__(self, app_prefix: str, dir_name: str):
        """Init CacheManager class
        For example on  linux with app_prefix = .qgis and dir_name = oslandia
        cache_path = /home/toto/.qgis/plugin_oslandia/

        :param app_prefix: Name of folder to prefix cache path
        :type app_prefix: str
        :param dir_name: Name of filder un prefix cache path
        :type dir_name: str
        """

        self.cache_dir = get_app_dir(dir_name=dir_name, app_prefix=app_prefix)

    @property
    def get_cache_path(self) -> Path:
        """Return cache path

        :return: Cache path
        :rtype: Path
        """
        return self.cache_dir

    def ensure_cache_dir_exists(self) -> bool:
        """Check if cache_dir exists

        :return: True the cache dir already exists, otherwise false.
        :rtype: bool
        """
        if not self.cache_dir.exists():
            log_hdlr.PlgLogger.log(
                message=f"The cache folder {self.cache_dir} doesn't exist.",
                log_level=0,
            )
            return False
        else:
            log_hdlr.PlgLogger.log(
                message=f"Cache dir {self.cache_dir} already exists.",
                log_level=0,
            )
            return True

    def create_cache_dir(self) -> None:
        """Create cache_dir"""
        if not self.ensure_cache_dir_exists():
            self.cache_dir.mkdir(parents=True)
            log_hdlr.PlgLogger.log(
                message=f"Cache dir {self.cache_dir} has been created.",
                log_level=0,
            )

    def clear_cache(self) -> None:
        """Delete the cache_dir project"""
        if self.ensure_cache_dir_exists():
            shutil.rmtree(self.cache_dir)
            log_hdlr.PlgLogger.log(
                message=f"Cache dir {self.cache_dir} has been delated.",
                log_level=0,
            )
