#! python3  # noqa: E265

"""
    Main plugin module.
"""


# standard
from functools import partial
from pathlib import Path

from qgis.core import QgsApplication, QgsSettings
from qgis.gui import QgisInterface, QgsGui
from qgis.PyQt.QtCore import QCoreApplication, QLocale, QTranslator, QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QAction, QToolBar

# project
from oslandia.__about__ import (
    DIR_PLUGIN_ROOT,
    __icon_path__,
    __title__,
    __uri_homepage__,
)
from oslandia.gitlab_api.custom_exceptions import UnavailableUserException
from oslandia.gitlab_api.user import UserRequestManager

# PyQGIS
from oslandia.gui.dlg_authentication import AuthenticationDialog
from oslandia.gui.dlg_create_issue import CreateIssueDialog
from oslandia.gui.dlg_plugins_browser import PluginsBrowser
from oslandia.gui.dlg_settings import PlgOptionsFactory
from oslandia.gui.dlg_user import UserDialog
from oslandia.gui.dlg_view_issues import ViewIssueDialog
from oslandia.newsfeed.rss_reader import RssReader
from oslandia.toolbelt import PlgLogger
from oslandia.toolbelt.plugin_manager import PluginManager

# ############################################################################
# ########## Classes ###############
# ##################################


class OslandiaPlugin:
    def __init__(self, iface: QgisInterface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class which \
        provides the hook by which you can manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.wizard = None
        self.log = PlgLogger().log
        self.action_settings = None
        self.action_help = None
        self.rss_reader = RssReader()
        self.action_create_issue = None
        self.dlg_create_issue = None
        self.action_view_issue = None
        self.action_connect = None
        self.action_user = None
        self.dlg_view_issue = None
        self.display_oslandia_plugins = None
        self.plugin_manager = PluginManager()
        self.plugin_manager.update_json_plugins()
        self.plugin_manager.download_plugins_icons_to_cache()

        # translation
        # initialize the locale
        self.locale: str = QgsSettings().value("locale/userLocale", QLocale().name())[
            0:2
        ]
        locale_path: Path = (
            DIR_PLUGIN_ROOT
            / "resources"
            / "i18n"
            / f"{__title__.lower()}_{self.locale}.qm"
        )
        self.log(message=f"Translation: {self.locale}, {locale_path}", log_level=4)
        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path.resolve()))
            QCoreApplication.installTranslator(self.translator)

        # plugin settings

        self.toolbar = None

        self._dlg_display_plugins = None

    def initGui(self):
        """Set up plugin UI elements."""
        self.iface.initializationCompleted.connect(self.post_ui_init)

        # settings page within the QGIS preferences menu
        self.options_factory = PlgOptionsFactory()
        self.iface.registerOptionsWidgetFactory(self.options_factory)

        # -- Actions
        self.action_help = QAction(
            QgsApplication.getThemeIcon("mActionHelpContents.svg"),
            self.tr("Help"),
            self.iface.mainWindow(),
        )
        self.action_help.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.action_settings = QAction(
            QgsApplication.getThemeIcon("console/iconSettingsConsole.svg"),
            self.tr("Settings"),
            self.iface.mainWindow(),
        )
        self.action_settings.triggered.connect(
            lambda: self.iface.showOptionsDialog(
                currentPage="mOptionsPage{}".format(__title__)
            )
        )

        self.action_create_issue = QAction(
            QIcon(str(DIR_PLUGIN_ROOT / "resources/images/create_issue.png")),
            self.tr("Create issue"),
            self.iface.mainWindow(),
        )

        self.action_create_issue.triggered.connect(self._create_issue)

        self.action_view_issue = QAction(
            QIcon(str(DIR_PLUGIN_ROOT / "resources/images/view_issue.png")),
            self.tr("View issue"),
            self.iface.mainWindow(),
        )

        self.display_oslandia_plugins = QAction(
            QIcon(str(DIR_PLUGIN_ROOT / "resources/images/plugin.png")),
            self.tr("Plugins by Oslandia"),
            self.iface.mainWindow(),
        )

        self.action_view_issue.triggered.connect(self._view_issue)

        self.action_connect = QAction(
            QIcon(str(DIR_PLUGIN_ROOT / "resources/images/connect.png")),
            self.tr("Connect"),
            self.iface.mainWindow(),
        )

        self.action_connect.triggered.connect(self._connect)

        self.action_user = QAction(
            QgsApplication.getThemeIcon("user.svg"),
            self.tr("User"),
            self.iface.mainWindow(),
        )

        self.action_user.triggered.connect(self._user)

        # -- Menu
        self.iface.addPluginToMenu(__title__, self.action_connect)
        self.iface.addPluginToMenu(__title__, self.action_user)
        self.iface.addPluginToMenu(__title__, self.action_create_issue)
        self.iface.addPluginToMenu(__title__, self.action_view_issue)
        self.iface.addPluginToMenu(__title__, self.display_oslandia_plugins)
        self.iface.addPluginToMenu(__title__, self.action_settings)
        self.iface.addPluginToMenu(__title__, self.action_help)

        # -- Toolbar
        self.toolbar = QToolBar("OslandiaToolbar")
        self.iface.addToolBar(self.toolbar)
        self.toolbar.addAction(self.action_connect)
        self.toolbar.addAction(self.action_user)
        self.toolbar.addAction(self.action_view_issue)
        self.toolbar.addAction(self.action_create_issue)
        self.toolbar.addAction(self.display_oslandia_plugins)

        # documentation
        self.iface.pluginHelpMenu().addSeparator()
        self.action_help_plugin_menu_documentation = QAction(
            QIcon(str(__icon_path__)),
            f"{__title__} - Documentation",
            self.iface.mainWindow(),
        )
        self.action_help_plugin_menu_documentation.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.iface.pluginHelpMenu().addAction(
            self.action_help_plugin_menu_documentation
        )

        self.display_oslandia_plugins.triggered.connect(
            self.display_oslandia_plugins_dialog
        )

        self.action_create_issue.setVisible(False)
        self.action_view_issue.setVisible(False)
        instance_gui = QgsGui.instance()
        instance_gui.optionsChanged.connect(self._check_available_actions)
        self.iface.initializationCompleted.connect(self._check_available_actions)

    def _check_available_actions(self) -> None:
        """Define available actions from current connection"""
        # Check connection to API by getting token user information
        try:
            manager = UserRequestManager()
            manager.get_current_user()
            self.action_connect.setVisible(False)
            self.action_user.setVisible(True)
            self.action_create_issue.setVisible(True)
            self.action_view_issue.setVisible(True)
        except UnavailableUserException:
            self.action_connect.setVisible(True)
            self.action_user.setVisible(False)
            self.action_create_issue.setVisible(False)
            self.action_view_issue.setVisible(False)

    def _create_issue(self):
        """Display create issue dialog"""
        if not self.dlg_create_issue:
            self.dlg_create_issue = CreateIssueDialog()
            self.dlg_create_issue.setModal(True)
        self.dlg_create_issue.show()

    def _view_issue(self):
        """Display view issue dialog"""
        if not self.dlg_view_issue:
            self.dlg_view_issue = ViewIssueDialog()
            self.dlg_view_issue.setModal(True)
        self.dlg_view_issue.show()

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)

    def unload(self):
        """Cleans up when plugin is disabled/uninstalled."""
        # -- Clean up menu
        self.iface.removePluginMenu(__title__, self.action_help)
        self.iface.removePluginMenu(__title__, self.action_settings)
        self.iface.removePluginMenu(__title__, self.action_connect)
        self.iface.removePluginMenu(__title__, self.action_user)
        self.iface.removePluginMenu(__title__, self.action_create_issue)
        self.iface.removePluginMenu(__title__, self.action_view_issue)
        self.iface.removePluginMenu(__title__, self.display_oslandia_plugins)
        # -- Clean up preferences panel in QGIS settings
        self.iface.unregisterOptionsWidgetFactory(self.options_factory)

        # remove from QGIS help/extensions menu
        if self.action_help_plugin_menu_documentation:
            self.iface.pluginHelpMenu().removeAction(
                self.action_help_plugin_menu_documentation
            )

        # remove actions
        del self.action_settings
        del self.action_help

        # remove toolbar :
        self.toolbar.deleteLater()

    def post_ui_init(self):
        """Things to do when QGIS ui is completed, without blocking QGIS
            initialization (First Content Pain).

        Becareful, this signal is only emitted when the interface is initialized, not
        refreshed or modified. For example, it's not emitted when the plugin is
        manually activated through the extensions manager and nor when another
        processing modify the QGIS GUI.
        """
        self.rss_reader.process()

    def run(self):
        """Main process.

        :raises Exception: if there is no item in the feed
        """
        try:
            self.log(
                message=self.tr("Everything ran OK."),
                log_level=3,
                push=False,
            )
        except Exception as err:
            self.log(
                message=self.tr("Houston, we've got a problem: {}".format(err)),
                log_level=2,
                push=True,
            )

    def display_oslandia_plugins_dialog(self) -> None:
        """Display instance duckdb add layer dialog"""
        if self._dlg_display_plugins is None:
            self._dlg_display_plugins = PluginsBrowser()

        self._dlg_display_plugins.show()

    def _connect(self) -> None:
        connection_dialog = AuthenticationDialog(self.iface.mainWindow())
        if connection_dialog.exec_():
            self._check_available_actions()

    def _user(self) -> None:
        user_dialog = UserDialog(self.iface.mainWindow())
        user_dialog.exec_()
