from qgis.PyQt.QtCore import QModelIndex, QObject, Qt, QVariant
from qgis.PyQt.QtGui import QColor, QStandardItemModel

from oslandia.gitlab_api.issue import Issues, IssuesRequestManager
from oslandia.toolbelt import PlgLogger


class IssuesListModel(QStandardItemModel):
    CREATED_DATE_COL = 0
    NAME_COL = 1
    STATE_COL = 2
    WEB_URL_COL = 3
    DESCRIPTION_COL = 4

    def __init__(self, parent: QObject = None):
        """
        QStandardItemModel for datastore list display

        Args:
            parent: QObject parent
        """
        super().__init__(parent)
        self.log = PlgLogger().log
        self.setHorizontalHeaderLabels(
            [self.tr("Created"), self.tr("Title"), self.tr("State"), self.tr("URL")]
        )

    def flags(self, index: QModelIndex):
        """Define flag to disable edition

        :param index: model index
        :type index: QModelIndex
        :return: _description_
        :rtype: _type_
        """
        default_flags = super().flags(index)
        return default_flags & ~Qt.ItemIsEditable  # Disable editing

    def set_project(self, project_id: str) -> None:
        """
        Refresh QStandardItemModel data with current project_id members

        """
        self.removeRows(0, self.rowCount())

        manager = IssuesRequestManager()
        issues = manager.get_issues_list(project_id)
        for issue in issues:
            self.insert_issues(issue)

    def data(self, index: QModelIndex, role=Qt.DisplayRole):
        """Custom data for background role

        :param index: model index
        :type index: QModelIndex
        :param role: data role, defaults to Qt.DisplayRole
        :type role: _type_, optional
        :return: _description_
        :rtype: _type_
        """
        if not index.isValid():
            return QVariant()

        if (
            role == Qt.BackgroundRole
            and self.data(self.index(index.row(), self.STATE_COL), Qt.DisplayRole)
            != "opened"
        ):
            return QColor("whitesmoke")

        return super().data(index, role)

    def insert_issues(self, issues: Issues) -> None:
        """Insert members into model

        :param members: members to insert
        :type members: members

        """
        row = self.rowCount()
        self.insertRow(row)
        self.setData(self.index(row, self.CREATED_DATE_COL), issues.created_date)
        self.setData(self.index(row, self.STATE_COL), issues.state)
        self.setData(self.index(row, self.WEB_URL_COL), f"#{issues.iid}")
        self.setData(self.index(row, self.WEB_URL_COL), issues.web_url, Qt.UserRole)
        self.setData(self.index(row, self.WEB_URL_COL), issues.web_url, Qt.ToolTipRole)
        self.setData(self.index(row, self.DESCRIPTION_COL), issues.description)

        self.setData(self.index(row, self.NAME_COL), issues.name)
        self.setData(self.index(row, self.NAME_COL), issues, Qt.UserRole)
        self.setData(self.index(row, self.NAME_COL), issues.description, Qt.ToolTipRole)


# class IssueView(QStandardItemModel):
#     NAME_COL = 0
#     DESCRIPTION_COL = 1

#     def __init__(self, parent: QObject = None):
#         """
#         QStandardItemModel for datastore list display

#         Args:
#             parent: QObject parent
#         """
#         super().__init__(parent)
#         self.log = PlgLogger().log
#         self.setHorizontalHeaderLabels(
#             [ self.tr("name"), self.tr("description")]
#         )

#     def flags(self, index):
#         default_flags = super().flags(index)
#         return default_flags & ~Qt.ItemIsEditable  # Disable editing

#     def set_issue(self, project_id: str, issue_id: str) -> None:
#         """
#         Refresh QStandardItemModel data with current project_id members

#         """
#         self.removeRows(0, self.rowCount())

#         manager = IssuesRequestManager()
#         issue = manager.get_issue(project_id, issue_id)
#         self.insert_issues(issue)
#         return issue
#     def insert_issues(self, issues: Issues) -> None:
#         """Insert members into model

#         :param members: members to insert
#         :type members: members

#         """
#         row = self.rowCount()
#         self.insertRow(row)
#         self.setData(self.index(row, self.DESCRIPTION_COL), issues.description)
#         self.setData(self.index(row, self.NAME_COL), issues.name)
#         self.setData(self.index(row, self.NAME_COL), issues, Qt.UserRole)
