import json
from dataclasses import dataclass
from datetime import datetime
from typing import List

# PyQGIS
from qgis.PyQt.Qt import QUrl

# plugin
from oslandia.toolbelt import PlgLogger, PlgOptionsManager
from oslandia.toolbelt.network_manager import NetworkRequestsManager


@dataclass
class Project:
    id: str
    name: str
    group: str
    archived: bool
    created_at: datetime


class ProjectRequestManager:

    MAX_LIMIT = 50

    def get_base_url_no_group(self) -> str:
        """
        Get base url for project of a group

        Args:
            group: (str) group id

        Returns: url for projects info

        """
        return f"{self.plg_settings.gitlab_api_url_base}"

    def get_base_url(self, group: str) -> str:
        """
        Get base url for project of a group

        Args:
            group: (str) group id

        Returns: url for projects info

        """
        return f"{self.plg_settings.gitlab_api_url_base}groups/{group}/projects"

    def __init__(self):
        """
        Helper for stored_data request

        """
        self.log = PlgLogger().log
        self.ntwk_requester_blk = NetworkRequestsManager()
        self.plg_settings = PlgOptionsManager.get_plg_settings()

    def get_project_list(self, group: str) -> List[Project]:
        """
        Get list of project

        Args:
            group: (str) group id

        Returns: list of available project, raise ReadStoredDataException otherwise
        """
        self.log(f"{__name__}.get_project_list(group:{group})")

        i = 1
        result = []
        result_page = self._get_project_list(group, i, self.MAX_LIMIT)
        while len(result_page) != 0:
            result += result_page
            i = i + 1
            result_page = self._get_project_list(group, i, self.MAX_LIMIT)
        return result

    def _get_project_list(
        self, group: str, page: int = 1, limit: int = MAX_LIMIT
    ) -> List[Project]:
        """
        Get list of project

        Args:
            group: (str) group id
            page: (int) page number (start at 1)
            limit: (int)

        Returns: list of available project
        """

        # send request
        req_reply = self.ntwk_requester_blk.get_url(
            QUrl(f"{self.get_base_url(group)}?page={page}&per_page={limit}"),
            self.plg_settings.authentification_config_id,
        )
        if req_reply:
            data = json.loads(req_reply.data().decode("utf-8"))

            return [
                Project(
                    id=p_data["id"],
                    name=p_data["name"],
                    group=p_data["namespace"]["name"],
                    archived=p_data["archived"],
                    created_at=datetime.fromisoformat(p_data["created_at"].strip()),
                )
                for p_data in data
            ]
        return []
