# standard
import json
from dataclasses import dataclass, field
from typing import List

# PyQGIS
from qgis.PyQt.Qt import QUrl

# plugin
from oslandia.gitlab_api.user import User
from oslandia.toolbelt import PlgLogger, PlgOptionsManager
from oslandia.toolbelt.network_manager import NetworkRequestsManager


@dataclass
class Note:
    id: str
    body: str
    system: bool
    author: User
    created_at: str
    updated_at: str


@dataclass
class Discussion:
    id: str
    notes: List[Note] = field(default_factory=lambda: [])


class DiscussionsRequestManager:

    MAX_LIMIT = 50

    def get_base_url(self, project_id: str, issue_iid: str) -> str:
        """
        Get base url for discussion of an issue

        Args:
            project_id: (str) project id
            issue_iid: (str) issue id

        Returns: url for discussion info

        """
        return f"{self.plg_settings.gitlab_api_url_base}projects/{project_id}/issues/{issue_iid}/discussions"

    def __init__(self):
        """
        Helper for discussions request

        """
        self.log = PlgLogger().log
        self.ntwk_requester_blk = NetworkRequestsManager()
        self.plg_settings = PlgOptionsManager.get_plg_settings()

    def get_discussions_list(self, project_id: str, issue_iid: str) -> List[Discussion]:
        """
        Get list of discussion

        Args:
            project: (str) project id
            issue_iid: (str) issue id

        Returns: list of available discussion
        """
        self.log(
            f"{__name__}.get_discussion_list(project:{project_id},project:{project_id})"
        )

        i = 1
        result = []
        result_page = self._get_discussions_list(
            project_id, issue_iid, i, self.MAX_LIMIT
        )
        while len(result_page) != 0:
            result += result_page
            i = i + 1
            result_page = self._get_discussions_list(
                project_id, issue_iid, i, self.MAX_LIMIT
            )

        return result

    def _get_discussions_list(
        self, project_id: str, issue_iid: str, page: int = 1, limit: int = MAX_LIMIT
    ) -> List[Discussion]:
        """
        Get list of discussions

        Args:
            project_id: (str) project id
            issue_iid: (str) issue iid
            page: (int) page number (start at 1)
            limit: (int)

        Returns: list of available discussions
        """

        # send request
        req_reply = self.ntwk_requester_blk.get_url(
            QUrl(
                f"{self.get_base_url(project_id,issue_iid)}?page={page}&per_page={limit}"
            ),
            self.plg_settings.authentification_config_id,
        )
        if req_reply:
            data = json.loads(req_reply.data().decode("utf-8"))
            return [
                Discussion(
                    id=p_data["id"],
                    notes=[
                        Note(
                            id=n_data["id"],
                            body=n_data["body"],
                            system=n_data["system"],
                            created_at=n_data["created_at"],
                            updated_at=n_data["updated_at"],
                            author=User(
                                id=n_data["author"]["id"],
                                name=n_data["author"]["name"],
                                username=n_data["author"]["username"],
                            ),
                        )
                        for n_data in p_data["notes"]
                    ],
                )
                for p_data in data
            ]
        return []
