from datetime import datetime, timedelta
from pathlib import Path

from oslandia.toolbelt.cache_manager import CacheManager
from oslandia.toolbelt.json_tools import json_to_dict
from oslandia.toolbelt.network_manager import NetworkRequestsManager


class PluginManager:
    def __init__(self):
        """Initialization."""
        self.PUBLIC_PLUGIN_URL = (
            "https://oslandia.gitlab.io/qgis/oslandia/public_plugins.json"
        )
        self.GITLAB_PLUGIN_URL = (
            "https://oslandia.gitlab.io/qgis/oslandia/gitlab_plugins.json"
        )
        self.PRIVATE_PLUGIN_URL = (
            "https://oslandia.gitlab.io/qgis/oslandia/private_plugins.json"
        )

        self.cache_manager = CacheManager(".qgis", "plugin_oslandia")
        self.network_manager = NetworkRequestsManager()

        # cache path json
        self.public_plugins = Path(
            self.cache_manager.get_cache_path / "public_plugins.json"
        )
        self.private_plugins = Path(
            self.cache_manager.get_cache_path / "private_plugins.json"
        )
        self.gitlab_plugins = Path(
            self.cache_manager.get_cache_path / "gitlab_plugins.json"
        )

    def update_json_plugins(self):
        """Update the list of plugins in the cache manager if they are more than a week old."""
        if not self.cache_manager.ensure_cache_dir_exists():
            self.cache_manager.create_cache_dir()

        self.download_if_needed(self.public_plugins, self.PUBLIC_PLUGIN_URL)
        self.download_if_needed(self.private_plugins, self.PRIVATE_PLUGIN_URL)
        self.download_if_needed(self.gitlab_plugins, self.GITLAB_PLUGIN_URL)

    def force_update_json_plugins(self):
        """Update the list of plugins in the cache manager without regard to date."""
        if not self.cache_manager.ensure_cache_dir_exists():
            self.cache_manager.create_cache_dir()

        # Directly download without checking the file age
        self.network_manager.download_file_to(
            self.PUBLIC_PLUGIN_URL, self.public_plugins
        )
        self.network_manager.download_file_to(
            self.PRIVATE_PLUGIN_URL, self.private_plugins
        )
        self.network_manager.download_file_to(
            self.GITLAB_PLUGIN_URL, self.gitlab_plugins
        )

    def download_if_needed(self, file_path: Path, url: str, days: int = 7) -> None:
        """Check if a download is required. A download is required if the file does not
        exist or is more than x days old (defined by days).

        :param file_path: Path file
        :type file_path: Path
        :param url: URL to download
        :type url: str
        :param days: File expiration time, defaults to 7
        :type days: int, optional
        """
        if not file_path.exists() or datetime.now() - datetime.fromtimestamp(
            file_path.stat().st_mtime
        ) >= timedelta(days=days):
            self.network_manager.download_file_to(url, file_path)
            self.network_manager.download_file_to(url, file_path)

    def get_last_modification_date(self, file_path: Path) -> datetime:
        """Return the last modification date of a file.

        :param file_path: Path to the file
        :type file_path: Path
        :return: The datetime of the last modification
        :rtype: datetime
        :raises FileNotFoundError: If the file does not exist
        """
        if not file_path.exists():
            raise FileNotFoundError("The specified file does not exist.")
        return datetime.fromtimestamp(file_path.stat().st_mtime)

    @property
    def get_date_json_plugins_update(self) -> str:
        """Check the last modification date of a specific file.

        :param file_name: Name of the file to check
        :type file_name: str
        :return: Last modification date and time as a formatted string
        :rtype: str
        :raises FileNotFoundError: If the file does not exist
        """
        file_path = Path(self.public_plugins)
        last_modification_date = self.get_last_modification_date(file_path)
        return last_modification_date.strftime("%Y-%m-%d %H:%M:%S")

    def download_plugins_icons_to_cache(self) -> None:
        """Download plugins icons to cache"""

        public_plugins = json_to_dict(self.public_plugins)
        gitlab_plugins = json_to_dict(self.gitlab_plugins)
        valid_extensions = [".png", ".svg", ".jpeg", ".jpg"]

        self.process_plugins(public_plugins["public_plugins"], valid_extensions)
        self.process_plugins(gitlab_plugins["gitlab_plugins"], valid_extensions)

    def process_plugins(self, plugins: list, valid_extensions: list) -> None:
        """Process a list of plugins and download their icons if needed"""

        for plugin in plugins:
            logo_url = plugin.get("logo")
            if logo_url:
                extension = next(
                    (ext for ext in valid_extensions if ext in logo_url),
                    None,
                )
                if extension:
                    icon_file = plugin.get("name").lower().replace(" ", "_") + extension
                    icon_path = Path(
                        self.cache_manager.get_cache_path / "icons" / icon_file
                    )
                    if not icon_path.exists():
                        self.network_manager.download_file_to(logo_url, icon_path)
