"""Widget for note display
"""

# standard
from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import QWidget

# project
from oslandia.gitlab_api.discussions import Note


class NoteWidget(QWidget):
    """QWidget to display note

    :param parent: dialog parent, defaults to None
    :type parent: Optional[QWidget], optional
    """

    def __init__(self, parent: Optional[QWidget] = None):
        super().__init__(parent)
        ui_path = Path(__file__).resolve(True).parent / "wdg_note.ui"
        uic.loadUi(ui_path, self)

        self.txt_body.setStyleSheet(
            """
            QTextEdit {
                border-radius: 15px;
                border: 2px solid lightblue;
                padding: 10px;
                background-color: white;
            }
        """
        )

        self.btn_replies.clicked.connect(self._replies_visibility_changed)

        self.btn_replies.setVisible(False)
        self.frm_replies.setVisible(False)

        self.read_only = False

        self.replies_widget = []

    def _replies_visibility_changed(self) -> None:
        is_visible = not self.frm_replies.isVisible()
        self.frm_replies.setVisible(is_visible)
        if is_visible:
            self.btn_replies.setText(self.tr("Hide replies"))
        else:
            self.btn_replies.setText(self.tr("Show replies"))

    def set_read_only(self, read_only: bool = True) -> None:
        self.read_only = read_only
        self.txt_body.setReadOnly(read_only)
        for widget in self.replies_widget:
            widget.setReadOnly(read_only)

    def set_note(self, note: Note) -> None:
        """Set note displayed

        :param note: note
        :type note: Note
        """
        self.txt_body.setText(note.body)
        self.lbl_name.setText(note.author.name)
        self.lbl_user_name.setText(f"@{note.author.username}")
        self.lbl_date.setText(note.created_at)

    def add_reply(self, reply: Note) -> None:
        """Add a reply in note

        :param reply: reply
        :type reply: Note
        """
        self.btn_replies.setVisible(True)
        self.frm_replies.setVisible(True)
        reply_widget = NoteWidget(self)
        reply_widget.set_note(reply)
        reply_widget.set_read_only(self.read_only)
        self.lyt_replies.addWidget(reply_widget)
        self.replies_widget.append(reply_widget)
