"""Widget for issue display
"""

from datetime import datetime

# standard
from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.PyQt import QtCore, uic
from qgis.PyQt.QtCore import QSortFilterProxyModel, Qt
from qgis.PyQt.QtWidgets import QWidget

# project
from oslandia.gitlab_api.discussions import DiscussionsRequestManager
from oslandia.gitlab_api.issue import Issues
from oslandia.gitlab_api.labels import LabelsRequestManager
from oslandia.gui.mdl_members import MembersListModel
from oslandia.gui.wdg_note import NoteWidget


class IssueWidget(QWidget):
    """QWidget to display issue

    :param parent: dialog parent, defaults to None
    :type parent: Optional[QWidget], optional
    """

    # Signal to indicate changes in issue displayed
    issueChanged = QtCore.pyqtSignal()

    def __init__(self, parent: Optional[QWidget] = None):
        super().__init__(parent)
        ui_path = Path(__file__).resolve(True).parent / "wdg_issue.ui"
        uic.loadUi(ui_path, self)

        self.issue = None

        self.mdl_members = MembersListModel(self)
        self.mdl_members.sort(MembersListModel.NAME_COL)

        self.proxy_mdl_members = QSortFilterProxyModel(self)
        self.proxy_mdl_members.setSourceModel(self.mdl_members)
        self.proxy_mdl_members.setSortCaseSensitivity(Qt.CaseInsensitive)
        self.proxy_mdl_members.sort(MembersListModel.NAME_COL, Qt.AscendingOrder)
        self.cbx_assignee.setModel(self.proxy_mdl_members)

        # QgsCheckableComboBox doesn't support model use

        self.lne_title.textChanged.connect(self.issueChanged.emit)
        self.txe_description.textChanged.connect(self.issueChanged.emit)
        self.cbx_labels.currentIndexChanged.connect(self.issueChanged.emit)
        self.cbx_assignee.currentIndexChanged.connect(self.issueChanged.emit)

        self.txe_description.setPlaceholderText(self.tr("Description *"))

        self.wdg_note_creation.noteAdded.connect(self._refresh_notes)

        self.read_only = False

    def set_current_project_id(self, project_id: str) -> None:
        """Define current project to define labels and members available

        :param project_id: _description_
        :type project_id: str
        """
        label_manager = LabelsRequestManager()
        labels = label_manager.get_labels_list(project_id=project_id)
        self.cbx_labels.clear()
        for label in labels:
            self.cbx_labels.addItemWithCheckState(label.name, Qt.CheckState.Unchecked)

        self.mdl_members.set_project(project_id)
        self.proxy_mdl_members.sort(MembersListModel.NAME_COL, Qt.AscendingOrder)

    def set_read_only(self, read_only: bool = True) -> None:
        """Disable / Enable edition of issue

        :param read_only: True to disable edition, Falst to enable edition, defaults to True
        :type read_only: bool, optional
        """
        self.lne_title.setReadOnly(read_only)
        self.txe_description.setReadOnly(read_only)
        self.cbx_labels.setAttribute(Qt.WA_TransparentForMouseEvents, read_only)
        self.cbx_labels.setFocusPolicy(Qt.NoFocus if read_only else Qt.StrongFocus)
        self.cbx_assignee.setAttribute(Qt.WA_TransparentForMouseEvents, read_only)
        self.cbx_assignee.setFocusPolicy(Qt.NoFocus if read_only else Qt.StrongFocus)
        self.grp_comment.setVisible(read_only)
        self.read_only = read_only

    @staticmethod
    def clear_layout(layout):
        if layout is not None:
            while layout.count():
                item = layout.takeAt(0)
                widget = item.widget()
                if widget is not None:
                    widget.deleteLater()
                else:
                    layout.removeItem(item)

    def set_issue(self, issue: Issues) -> None:
        """Set issue displayed

        :param issue: issue
        :type issue: Issues
        """
        self.issue = issue
        self.lne_title.setText(issue.name)
        self.txe_description.setReadOnly(self.read_only)
        if self.read_only:
            self.grp_comment.setVisible(True)
        else:
            self.grp_comment.setVisible(False)
        self.txe_description.setText(issue.description)

        self.cbx_labels.deselectAllOptions()
        self.cbx_labels.setCheckedItems(issue.labels)
        if issue.assignee_id:
            for row in range(self.mdl_members.rowCount()):
                index = self.proxy_mdl_members.index(row, self.mdl_members.NAME_COL)
                member = self.proxy_mdl_members.data(
                    index,
                    Qt.UserRole,
                )
                if member and member.id == issue.assignee_id:
                    self.cbx_assignee.setCurrentIndex(row)
                    break
        # A discussions as note only if in read mode
        if self.read_only:
            self._refresh_notes()

        self.wdg_note_creation.set_issue(issue)

    def _refresh_notes(self) -> None:
        """Refresh notes for current displayed issue"""
        self.clear_layout(self.lyt_notes)
        if self.issue:
            discussion_manager = DiscussionsRequestManager()
            discussions = discussion_manager.get_discussions_list(
                self.issue.project_id, self.issue.iid
            )
            for discussion in discussions:
                note_widget = None
                for note in discussion.notes:
                    if not note.system:
                        # First note is a comment
                        if not note_widget:
                            note_widget = NoteWidget(self)
                            note_widget.set_read_only(self.read_only)
                            note_widget.set_note(note)
                            self.lyt_notes.addWidget(note_widget)
                        # Other notes are reply
                        else:
                            note_widget.add_reply(note)

    def _selected_assignee_id(self) -> Optional[int]:
        """Get selected assignee id

        :return: selected assignee id
        :rtype: Optional[int]
        """
        assignee_current_index = self.cbx_assignee.currentIndex()
        if assignee_current_index != -1:
            member = self.mdl_members.data(
                self.proxy_mdl_members.mapToSource(
                    self.proxy_mdl_members.index(
                        assignee_current_index, MembersListModel.NAME_COL
                    )
                ),
                Qt.UserRole,
            )
            if member:
                return member.id
        return None

    def get_issue(self) -> Issues:
        """Get issue displayed

        :return: issue
        :rtype: Issues
        """

        return Issues(
            id=self.issue.id if self.issue else "",
            project_id=self.issue.project_id if self.issue else "",
            name=self.lne_title.text(),
            description=self.txe_description.toPlainText(),
            state=self.issue.state if self.issue else "",
            created_date=datetime.now(),  # TODO define date,
            assignee_id=self._selected_assignee_id(),
            labels=self.cbx_labels.checkedItems(),
            web_url=self.issue.web_url if self.issue else "",
            iid=self.issue.iid if self.issue else "",
        )
