#! python3  # noqa: E265

"""
    Plugin settings.
"""

# standard
from dataclasses import asdict, dataclass, fields
from pathlib import Path

# PyQGIS
from qgis.core import QgsSettings

# package
import oslandia.toolbelt.log_handler as log_hdlr
from oslandia.__about__ import __title__, __version__
from oslandia.toolbelt.application_folder import get_app_dir


# ############################################################################
# ########## Classes ###############
# ##################################
@dataclass
class Tenant:
    # name_tenant: str : il sera la clé du settings
    config: str


@dataclass
class PlgSettingsStructure:
    """Plugin settings structure and defaults values."""

    # global
    debug_mode: bool = False
    version: str = __version__
    local_app_folder: Path = get_app_dir(dir_name="cache")

    # RSS feed
    rss_source: str = "https://oslandia.com/feed/"
    rss_poll_frequency_hours: int = 24
    latest_content_guid: str = ""
    notify_push_info: bool = True
    notify_push_duration: int = 30

    # network and authentication
    authentification_config_id: str = ""
    gitlab_url_instance: str = "https://git.oslandia.net/"
    gitlab_api_version: str = "4"
    gitlab_project_id: str = "782"
    gitlab_api_page_size: int = 50
    gitlab_id_assistance: str = ""
    qgis_auth_id: str = None

    # group name
    gitlab_group_name: str = "Assistance"

    # issue creation option
    allow_qgis_info_add_in_new_issue: bool = False
    allow_plugin_info_add_in_new_issue: bool = False

    @property
    def gitlab_api_url_base(self) -> str:
        """Get GitLab instance API base URL.

        :return: https://{instance}/api/v{api_version}
        :rtype: str
        """
        return f"{self.gitlab_url_instance}api/v{self.gitlab_api_version}/"

    @property
    def gitlab_api_url_pat(self) -> str:
        """Get GitLab instance API URL to list Personal Access Tokens (PAT).

        :return: https://{instance}/api/v{api_version}/personal_access_tokens/
        :rtype: str
        """
        return f"{self.gitlab_api_url_base}personal_access_tokens/"


class PlgOptionsManager:
    @staticmethod
    def get_plg_settings() -> PlgSettingsStructure:
        """Load and return plugin settings as a dictionary. \
        Useful to get user preferences across plugin logic.

        :return: plugin settings
        :rtype: PlgSettingsStructure
        """
        # get dataclass fields definition
        settings_fields = fields(PlgSettingsStructure)
        local_app_folder: Path = get_app_dir(dir_name="cache")

        # retrieve settings from QGIS/Qt
        settings = QgsSettings()
        settings.beginGroup(__title__)

        # map settings values to preferences object
        li_settings_values = []
        for i in settings_fields:
            li_settings_values.append(
                settings.value(key=i.name, defaultValue=i.default, type=i.type)
            )

        # instanciate new settings object
        options = PlgSettingsStructure(*li_settings_values)

        settings.endGroup()

        return options

    @staticmethod
    def get_value_from_key(key: str, default=None, exp_type=None):
        """Load and return plugin settings as a dictionary. \
        Useful to get user preferences across plugin logic.

        :return: plugin settings value matching key
        """
        if not hasattr(PlgSettingsStructure, key):
            log_hdlr.PlgLogger.log(
                message="Bad settings key. Must be one of: {}".format(
                    ",".join(PlgSettingsStructure._fields)
                ),
                log_level=1,
            )
            return None

        settings = QgsSettings()
        settings.beginGroup(__title__)

        try:
            out_value = settings.value(key=key, defaultValue=default, type=exp_type)
        except Exception as err:
            log_hdlr.PlgLogger.log(
                message="Error occurred trying to get settings: {}.Trace: {}".format(
                    key, err
                )
            )
            out_value = None

        settings.endGroup()

        return out_value

    @classmethod
    def set_value_from_key(cls, key: str, value) -> bool:
        """Set plugin QSettings value using the key.

        :param key: QSettings key
        :type key: str
        :param value: value to set
        :type value: depending on the settings
        :return: operation status
        :rtype: bool
        """
        if not hasattr(PlgSettingsStructure, key):
            log_hdlr.PlgLogger.log(
                message="Bad settings key. Must be one of: {}".format(
                    ",".join(PlgSettingsStructure._fields)
                ),
                log_level=2,
            )
            return False

        settings = QgsSettings()
        settings.beginGroup(__title__)

        try:
            settings.setValue(key, value)
            out_value = True
        except Exception as err:
            log_hdlr.PlgLogger.log(
                message="Error occurred trying to set settings: {}.Trace: {}".format(
                    key, err
                )
            )
            out_value = False

        settings.endGroup()

        return out_value

    @classmethod
    def save_from_object(cls, plugin_settings_obj: PlgSettingsStructure):
        """Load and return plugin settings as a dictionary. \
        Useful to get user preferences across plugin logic.

        :return: plugin settings value matching key
        """
        settings = QgsSettings()
        settings.beginGroup(__title__)

        for k, v in asdict(plugin_settings_obj).items():
            cls.set_value_from_key(k, v)
        settings.endGroup()
