#! python3  # noqa: E265

"""
    Main plugin module.
"""


# standard
from functools import partial
from pathlib import Path

# PyQGIS
from qgis.core import QgsApplication, QgsSettings
from qgis.gui import QgisInterface
from qgis.PyQt.QtCore import QCoreApplication, QLocale, QTranslator, QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QAction, QToolBar

# project
from oslandia.__about__ import (
    DIR_PLUGIN_ROOT,
    __icon_path__,
    __title__,
    __uri_homepage__,
)
from oslandia.gui.dlg_create_issue import CreateIssueDialog
from oslandia.gui.dlg_plugins_browser import PluginsBrowser
from oslandia.gui.dlg_settings import PlgOptionsFactory
from oslandia.gui.dlg_view_issues import ViewIssueDialog
from oslandia.newsfeed.rss_reader import RssReader
from oslandia.toolbelt import PlgLogger
from oslandia.toolbelt.plugin_manager import PluginManager

# ############################################################################
# ########## Classes ###############
# ##################################


class OslandiaPlugin:
    def __init__(self, iface: QgisInterface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class which \
        provides the hook by which you can manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.wizard = None
        self.log = PlgLogger().log
        self.action_settings = None
        self.action_help = None
        self.rss_reader = RssReader()
        self.action_create_issue = None
        self.dlg_create_issue = None
        self.action_view_issue = None
        self.dlg_view_issue = None
        self.display_oslandia_plugins = None
        self.plugin_manager = PluginManager()
        self.plugin_manager.update_json_plugins()
        self.plugin_manager.download_plugins_icons_to_cache()

        # translation
        # initialize the locale
        self.locale: str = QgsSettings().value("locale/userLocale", QLocale().name())[
            0:2
        ]
        locale_path: Path = (
            DIR_PLUGIN_ROOT
            / "resources"
            / "i18n"
            / f"{__title__.lower()}_{self.locale}.qm"
        )
        self.log(message=f"Translation: {self.locale}, {locale_path}", log_level=4)
        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path.resolve()))
            QCoreApplication.installTranslator(self.translator)

        # plugin settings

        self.toolbar = None

        self._dlg_display_plugins = None

    def initGui(self):
        """Set up plugin UI elements."""
        self.iface.initializationCompleted.connect(self.post_ui_init)

        # settings page within the QGIS preferences menu
        self.options_factory = PlgOptionsFactory()
        self.iface.registerOptionsWidgetFactory(self.options_factory)

        # -- Actions
        self.action_help = QAction(
            QgsApplication.getThemeIcon("mActionHelpContents.svg"),
            self.tr("Help"),
            self.iface.mainWindow(),
        )
        self.action_help.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.action_settings = QAction(
            QgsApplication.getThemeIcon("console/iconSettingsConsole.svg"),
            self.tr("Settings"),
            self.iface.mainWindow(),
        )
        self.action_settings.triggered.connect(
            lambda: self.iface.showOptionsDialog(
                currentPage="mOptionsPage{}".format(__title__)
            )
        )

        self.action_create_issue = QAction(
            QIcon(str(DIR_PLUGIN_ROOT / "resources/images/icons/User.svg")),
            self.tr("Create issue"),
            self.iface.mainWindow(),
        )

        self.action_create_issue.triggered.connect(self._create_issue)

        self.action_view_issue = QAction(
            QIcon(str(DIR_PLUGIN_ROOT / "resources/images/icons/User.svg")),
            self.tr("View issue"),
            self.iface.mainWindow(),
        )

        self.action_view_issue.triggered.connect(self._view_issue)

        # -- Menu
        self.iface.addPluginToMenu(__title__, self.action_settings)
        self.iface.addPluginToMenu(__title__, self.action_help)
        self.iface.addPluginToWebMenu(__title__, self.action_create_issue)
        self.iface.addPluginToWebMenu(__title__, self.action_view_issue)

        # -- Toolbar
        self.toolbar = QToolBar("OslandiaToolbar")
        self.iface.addToolBar(self.toolbar)
        self.toolbar.addAction(self.action_create_issue)
        self.toolbar.addAction(self.action_view_issue)

        # documentation
        self.iface.pluginHelpMenu().addSeparator()
        self.action_help_plugin_menu_documentation = QAction(
            QIcon(str(__icon_path__)),
            f"{__title__} - Documentation",
            self.iface.mainWindow(),
        )
        self.action_help_plugin_menu_documentation.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.iface.pluginHelpMenu().addAction(
            self.action_help_plugin_menu_documentation
        )

        # Display oslandia plugins
        self.display_oslandia_plugins = QAction(
            QIcon(str(DIR_PLUGIN_ROOT / "resources/images/plugin.png")),
            self.tr("Plugins by Oslandia"),
            self.iface.mainWindow(),
        )
        self.iface.addToolBarIcon(self.display_oslandia_plugins)
        self.display_oslandia_plugins.triggered.connect(
            self.display_oslandia_plugins_dialog
        )

        # -- Menu
        self.iface.addPluginToMenu(__title__, self.display_oslandia_plugins)

    def _create_issue(self):
        """Display create issue dialog"""
        if not self.dlg_create_issue:
            self.dlg_create_issue = CreateIssueDialog()
            self.dlg_create_issue.setModal(True)
        self.dlg_create_issue.show()

    def _view_issue(self):
        """Display view issue dialog"""
        if not self.dlg_view_issue:
            self.dlg_view_issue = ViewIssueDialog()
            self.dlg_view_issue.setModal(True)
        self.dlg_view_issue.show()

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)

    def unload(self):
        """Cleans up when plugin is disabled/uninstalled."""
        # -- Clean up menu
        self.iface.removePluginMenu(__title__, self.action_help)
        self.iface.removePluginMenu(__title__, self.action_settings)
        self.iface.removePluginWebMenu(__title__, self.action_create_issue)
        self.iface.removePluginWebMenu(__title__, self.action_view_issue)
        self.iface.removeToolBarIcon(self.display_oslandia_plugins)
        self.iface.removePluginMenu(__title__, self.display_oslandia_plugins)
        # -- Clean up preferences panel in QGIS settings
        self.iface.unregisterOptionsWidgetFactory(self.options_factory)

        # remove from QGIS help/extensions menu
        if self.action_help_plugin_menu_documentation:
            self.iface.pluginHelpMenu().removeAction(
                self.action_help_plugin_menu_documentation
            )

        # remove actions
        del self.action_settings
        del self.action_help

        # remove toolbar :
        self.toolbar.deleteLater()

    def post_ui_init(self):
        """Things to do when QGIS ui is completed, without blocking QGIS
            initialization (First Content Pain).

        Becareful, this signal is only emitted when the interface is initialized, not
        refreshed or modified. For example, it's not emitted when the plugin is
        manually activated through the extensions manager and nor when another
        processing modify the QGIS GUI.
        """
        self.rss_reader.process()

    def run(self):
        """Main process.

        :raises Exception: if there is no item in the feed
        """
        try:
            self.log(
                message=self.tr("Everything ran OK."),
                log_level=3,
                push=False,
            )
        except Exception as err:
            self.log(
                message=self.tr("Houston, we've got a problem: {}".format(err)),
                log_level=2,
                push=True,
            )

    def display_oslandia_plugins_dialog(self) -> None:
        """Display instance duckdb add layer dialog"""
        if self._dlg_display_plugins is None:
            self._dlg_display_plugins = PluginsBrowser()

        self._dlg_display_plugins.show()
