"""Dialog for issue view
"""

# standard
from pathlib import Path
from typing import Optional

from qgis.PyQt import uic
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import QAbstractItemView, QDialog, QWidget

# project
from oslandia.gitlab_api.group import GroupRequestManager
from oslandia.gitlab_api.project import ProjectRequestManager
from oslandia.gui.mdl_issue import IssuesListModel
from oslandia.gui.mdl_project import ProjectListModel

# PyQGIS
from oslandia.gui.proxy_model_issue import IssueProxyModel
from oslandia.gui.proxy_model_project import ProjectProxyModel
from oslandia.toolbelt.network_manager import NetworkRequestsManager
from oslandia.toolbelt.preferences import PlgOptionsManager


class ViewIssueDialog(QDialog):
    """QDialog to display issue from a selected project

    :param parent: dialog parent, defaults to None
    :type parent: Optional[QWidget], optional
    """

    def __init__(self, parent: Optional[QWidget] = None):
        super().__init__(parent)
        ui_path = Path(__file__).resolve(True).parent / "dlg_view_issues.ui"
        uic.loadUi(ui_path, self)

        self.setWindowTitle(self.tr("Project issues"))

        self.plg_settings = PlgOptionsManager.get_plg_settings()
        self.ntwk_requester_blk = NetworkRequestsManager()
        self.project_request = ProjectRequestManager()
        self.mdl_project = ProjectListModel(self)

        assitance_group = GroupRequestManager().get_group_from_name(
            self.plg_settings.gitlab_group_name
        )
        if assitance_group:
            self.mdl_project.set_group(assitance_group.id)

        self.proxy_mdl = ProjectProxyModel(self)
        self.proxy_mdl.setSourceModel(self.mdl_project)
        self.proxy_mdl.setSortCaseSensitivity(Qt.CaseInsensitive)
        self.proxy_mdl.sort(ProjectListModel.NAME_COL, Qt.AscendingOrder)
        self.cbx_projects.setModel(self.proxy_mdl)
        self.cbx_projects.setModelColumn(ProjectListModel.NAME_COL)
        self.proxy_mdl.show_archived(self.ckb_display_archived.isChecked())

        self.mdl_issue = IssuesListModel(self)
        self.proxy_mdl_issue = IssueProxyModel(self)
        self.proxy_mdl_issue.setSourceModel(self.mdl_issue)
        self.proxy_mdl_issue.show_closed(self.ckb_display_closed.isChecked())

        self.tbv_issues.setModel(self.proxy_mdl_issue)
        self.tbv_issues.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tbv_issues.setSelectionBehavior(QAbstractItemView.SelectRows)

        self.ckb_display_archived.clicked.connect(self._project_filter_changed)
        self.ckb_display_closed.clicked.connect(self._issue_filter_changed)
        self.cbx_projects.currentIndexChanged.connect(self._project_changed)

        self.tbv_issues.selectionModel().selectionChanged.connect(self._issue_changed)

        self.wdg_issue.set_read_only(True)
        self._project_changed()

    def _project_filter_changed(self) -> None:
        """Update project filter model"""
        self.proxy_mdl.show_archived(self.ckb_display_archived.isChecked())

    def _issue_filter_changed(self) -> None:
        """Update project filter model"""
        self.proxy_mdl_issue.show_closed(self.ckb_display_closed.isChecked())

    def _project_changed(self) -> None:
        """Update issue widget for current project"""
        project_id = self._selected_project_id()
        self.mdl_issue.set_project(project_id)
        self.wdg_issue.set_current_project_id(project_id)

    def _selected_project_id(self) -> str:
        """Get selected project id

        :return: selected project id
        :rtype: str
        """
        project_index = self.cbx_projects.currentIndex()
        return self.mdl_project.data(
            self.proxy_mdl.mapToSource(
                self.proxy_mdl.index(project_index, ProjectListModel.ID)
            )
        )

    def _issue_changed(self) -> None:
        """Update visible issue"""
        selected_index = self.tbv_issues.selectionModel().selectedRows()
        if selected_index:
            issue = self.mdl_issue.data(
                self.mdl_issue.index(selected_index[0].row(), IssuesListModel.NAME_COL),
                Qt.UserRole,
            )
            self.wdg_issue.set_issue(issue)
