# standard
from pathlib import Path
from typing import Union

from packaging import version

# qgis
from qgis.core import QgsApplication
from qgis.PyQt import uic
from qgis.PyQt.QtCore import QSize, QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QDialog, QTableWidgetItem

# plugin
from oslandia.__about__ import DIR_PLUGIN_ROOT
from oslandia.toolbelt.cache_manager import CacheManager
from oslandia.toolbelt.json_tools import get_key_by_name_in_dict, json_to_dict


class PluginsBrowser(QDialog):
    def __init__(self, parent=None):
        # init module and ui
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        # icon
        self.setWindowIcon(
            QIcon(str(DIR_PLUGIN_ROOT.joinpath("resources/images/plugin.png")))
        )

        self.table.itemSelectionChanged.connect(self.display_informations)
        self.table.setIconSize(QSize(25, 25))
        self.cache_manager = CacheManager(".qgis", "plugin_oslandia")
        self._public_dict = {}
        self._gitlab_dict = {}
        self._private_dict = {}
        self.get_json_data()
        self.populate_table()

    def get_json_data(self) -> None:
        """Retrieve jsons from the cache and store them in a dict"""
        if not self._public_dict and not self._gitlab_dict and not self._private_dict:
            cache_path = self.cache_manager.get_cache_path
            self._public_dict = json_to_dict(cache_path / "public_plugins.json")
            self._gitlab_dict = json_to_dict(cache_path / "gitlab_plugins.json")
            self._private_dict = json_to_dict(cache_path / "private_plugins.json")

        self.dict_duplicate_cleaning()

    @property
    def get_qgis_language(self) -> str:
        """Retrieve the current language of QGIS.

        This function returns the locale name of the QGIS application, which indicates the current language
        setting of the software.

        :return: The language of QGIS in 'language_COUNTRY' format (e.g., 'fr').
        :rtype: str
        """
        # Get the current language of QGIS
        language = QgsApplication.locale()
        return language

    def dict_duplicate_cleaning(self) -> None:
        """In the plugins, some are duplicates when there was an experimental version
        before a non-experimental one. We need to clean them up.
        """
        cleaned_public_plugins = {}

        for plugin in self._public_dict["public_plugins"]:
            name = plugin["name"]
            current_version = plugin["version"]

            if name in cleaned_public_plugins:
                if version.parse(current_version) > version.parse(
                    cleaned_public_plugins[name]["version"]
                ):
                    cleaned_public_plugins[name] = plugin
            else:
                cleaned_public_plugins[name] = plugin

        self._public_dict["public_plugins"] = list(cleaned_public_plugins.values())

    def populate_table(self) -> None:
        """Populate the table with public, gitlab, and private plugins."""
        self.add_plugins_to_table(self._public_dict["public_plugins"], "public")
        self.add_plugins_to_table(self._gitlab_dict["gitlab_plugins"], "gitlab")
        self.add_plugins_to_table(self._private_dict["private_plugins"], "private")

        self.table.setSortingEnabled(True)

    def add_plugins_to_table(self, plugins: list, source: str) -> None:
        """Add a list of plugins to the table.

        :param plugins: List of plugins to add
        :type plugins: List
        :param source: Source of the plugins, used for the 'WhatsThis' field
        :type source: str
        """
        for plugin in plugins:
            row_position = self.table.rowCount()
            self.table.insertRow(row_position)
            item = QTableWidgetItem(plugin.get("name"))

            # If the plugin has a logo, retrieve and set the icon
            logo = plugin.get("logo")
            if logo:
                icon = self.get_plugin_icon(plugin.get("name"), logo)
                if icon:
                    item.setIcon(icon)

            item.setWhatsThis(source)
            self.table.setItem(row_position, 0, item)

    def get_plugin_icon(self, plugin_name: str, logo: str) -> Union[QIcon, None]:
        """Generate an icon for the plugin based on its logo path.

        :param plugin_name: Name of the plugin
        :type plugin_name: str
        :param logo: Path or name of the logo file
        :type logo: str
        :return: QIcon if logo extension is valid, otherwise None
        :rtype: Optional[QIcon]
        """
        valid_extensions = (".png", ".svg", ".jpg", ".jpeg")
        extension = next((ext for ext in valid_extensions if ext in logo), None)

        if extension is None:
            return None

        icon_file = plugin_name.lower().replace(" ", "_") + extension
        icon_path = Path(self.cache_manager.get_cache_path / "icons" / icon_file)

        return QIcon(str(icon_path))

    def get_name_of_selected_item(self) -> str:
        """Return the tenant name of selected item in table

        :return: Tenant name
        :rtype: str
        """
        selected_items = self.table.selectedItems()
        if selected_items:
            row = selected_items[0].row()
            name_item = self.table.item(row, 0)
            if name_item:
                name = name_item.text()
                return name
        return ""

    def get_what_this_of_selected_item(self) -> str:
        """Return the tenant name of selected item in table

        :return: Tenant name
        :rtype: str
        """
        selected_items = self.table.selectedItems()
        if selected_items:
            row = selected_items[0].row()
            item = self.table.item(row, 0)
            if item:
                whatsThis = item.whatsThis()
                return whatsThis
        return ""

    def get_dict_based_on_selection(self) -> dict:
        """
        This method returns the appropriate dictionary based on the selected item.

        :return: The dictionary corresponding to the selected item.
        :rtype: dict
        """
        selection = self.get_what_this_of_selected_item()
        if selection == "public":
            return self._public_dict
        elif selection == "gitlab":
            return self._gitlab_dict
        elif selection == "private":
            return self._private_dict
        else:
            return {}

    @property
    def get_selected_item_text(self) -> str:
        """
        This method returns the text of the selected item in the table.

        :return: The text of the selected item, or None if no item is selected. If an item is selected but has no text, an empty string is returned.
        :rtype: str
        """
        selected_items = self.table.selectedItems()

        if selected_items:
            row = selected_items[0].row()
            item = self.table.item(row, 0)
            if item:
                return item.text()
            else:
                return ""
        else:
            return ""

    def setup_button_with_url(self, button, url: str):
        """
        Configures a QPushButton to open a URL in the default web browser.
        The button is enabled or disabled based on the validity of the URL.

        :param button: Instance of the QPushButton to configure
        :param url: URL to be opened when the button is clicked
        """
        try:
            button.clicked.disconnect()
        except:
            pass
        if url:
            button.setEnabled(True)
        else:
            button.setEnabled(False)
            return

        def open_url():
            QDesktopServices.openUrl(QUrl(url))

        button.clicked.connect(open_url)

    def display_informations(self) -> None:
        """
        Displays the detailed information of the selected item and configures the UI elements accordingly.

        Updates text fields and buttons based on the selected item's details,
        including name, version, description, and various URLs.

        :return: None
        """
        # Set the name with bold font and larger size
        self.set_text_and_font(
            self.name, self.get_selected_item_text, bold=True, font_size=14
        )

        # Set various text fields (Version, About, Tags, Funders)
        self.set_text_by_key(self.lbl_version, "version")
        self.set_text_by_key(self.le_about, "about")
        self.set_text_by_key(self.lbl_tags, "tags")
        self.set_text_by_key(self.lbl_funders, "funders")

        # Handle localized description
        description = self.get_key_by_name_in_dict("description")
        localized_description = self.get_localized_text(description)
        self.le_description.setText(localized_description)

        # Set buttons with URLs for Documentation, Repository, Blog, and Issue
        self.set_button_with_url(self.btn_doc, "doc_url")
        self.set_button_with_url(self.btn_repo, "code_url")
        self.set_button_with_url(self.btn_blog, "blog_link")
        self.set_button_with_url(self.btn_issue, "issue_to_be_funded")

    def set_text_and_font(
        self, widget, text: str, bold: bool = False, font_size: int = None
    ) -> None:
        """
        Sets the text and font properties of a widget, including optional boldness and font size adjustments.

        :param widget: The widget to set the text and font on (e.g., QLabel, QPushButton)
        :param text: The text to display in the widget
        :param bold: Whether to apply bold formatting to the font (default: False)
        :param font_size: The size of the font to apply (optional)
        :return: None
        """
        widget.setText(text)
        if bold or font_size:
            font = widget.font()
            if bold:
                font.setBold(True)
            if font_size:
                font.setPointSize(font_size)
            widget.setFont(font)

    def set_text_by_key(self, widget, key: str) -> None:
        """
        Updates the widget text based on a key retrieved from a dictionary.

        :param widget: The widget to update with the retrieved text
        :param key: The key to look up in the dictionary for the value
        :return: None
        """
        text = self.get_key_by_name_in_dict(key)
        widget.setText(text)

    def get_key_by_name_in_dict(self, key: str) -> str:
        """
        Retrieves a value from the selected item's dictionary using the specified key.

        :param key: The key to look up in the selected item's dictionary
        :return: The corresponding value for the key in the dictionary as a string
        """
        return get_key_by_name_in_dict(
            self.get_selected_item_text, self.get_dict_based_on_selection(), key=key
        )

    def get_localized_text(self, description_dict: dict) -> str:
        """
        Returns the localized text for the description based on the selected language.

        If the QGIS language is set to French and the description contains a French translation,
        it will return the French version; otherwise, it will default to the English version.

        :param description_dict: A dictionary containing 'fr' and 'en' keys for the description
        :return: The localized description text (either French or English)
        """
        if self.get_qgis_language == "fr" and description_dict.get("fr"):
            return description_dict["fr"]
        return description_dict["en"]

    def set_button_with_url(self, button, key: str) -> None:
        """
        Configures a QPushButton to open a URL when clicked, using the specified key to retrieve the URL.

        The button is enabled or disabled based on the validity of the URL.

        :param button: Instance of the QPushButton to configure
        :param key: The key used to retrieve the URL from the selected item's dictionary
        :return: None
        """
        url = self.get_key_by_name_in_dict(key)
        self.setup_button_with_url(button, url)
