import json
from dataclasses import dataclass, field
from datetime import datetime
from typing import List, Optional

# PyQGIS
from qgis.PyQt.Qt import QByteArray, QUrl

# plugin
from oslandia.toolbelt import PlgLogger, PlgOptionsManager
from oslandia.toolbelt.network_manager import NetworkRequestsManager


@dataclass
class Issue:
    name: str
    description: str


@dataclass
class Issues:
    id: str
    name: str
    date: datetime
    state: str
    description: str
    labels: List[str] = field(default_factory=lambda: [])
    assignee_id: Optional[int] = None

    def valid_for_creation(self) -> bool:
        """Check if issue is valid for creation

        :return: True if issue is valid, False otherwise
        :rtype: bool
        """
        valid = self.name.strip() != ""
        return valid


class IssuesRequestManager:

    MAX_LIMIT = 50

    def get_base_url(self, project_id: str) -> str:
        """
        Get base url for members of a group

        Args:
            project: (str) project id

        Returns: url for members info

        """
        return f"{self.plg_settings.gitlab_api_url_base}projects/{project_id}/issues"

    def __init__(self):
        """
        Helper for stored_data request

        """
        self.log = PlgLogger().log
        self.ntwk_requester_blk = NetworkRequestsManager()
        self.plg_settings = PlgOptionsManager.get_plg_settings()

    def get_issues_list(self, project: str) -> List[Issues]:
        """
        Get list of members

        Args:
            project: (str) project id

        Returns: list of available members, raise ReadStoredDataException otherwise
        """
        self.log(f"{__name__}.get_issue_list(project:{project})")

        i = 1
        result = []
        result_page = self._get_issues_list(project, i, self.MAX_LIMIT)
        while len(result_page) != 0:
            result += result_page
            i = i + 1
            result_page = self._get_issues_list(project, i, self.MAX_LIMIT)

        return result

    def _get_issues_list(
        self, project_id: str, page: int = 1, limit: int = MAX_LIMIT
    ) -> List[Issues]:
        """
        Get list of issues

        Args:
            project: (str) project id
            page: (int) page number (start at 1)
            limit: (int)

        Returns: list of available members
        """

        # send request
        req_reply = self.ntwk_requester_blk.get_url(
            QUrl(
                f"{self.plg_settings.gitlab_api_url_base}/projects/{project_id}/issues?page={page}&per_page={limit}"
            ),
            self.plg_settings.authentification_config_id,
        )
        data = json.loads(req_reply.data().decode("utf-8"))
        return [
            Issues(
                id=p_data["id"],
                name=p_data["title"],
                date=p_data["created_at"].strip(),
                state=p_data["state"],
                description=p_data["description"],
                labels=p_data["labels"],
                assignee_id=p_data["assignee"]["id"] if p_data["assignee"] else None,
            )
            for p_data in data
        ]

    def create_issue(self, project_id: str, issue: Issues) -> None:
        """
        Create an issue in a project

        Args:
            project_id: (str) project id
            issue: (Issues) issue to create
        """
        self.log(f"{__name__}.create_issue()")

        # encode data
        data = QByteArray()
        data_map = {
            "title": issue.name,
            "description": issue.description,
            "labels": issue.labels,
        }
        if issue.assignee_id:
            data_map["assignee_id"] = issue.assignee_id

        data.append(json.dumps(data_map))

        self.ntwk_requester_blk.post_url(
            url=QUrl(self.get_base_url(project_id=project_id)),
            config_id=self.plg_settings.authentification_config_id,
            data=data,
            content_type_header="application/json",
        )
