# standard
from __future__ import annotations

from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.core import Qgis
from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import QWidget

# project
from oslandia.toolbelt import PlgLogger, PlgOptionsManager
from oslandia.toolbelt.preferences import PlgSettingsStructure


class WdgSettingsRss(QWidget):
    def __init__(self, parent: Optional[QWidget] = None) -> None:
        """Widget for the RSS section in the settings dialog.

        :param parent: parent widget, defaults to None
        :type parent: QObject, optional
        """
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        self.log = PlgLogger().log
        self.plg_settings = PlgOptionsManager()

        self.initGui()

    def initGui(self) -> None:  # noqa: N802
        """Set up RSS-related UI elements."""
        self.btn_reset_rss_history.pressed.connect(self.on_reset_read_history)

    def load_settings(self) -> None:
        """Load RSS options from QgsSettings into the widget."""
        settings = self.plg_settings.get_plg_settings()

        self.lne_rss_url.setText(settings.rss_source)
        self.chb_notification_rss_new_content.setChecked(settings.notify_push_info)
        self.chb_view_only_rss_unread.setChecked(settings.rss_view_only_unread)
        self.sbx_rss_poll_frequency.setValue(settings.rss_poll_frequency_hours)
        self.sbx_notif_duration.setValue(settings.notify_push_duration)
        self.rss_category_filters.setText(settings.rss_category_filters)

    def save_settings(self) -> None:
        """Persist RSS options from the widget into QgsSettings."""
        settings = self.plg_settings.get_plg_settings()
        settings.notify_push_info = self.chb_notification_rss_new_content.isChecked()
        settings.notify_push_duration = self.sbx_notif_duration.value()
        settings.rss_poll_frequency_hours = self.sbx_rss_poll_frequency.value()
        settings.rss_category_filters = self.rss_category_filters.text()
        settings.rss_view_only_unread = self.chb_view_only_rss_unread.isChecked()

        self.plg_settings.save_from_object(settings)

    def on_reset_read_history(self) -> None:
        """Reset latest_content_guid to an empty string."""
        new_settings = PlgSettingsStructure(latest_content_guid="")
        self.plg_settings.save_from_object(new_settings)

        self.log(
            message=self.tr("Read history has been reset."),
            log_level=Qgis.MessageLevel.Success,
            duration=2,
            push=True,
            parent_location=self,
        )
