# standard
from __future__ import annotations

from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import QWidget

# project
from oslandia.toolbelt import PlgLogger, PlgOptionsManager


class WdgSettingsNotifications(QWidget):
    def __init__(self, parent: Optional[QWidget] = None) -> None:
        """Widget for the notifications section in the settings dialog.

        :param parent: parent widget, defaults to None
        :type parent: QObject, optional
        """
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        self.log = PlgLogger().log
        self.plg_settings = PlgOptionsManager()

        self.initGui()

    def initGui(self) -> None:  # noqa: N802
        """Initialize notification-related UI elements."""
        self.opt_enable_notification.setChecked(False)
        self.sbx_notification_frequency.setEnabled(False)

        def enable_notification_options(enable: bool):
            """Enable/disable notification options

            :param enable: True to enable options, False to disable
            :type enable: bool
            """
            self.lne_excluded_username_list.setEnabled(enable)
            self.sbx_notification_frequency.setEnabled(enable)

        self.opt_enable_notification.clicked.connect(enable_notification_options)
        self.lne_excluded_username_list.setToolTip(
            self.tr("List of usernames separated by ,")
        )

        # Maximum notification frequency 1h
        self.sbx_notification_frequency.setMaximum(3600)
        self.sbx_notification_frequency.setSingleStep(5)

    def load_settings(self) -> None:
        """Load notification settings into the widget."""
        settings = self.plg_settings.get_plg_settings()
        # Enable smaller notification frequency in debug mode
        self.sbx_notification_frequency.setMinimum(5 if settings.debug_mode else 30)

        self.opt_enable_notification.setChecked(settings.enable_notification)
        self.sbx_notification_frequency.setEnabled(settings.enable_notification)
        self.sbx_notification_frequency.setValue(settings.notification_frequency)
        self.lne_excluded_username_list.setText(settings.excluded_username_list)
        self.lne_excluded_username_list.setEnabled(settings.enable_notification)

    def save_settings(self) -> None:
        """Save notification settings from the widget."""
        settings = self.plg_settings.get_plg_settings()
        settings.enable_notification = self.opt_enable_notification.isChecked()
        settings.notification_frequency = self.sbx_notification_frequency.value()
        settings.excluded_username_list = self.lne_excluded_username_list.text()
        self.plg_settings.save_from_object(settings)
