# standard
from __future__ import annotations

import platform
from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.core import QgsApplication
from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import QWidget

# project
from oslandia.toolbelt import PlgLogger, PlgOptionsManager


class WdgSettingsExperimentalFeatures(QWidget):
    def __init__(self, parent: Optional[QWidget] = None) -> None:
        """Widget for experimental features settings in the settings dialog.

        :param parent: parent widget, defaults to None
        :type parent: QObject, optional
        """
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        self.log = PlgLogger().log
        self.plg_settings = PlgOptionsManager()

        self.initGui()

    def initGui(self) -> None:  # noqa: N802
        """Initialize experimental features UI elements."""
        # Markdown render
        self.opt_use_gitlab_markdown_api.setIcon(
            QgsApplication.getThemeIcon("pluginExperimental.png")
        )
        self.opt_use_gitlab_markdown_api.clicked.connect(
            self._use_gitlab_api_markdown_changed
        )

        # Experimental features
        self.opt_enable_user_selection_screenshot.setIcon(
            QgsApplication.getThemeIcon("pluginExperimental.png")
        )
        self.opt_enable_user_selection_screenshot.setToolTip(
            self.tr(
                "Supported on Windows.\nNeed to activate debug mode to activate feature on other operating system."
            )
        )

    def load_settings(self) -> None:
        """Load experimental feature settings into the widget."""
        settings = self.plg_settings.get_plg_settings()

        # Experimental features
        self.opt_enable_user_selection_screenshot.setChecked(
            settings.enable_user_selection_screenshot
        )

        # Enable user selection screenshot only for Windows or if debug activated
        self.opt_enable_user_selection_screenshot.setEnabled(
            platform.system() == "Windows" or settings.debug_mode
        )

        # Markdown render
        self.opt_use_gitlab_markdown_api.setChecked(settings.use_gitlab_markdown_api)
        self.opt_use_gitlab_markdown_format.setChecked(
            settings.use_gitlab_markdown_format
        )
        self._use_gitlab_api_markdown_changed()

    def save_settings(self) -> None:
        """Save experimental feature settings from the widget."""
        settings = self.plg_settings.get_plg_settings()

        # Markdown
        settings.use_gitlab_markdown_api = self.opt_use_gitlab_markdown_api.isChecked()
        settings.use_gitlab_markdown_format = (
            self.opt_use_gitlab_markdown_format.isChecked()
        )

        # Experimental
        settings.enable_user_selection_screenshot = (
            self.opt_enable_user_selection_screenshot.isChecked()
        )

        self.plg_settings.save_from_object(settings)

    def _use_gitlab_api_markdown_changed(self) -> None:
        """Disable option gitlab markdown format if gitlab markdown API not enabled"""
        self.opt_use_gitlab_markdown_format.setEnabled(
            self.opt_use_gitlab_markdown_api.isChecked()
        )
