"""Widget for rss item display"""

# standard
from datetime import datetime, timedelta
from functools import partial
from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.PyQt import QtCore, uic
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QIcon, QImage, QPixmap
from qgis.PyQt.QtWidgets import QLabel, QMenu, QWidget

# project
from oslandia.__about__ import DIR_PLUGIN_ROOT
from oslandia.newsfeed.mdl_rss_item import RssItem
from oslandia.newsfeed.rss_reader import RssReader
from oslandia.toolbelt.cache_manager import CacheManager
from oslandia.toolbelt.log_handler import PlgLogger
from oslandia.toolbelt.network_manager import NetworkRequestsManager
from oslandia.toolbelt.preferences import PlgOptionsManager


class QTagLabel(QLabel):
    """Custom widget for RSS tags (categories)"""

    filtersChanged = QtCore.pyqtSignal(str, str)

    def __init__(self):
        super(QTagLabel, self).__init__()

    def contextMenuEvent(self, event):
        """ContextMenu event : add or remove menu for tags"""
        menu = QMenu(self)
        menu.setStyleSheet("background-color: #eee; color: #000")
        plg_settings = PlgOptionsManager.get_plg_settings()
        existing_filters = []
        if plg_settings.rss_category_filters.strip() != "":
            existing_filters = [
                x.strip().lower() for x in plg_settings.rss_category_filters.split(",")
            ]

        action_add = menu.addAction(self.tr("Add to filters"))
        action_add.setIcon(
            QIcon(str(DIR_PLUGIN_ROOT / "resources/images/add_to_filters.svg"))
        )
        action_add.triggered.connect(
            lambda state, action="add": self.add_or_remove_filters(action)
        )
        action_add.setVisible(True)
        if self.text().lower() in existing_filters:
            action_add.setVisible(False)

        action_remove = menu.addAction(self.tr("Remove from filters"))
        action_remove.setIcon(
            QIcon(str(DIR_PLUGIN_ROOT / "resources/images/remove_from_filters.svg"))
        )
        action_remove.triggered.connect(
            lambda state, action="remove": self.add_or_remove_filters(action)
        )
        action_remove.setVisible(True)
        if self.text().lower() not in existing_filters:
            action_remove.setVisible(False)

        menu.exec(event.globalPos())

    def add_or_remove_filters(self, action: str = "add"):
        """Emit an add or remove signal tag to change filters in settings

        :param add: tells if it's add or remove
        :type add: string
        """
        self.filtersChanged.emit(action, self.text())


class RssItemWidget(QWidget):
    """QWidget for rss item

    :param parent: dialog parent, defaults to None
    :type parent: Optional[QWidget], optional
    """

    filtersChanged = QtCore.pyqtSignal(str, str)

    def __init__(self, parent: Optional[QWidget] = None):
        super().__init__(parent)
        ui_path = Path(__file__).resolve(True).parent / "wdg_rss_item.ui"
        uic.loadUi(ui_path, self)
        self.parent = parent
        self.rss_reader = RssReader()
        self.cache_manager = CacheManager(".oslandia", "cache")
        self.network_manager = NetworkRequestsManager()
        self.log = PlgLogger().log
        self.path_rss_local = Path(self.cache_manager.get_cache_path / "rss_images/")

    def set_item(self, item: RssItem, nb_cars_allowed: int = 270) -> None:
        """Set rss item displayed

        :param item: RSS item
        :type item: RssItem
        :param nb_cars_allowed: Nb cars allowed to be displayed
        :type nb_cars_allowed: int
        """
        if self.rss_reader.is_read(item):
            self.is_read.setStyleSheet("background-color: rgba(0,0,0,0)")

        self.title = item.title
        self.abstract = item.abstract
        self.categories = item.categories
        self.item_date.setText(item.get_date())

        self.update_categories(self.rss_reader.is_read(item))

        self.update_description(nb_cars_allowed)
        self.read_more_button.clicked.connect(partial(self.set_read, item=item))

        # Thumbnail
        cache_days = 30
        image_url = item.image_url[0]
        file_name = item.image_url[0].rsplit("/", 1)[-1]
        file_path = Path(self.path_rss_local / file_name)
        if not file_path.exists() or datetime.now() - datetime.fromtimestamp(
            file_path.stat().st_mtime
        ) >= timedelta(days=cache_days):
            self.network_manager.download_file_to(image_url, file_path)

        # Load when the image is available
        image = QImage()
        if not file_path.exists():
            file_path = str(DIR_PLUGIN_ROOT / "resources/images/default_rss_img.png")

        with open(file_path, "rb") as f:
            image.loadFromData(f.read())
            image = image.scaledToWidth(300, Qt.TransformationMode.SmoothTransformation)
            pixmap = QPixmap(image)
            self.item_img.setPixmap(pixmap)

            # GrayScale if the news has been read
            if self.rss_reader.is_read(item):
                self.item_img.setEnabled(False)
                # Change Readmore button caption
                self.read_more_button.setText(self.tr("Read again..."))

        # Button read more onhover
        self.read_more_button.setStyleSheet("""
            QPushButton:hover {
                color: #3a74ac;
            }
        """)

    def update_description(self, nb_cars_allowed: int):
        """Update rss item description

        :param item: nb_cars_allowed
        :type item: int
        """
        # Max caracters from title + abstract
        self.item_text.clear()
        nb_cars_left = nb_cars_allowed - len(self.title)
        abstract = self.abstract
        if len(self.abstract) >= nb_cars_left:
            abstract = self.abstract[:nb_cars_left] + "..."

        self.item_text.append(f"<b>{self.title}</b> : {abstract}")
        self.item_text.setFixedHeight(100)

    def update_categories(self, is_read: bool = False):
        """Update rss item categories

        :param is_read: the item has been read
        :type item: bool
        """
        # Display categories

        if self.item_categories.count() == 0:
            # Creation
            for cat in self.categories:
                if cat != "NewsFR":
                    style = self.get_category_style(is_read, cat)
                    categorie = QTagLabel()
                    categorie.filtersChanged.connect(
                        lambda action, tag: self.filtersChanged.emit(action, tag)
                    )
                    categorie.setText(cat)
                    categorie.setStyleSheet(style)
                    self.item_categories.addWidget(categorie)
        else:
            # Update
            for i in range(self.item_categories.count()):
                cat = self.item_categories.itemAt(i).widget()
                cat.setStyleSheet(self.get_category_style(is_read, cat.text()))

    def get_category_style(self, is_read: bool, cat: str) -> str:
        """
        Docstring pour get_category_style

        :param is_read: tells if the news has been already read
        :param cat: category to stylise
        """
        bg = "#aaaaaa" if is_read else "#3a74ac"
        color = "#ffffff"
        if self.rss_reader.is_category_in_filters(cat):
            color = "#e9c89e"
        style = "QLabel::hover{background-color : #83b4e2;}"
        style += f"QLabel{{background-color: {bg}; border-radius: 10px; color:{color}; padding: 1px 2px 1px 2px; margin-right:1px;}}"
        return style

    def set_read(self, item: RssItem) -> None:
        """Mark the item as read

        :param guid: ID of the RSS item
        """
        if item:
            self.rss_reader.on_read_item(item)
            self.read_more_button.setText(self.tr("Read again..."))
            self.is_read.setStyleSheet("background-color: rgba(0,0,0,0)")
            self.item_img.setEnabled(False)
            self.update_categories(True)
