#! python3  # noqa: E265

"""RSS dialog widget."""

# standard
from pathlib import Path

# PyQGIS
from qgis.gui import QgsDockWidget, QgsGui
from qgis.PyQt import uic
from qgis.PyQt.QtGui import QImage, QPixmap, QResizeEvent
from qgis.PyQt.QtWidgets import QScrollBar, QVBoxLayout, QWidget

# Plugin
from oslandia.__about__ import DIR_PLUGIN_ROOT
from oslandia.gui.wdg_rss_item import RssItemWidget
from oslandia.newsfeed.rss_reader import RssReader
from oslandia.toolbelt.preferences import PlgOptionsManager


class RssWidget(QgsDockWidget):
    def __init__(self, parent=None):
        """Widget to display RSS feed.

        :param parent: parent widget, defaults to None
        :type parent: QObject, optional
        """
        # init module and ui
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        self.plg_settings = PlgOptionsManager.get_plg_settings()

        # Dock widget on the right
        self.rss_reader = RssReader()
        self.rss = self.rss_reader.read_rss()
        self.is_loading = False
        self.nb_cars_allowed = 270

        banner_img = QImage()
        file_path = str(DIR_PLUGIN_ROOT / "resources/images/banner_rss.png")
        with open(file_path, "rb") as f:
            banner_img.loadFromData(f.read())
            banner = QPixmap.fromImage(banner_img)
        self.banner_rss.setPixmap(banner)

        self.init_scrollarea()

        self.rssScrollArea.verticalScrollBar().valueChanged.connect(
            lambda value: self.scrolled(self.rssScrollArea.verticalScrollBar())
        )
        self.load_rss_items()

        instance_gui = QgsGui.instance()
        instance_gui.optionsChanged.connect(self.update_with_plg_settings)

    def init_scrollarea(self):
        """Initiate the scrollbarlayout and widget"""
        self.container = QWidget()
        self.container.setStyleSheet("background-color: #ffffff")
        self.container_lay = QVBoxLayout(self.container)
        self.rssScrollArea.setWidget(self.container)

    @staticmethod
    def clear_layout(layout):
        """Clear a layout from all added widget

        :param layout: layout to clear
        :type layout: any Qt type layouy
        """
        if layout is not None:
            while layout.count():
                item = layout.takeAt(0)
                widget = item.widget()
                if widget is not None:
                    widget.deleteLater()
                else:
                    layout.removeItem(item)

    def resizeEvent(self, event: QResizeEvent) -> None:
        """Update nnumbers of cars available for display description of RSS item

        :param event: resize event
        :type event: QResizeEvent
        """
        self.nb_cars_allowed = int(event.size().width() * 0.6)

        for i in range(self.container_lay.count()):
            rss_item = self.container_lay.itemAt(i).widget()
            if rss_item:
                rss_item.update_description(self.nb_cars_allowed)

        super().resizeEvent(event)

    def load_rss_items(self):
        """Load rss items into the widget"""

        self.is_loading = True

        display_range = 6
        count_loaded_items = self.container_lay.count()
        if count_loaded_items >= len(self.rss):
            return

        stop_ind = count_loaded_items + display_range
        if stop_ind > len(self.rss):
            stop_ind = len(self.rss)

        i = count_loaded_items
        while i < stop_ind:
            item = self.rss[i]
            if self.plg_settings.rss_view_only_unread and self.rss_reader.is_read(item):
                stop_ind += 1
            else:
                rss_item = RssItemWidget(self)
                rss_item.filtersChanged.connect(self.update_filters)
                rss_item.set_item(item, self.nb_cars_allowed)
                self.container_lay.addWidget(rss_item)
            i += 1

        self.is_loading = False

    def scrolled(self, scrollbar: QScrollBar):
        """Callback for scroll event on the RSS area

        :param scrollbar: QScrollBar, scrollbar object
        """
        if not self.is_loading:
            self.load_rss_items()

    def update_with_plg_settings(self) -> None:
        """Force update of widget with new plugin settings"""
        plg_settings = PlgOptionsManager()
        self.plg_settings = plg_settings.get_plg_settings()
        # Clear the scrollarea
        self.clear_layout(self.container_lay)
        self.rss = self.rss_reader.read_rss()
        self.load_rss_items()

    def update_filters(self, action: str = "add", tag: str = ""):
        """Add or remove tag to filters in settings

        :param add: tells if if it"s and add or remove
        :type add: string
        """
        plg_settings = PlgOptionsManager.get_plg_settings()

        existing_filters = []
        if plg_settings.rss_category_filters.strip() != "":
            existing_filters = plg_settings.rss_category_filters.split(",")

        if action == "add" and tag not in existing_filters:
            # Add a filter
            existing_filters.append(tag)
        if action == "remove" and tag in existing_filters:
            # Remove a filter
            existing_filters.remove(tag)
        PlgOptionsManager.set_value_from_key(
            "rss_category_filters", ",".join(existing_filters)
        )

        # Refresh the list
        self.update_with_plg_settings()
