"""Widget for rss item display"""

# standard
from datetime import datetime, timedelta
from functools import partial
from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.PyQt import uic
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QImage, QPixmap
from qgis.PyQt.QtWidgets import QLabel, QWidget

# project
from oslandia.__about__ import DIR_PLUGIN_ROOT
from oslandia.newsfeed.mdl_rss_item import RssItem
from oslandia.newsfeed.rss_reader import RssReader
from oslandia.toolbelt.cache_manager import CacheManager
from oslandia.toolbelt.log_handler import PlgLogger
from oslandia.toolbelt.network_manager import NetworkRequestsManager


class RssItemWidget(QWidget):
    """QWidget for rss item

    :param parent: dialog parent, defaults to None
    :type parent: Optional[QWidget], optional
    """

    def __init__(self, parent: Optional[QWidget] = None):
        super().__init__(parent)
        ui_path = Path(__file__).resolve(True).parent / "wdg_rss_item.ui"
        uic.loadUi(ui_path, self)
        self.rss_reader = RssReader()
        self.cache_manager = CacheManager(".oslandia", "cache")
        self.network_manager = NetworkRequestsManager()
        self.log = PlgLogger().log
        self.path_rss_local = Path(self.cache_manager.get_cache_path / "rss_images/")

    def set_item(self, item: RssItem, nb_cars_allowed: int = 270) -> None:
        """Set rss item displayed

        :param item: RSS item
        :type item: RssItem
        :param nb_cars_allowed: Nb cars allowed to be displayed
        :type nb_cars_allowed: int
        """
        if self.rss_reader.is_read(item):
            self.is_read.setStyleSheet("background-color: rgba(0,0,0,0)")

        self.title = item.title
        self.abstract = item.abstract
        self.categories = item.categories
        self.item_date.setText(item.get_date())

        self.update_categories(self.rss_reader.is_read(item))

        self.update_description(nb_cars_allowed)
        self.read_more_button.clicked.connect(partial(self.set_read, item=item))

        # Thumbnail
        cache_days = 30
        image_url = item.image_url[0]
        file_name = item.image_url[0].rsplit("/", 1)[-1]
        file_path = Path(self.path_rss_local / file_name)
        if not file_path.exists() or datetime.now() - datetime.fromtimestamp(
            file_path.stat().st_mtime
        ) >= timedelta(days=cache_days):
            self.network_manager.download_file_to(image_url, file_path)

        # Load when the image is available
        image = QImage()
        if not file_path.exists():
            file_path = str(DIR_PLUGIN_ROOT / "resources/images/default_rss_img.png")

        with open(file_path, "rb") as f:
            image.loadFromData(f.read())
            image = image.scaledToWidth(300, Qt.TransformationMode.SmoothTransformation)
            pixmap = QPixmap(image)
            self.item_img.setPixmap(pixmap)

            # GrayScale if the news has been read
            if self.rss_reader.is_read(item):
                self.item_img.setEnabled(False)
                # Change Readmore button caption
                self.read_more_button.setText(self.tr("Read again..."))

        # Button read more onhover
        self.read_more_button.setStyleSheet("""
            QPushButton:hover {
                color: #3a74ac;
            }
        """)

    def update_description(self, nb_cars_allowed: int):
        """Update rss item description

        :param item: nb_cars_allowed
        :type item: int
        """
        # Max caracters from title + abstract
        self.item_text.clear()
        nb_cars_left = nb_cars_allowed - len(self.title)
        abstract = self.abstract
        if len(self.abstract) >= nb_cars_left:
            abstract = self.abstract[:nb_cars_left] + "..."

        self.item_text.append(f"<b>{self.title}</b> : {abstract}")
        self.item_text.setFixedHeight(100)

    def update_categories(self, is_read: bool = False):
        """Update rss item categories

        :param is_read: the item has been read
        :type item: bool
        """
        # Display categories
        bg = "#aaaaaa" if is_read else "#3a74ac"
        style = f"background: {bg}; border-radius: 10px; color:#fff; padding: 1px 2px 1px 2px; margin-right:1px"
        if self.item_categories.count() == 0:
            # Creation
            for cat in self.categories:
                if cat != "NewsFR":
                    categorie = QLabel()
                    categorie.setText(cat)
                    categorie.setStyleSheet(style)
                    self.item_categories.addWidget(categorie)
        else:
            # Update
            for i in range(self.item_categories.count()):
                cat = self.item_categories.itemAt(i).widget()
                cat.setStyleSheet(style)

    def set_read(self, item: RssItem) -> None:
        """Mark the item as read

        :param guid: ID of the RSS item
        """
        if item:
            self.rss_reader.on_read_item(item)
            self.read_more_button.setText(self.tr("Read again..."))
            self.is_read.setStyleSheet("background-color: rgba(0,0,0,0)")
            self.item_img.setEnabled(False)
            self.update_categories(True)
