#! python3  # noqa: E265

"""RSS dialog widget."""

# standard
from pathlib import Path

# PyQGIS
from qgis.gui import QgsDockWidget
from qgis.PyQt import uic
from qgis.PyQt.QtGui import QImage, QPixmap, QResizeEvent
from qgis.PyQt.QtWidgets import (
    QScrollBar,
    QVBoxLayout,
    QWidget,
)

# Plugin
from oslandia.__about__ import (
    DIR_PLUGIN_ROOT,
)
from oslandia.gui.wdg_rss_item import RssItemWidget
from oslandia.newsfeed.rss_reader import RssReader


class RssWidget(QgsDockWidget):
    def __init__(self, parent=None):
        """Widget to display RSS feed.

        :param parent: parent widget, defaults to None
        :type parent: QObject, optional
        """
        # init module and ui
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        # Dock widget on the right
        self.rss_reader = RssReader()
        self.rss = self.rss_reader.read_rss()
        self.nb_items = len(self.rss)
        self.is_loading = False
        self.nb_cars_allowed = 270

        banner_img = QImage()
        file_path = str(DIR_PLUGIN_ROOT / "resources/images/banner_rss.png")
        with open(file_path, "rb") as f:
            banner_img.loadFromData(f.read())
            banner = QPixmap.fromImage(banner_img)
        self.banner_rss.setPixmap(banner)

        container = QWidget()
        container.setStyleSheet("background-color: #ffffff")
        self.container_lay = QVBoxLayout(container)
        self.rssScrollArea.setWidget(container)

        self.rssScrollArea.verticalScrollBar().valueChanged.connect(
            lambda value: self.scrolled(self.rssScrollArea.verticalScrollBar())
        )
        self.load_rss_items()

    def resizeEvent(self, event: QResizeEvent) -> None:
        """Update nnumbers of cars available for display description of RSS item

        :param event: resize event
        :type event: QResizeEvent
        """
        self.nb_cars_allowed = int(event.size().width() * 0.6)

        for i in range(self.container_lay.count()):
            rss_item = self.container_lay.itemAt(i).widget()
            rss_item.update_description(self.nb_cars_allowed)

        super().resizeEvent(event)

    def load_rss_items(self):
        """Load rss items into the widget"""

        self.is_loading = True

        display_range = 6
        count_loaded_items = self.container_lay.count()
        if count_loaded_items >= self.nb_items:
            return

        stop = count_loaded_items + display_range
        if stop > self.nb_items:
            stop = self.nb_items

        for i in range(count_loaded_items, stop):
            item = self.rss[i]
            if i == self.container_lay.count():
                rss_item = RssItemWidget(self)
                rss_item.set_item(item, self.nb_cars_allowed)
                self.container_lay.addWidget(rss_item)
                self.container_lay.update()

        self.is_loading = False

    def scrolled(self, scrollbar: QScrollBar):
        """Callback for scroll event on the RSS area

        :param scrollbar: QScrollBar, scrollbar object
        """
        if not self.is_loading:
            self.load_rss_items()
