from abc import abstractmethod
from typing import Any, List, Optional

from qgis.core import QgsTask
from qgis.PyQt.QtCore import QObject, Qt

from oslandia.gitlab_api.gitlab_object import PageLoadResult
from oslandia.gitlab_api.members import Members, MembersRequestManager
from oslandia.gui.mdl_gitlab_object import AbstractGitlabObjectModel


class MembersListModel(AbstractGitlabObjectModel):
    NAME_COL = 0
    # AVATAR_IMAGE = 1

    def __init__(self, parent: QObject = None):
        """
        QStandardItemModel for datastore list display

        Args:
            parent: QObject parent
        """
        super().__init__(parent)
        self.allows_null_value = True

    @abstractmethod
    def _column_headers(self) -> List[str]:
        """Returns headers list"""
        return [
            self.tr("Name"),
        ]

    @abstractmethod
    def _object_data_for_role(
        self,
        gitlab_object: Any,
        col: int,
        role: Qt.ItemDataRole = Qt.ItemDataRole.DisplayRole,
    ) -> Any:
        """Return qt model data for a specific column and role

        :param gitlab_object: object to display
        :type gitlab_object: Any
        :param col: column for table display
        :type col: int
        :param role: qt role, defaults to Qt.ItemDataRole.DisplayRole
        :type role: _tyItemDataRole.pe_, optional
        :return: qt model data
        :rtype: Any
        """
        member: Members = gitlab_object

        if role == Qt.ItemDataRole.DisplayRole:
            res = {self.NAME_COL: member.name}
            return res.get(col, None)
        if role == Qt.ItemDataRole.UserRole:
            res = {self.NAME_COL: member}
            return res.get(col, None)

        return None

    def _page_load_description(self, page: int) -> str:
        """Test display in QGIS task manager for page load task

        :param page: loading page
        :type page: int
        :return: page load description
        :rtype: str
        """
        return self.tr(
            f"Load page {page}/{self.nb_page} of members for project {self.obj_filter}"
        )

    @abstractmethod
    def _load_object_page(
        self, task: Optional[QgsTask], obj_filter: Any, page: int
    ) -> PageLoadResult:
        """Load members list for a project for a specific page

        :param task: task used for load, can be None
        :type task: QgsTask, optional
        :param obj_filter: filter reprensenting project id. Must be a str
        :type obj_filter: Any
        :param page: page to load
        :type page: int
        :return: page load result
        :rtype: PageLoadResult
        """
        manager = MembersRequestManager()
        return manager.get_members_list_by_page(
            obj_filter, page=page, limit=self.NB_ITEM_BY_PAGE
        )
