from abc import abstractmethod
from time import sleep
from typing import Any, List, Optional

from qgis.core import QgsApplication, QgsTask
from qgis.PyQt import QtCore
from qgis.PyQt.QtCore import QAbstractTableModel, QModelIndex, QObject, Qt, QVariant

from oslandia.gitlab_api.gitlab_object import PageLoadResult
from oslandia.toolbelt import PlgLogger


class AbstractGitlabObjectModel(QAbstractTableModel):
    NB_ITEM_BY_PAGE = 5

    pageLoadedSignal = QtCore.pyqtSignal(int)

    def __init__(self, parent: QObject = None):
        """
        QAbstractTableModel for GitLab API object display. This is an abstract class.

        Data for GitLab API object are loaded for each available page in the API.
        Data load is done in a QgsTask.

        Method to implement:
        - _column_headers : return the column header list for the displayed data
        - _object_data_for_role : return the GitLab object data for a column and a Qt role
        - _load_data_for_page : return the result of a page load, will be used to get GitLab object for a row

        Optional implementation:
        - _loading_col : return column for loading text

        To define a filter that will be used to get content with _load_data_for_page you must use set_filter function.
        It will store a filter object and use it for each page load.

        Args:
            parent: QObject parent
        """
        super().__init__(parent)
        self.log = PlgLogger().log

        self.nb_row = 0
        self.nb_page = 0
        self._gitlab_object_row: List[Any] = []
        self.tasks: List[QgsTask] = []
        self.task_ids: List[int] = []
        self.enable_multiple_task = True
        self.allows_null_value = False
        self.obj_filter = None

    def wait_for_data_load(self) -> None:
        """Wait until all data is loaded.
        Check if current available data len is equal to number of expected row
        """
        # Don't check first row if null value allowed
        data_start = 1 if self.allows_null_value else 0
        while None in self._gitlab_object_row[data_start:-1]:
            QgsApplication.processEvents()
            sleep(0.1)

    @abstractmethod
    def _column_headers(self) -> List[str]:
        """Returns column header list

        :raises NotImplementedError: method must be implemented.
        :return: columns header
        :rtype: List[str]
        """
        raise NotImplementedError(
            "Must define header in implemented AbstractGitlabObjectModel"
        )

    @abstractmethod
    def _object_data_for_role(
        self, gitlab_object: Any, col: int, role=Qt.ItemDataRole.DisplayRole
    ) -> Any:
        """Return qt model data for a specific column and role

        :param gitlab_object: object to display
        :type gitlab_object: Any
        :param col: column for table display
        :type col: int
        :param role: qt role, defaults to Qt.ItemDataRole.DisplayRole
        :type role: _type_, optional
        :raises NotImplementedError: must be implemented
        :return: qt model data
        :rtype: Any
        """
        raise NotImplementedError(
            "Must define header in implemented AbstractGitlabObjectModel"
        )

    @abstractmethod
    def _load_object_page(
        self, task: Optional[QgsTask], obj_filter: Any, page: int
    ) -> PageLoadResult:
        """Load object for a specific page

        :param task: task used for load
        :type task: Optional[QgsTask]
        :param obj_filter: _description_
        :type obj_filter: str
        :param page: page to load
        :type page: int
        :raises NotImplementedError: method must be implemented
        :return: page load result
        :rtype: PageLoadResult
        """
        raise NotImplementedError(
            "Must define header in implemented AbstractGitlabObjectModel"
        )

    @abstractmethod
    def _loading_col(self) -> int:
        """Return column used to display loading text

        :return: loading column (default 0)
        :rtype: int
        """
        return 0

    def headerData(
        self,
        section: int,
        orientation: Qt.Orientation,
        role: Qt.ItemDataRole.DisplayRole,
    ) -> QVariant:
        """Define data for header, only display role is supported

        :param section: section number (column for Horizontal orientation , row for Vertical orientation)
        :type section: int
        :param orientation: orientation
        :type orientation: Qt.Orientation
        :param role: data role
        :type role: Qt.ItemDataRole.DisplayRole
        :return: header data
        :rtype: QVariant
        """
        if (
            orientation == Qt.Orientation.Horizontal
            and role == Qt.ItemDataRole.DisplayRole
        ):
            return self._column_headers()[section]
        return None

    def rowCount(self, parent=QModelIndex()) -> int:
        """Define number of row in model

        :param parent: parent, defaults to QModelIndex() not used in case of table
        :type parent: QModelIndex, optional
        :return: number of row
        :rtype: int
        """
        return self.nb_row

    def columnCount(self, parent=QModelIndex()) -> int:
        """Define number of column in model

        :param parent: parent, defaults to QModelIndex() not used in case of table
        :type parent: QModelIndex, optional
        :return: number of column
        :rtype: int
        """
        return len(self._column_headers())

    def data(self, index: QModelIndex, role=Qt.ItemDataRole.DisplayRole) -> Any:
        """Define data for model index and role.
        Use data from cache or display loading information before cache load

        :param index: model index
        :type index: QModelIndex
        :param role: data role, defaults to Qt.ItemDataRole.DisplayRole
        :type role: Qt.ItemDataRole, optional
        :return: model data for index and role
        :rtype: Any
        """
        if not index.isValid():
            return None

        row, col = index.row(), index.column()

        # No value for null value row
        if self.allows_null_value and index.row() == 0:
            return None

        # If row loaded, get object and return qt model data from implementation
        if gitlab_object := self._gitlab_object_row[row]:
            return self._object_data_for_role(gitlab_object, col, role)

        # Display loading text
        if col == self._loading_col() and role == Qt.ItemDataRole.DisplayRole:
            return self.tr("Loading...")
        else:
            return None

    def flags(self, index: QModelIndex):
        """Define flag to disable edition

        :param index: model index
        :type index: QModelIndex
        :return: _description_
        :rtype: _type_
        """
        default_flags = super().flags(index)
        return default_flags & ~Qt.ItemFlag.ItemIsEditable  # Disable editing

    def set_filter(self, obj_filter: Any) -> None:
        """
        Define filter used to load data.
        Current data is cleared and we launch QgsTask to load all available pages.

        """
        self.obj_filter = obj_filter

        # Remove all tasks object
        self.tasks.clear()
        # Cancel running task to avoid loading invalid data
        task_manager = QgsApplication.taskManager()
        for task_id in self.task_ids:
            if task := task_manager.task(task_id):
                task.cancel()

        # Clear all data
        self.beginResetModel()
        self._gitlab_object_row.clear()

        # Load first page
        result = self._load_object_page(None, obj_filter, 1)
        self.nb_row = result.nb_object

        if self.allows_null_value:
            self.nb_row += 1

        self._gitlab_object_row = [None] * self.nb_row
        self.nb_page = result.nb_page
        self._page_loaded(None, result)

        # Load all pages in multiple QgsTask
        if self.enable_multiple_task:
            for page in range(2, self.nb_page + 1):
                self._add_page_load_task(page)

        self.endResetModel()

    def _page_loaded(self, exception, result: PageLoadResult) -> None:
        """Function call after QgsTask finished for issue load

        :param exception: exception raised during task
        :type exception: _type_
        :param result: result of issue load
        :type result: PageLoadResult
        """
        # Add page content for each row
        page: int = result.page
        self.pageLoadedSignal.emit(page)
        gitlab_object_list: list[Any] = result.gitlab_object_list
        for i, gitlab_object in enumerate(gitlab_object_list):
            row = i + (page - 1) * self.NB_ITEM_BY_PAGE
            if self.allows_null_value:
                row += 1
            self.set_row_object(row, gitlab_object)

        # If multiple task disabled, ask for next page
        # Otherwise the multiple task were already created when setting project
        next_page = page + 1
        if not self.enable_multiple_task and next_page <= self.nb_page:
            self._add_page_load_task(next_page)

    def _page_load_description(self, page: int) -> str:
        """Test display in QGIS task manager for page load task

        :param page: loading page
        :type page: int
        :return: page load description
        :rtype: str
        """
        return self.tr(
            f"Load page {page}/{self.nb_page} for obj_filter {self.obj_filter}"
        )

    def _add_page_load_task(self, page: int) -> None:
        """Add a QgsTask for page load

        :param page: page to load
        :type page: int
        """
        task = QgsTask.fromFunction(
            self._page_load_description(page),
            self._load_object_page,
            on_finished=self._page_loaded,
            obj_filter=self.obj_filter,
            flags=QgsTask.Flag.Silent,
            page=page,
        )
        self.tasks.append(task)
        self.task_ids.append(QgsApplication.taskManager().addTask(task))

    def set_row_object(self, row: int, gitlab_object: Any) -> None:
        """Define data for a row

        :param row: row
        :type row: int
        :param gitlab_object: issue
        :type gitlab_object: Any
        """
        # Store in row dict
        self._gitlab_object_row[row] = gitlab_object

        # Indicate that model has changed
        super().dataChanged.emit(
            self.index(row, 0), self.index(row, self.columnCount() - 1)
        )
