"""Dialog for issue view"""

# standard
from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.core import QgsApplication
from qgis.PyQt import uic
from qgis.PyQt.QtCore import QItemSelection, QItemSelectionModel, Qt
from qgis.PyQt.QtWidgets import QAbstractItemView, QHeaderView, QMessageBox, QWidget
from qgis.utils import OverrideCursor

# project
from oslandia.gitlab_api.issue import Issues
from oslandia.gitlab_api.project import Project, ProjectRequestManager
from oslandia.gitlab_api.user import UserRequestManager
from oslandia.gui.delegate_issue_table_view import IssueTableViewDelegate
from oslandia.gui.dlg_create_issue import CreateIssueDialog
from oslandia.gui.mdl_issue import IssuesListModel
from oslandia.gui.mdl_project import ProjectListModel
from oslandia.gui.mdl_project_manager import ProjectListModelManager
from oslandia.gui.proxy_model_issue import IssueProxyModel
from oslandia.gui.proxy_model_project import ProjectProxyModel

# PyQGIS
from oslandia.gui.task_gitlab_notification import GitLabNotificationManager
from oslandia.gui.wdg_issue import IssueWidget
from oslandia.toolbelt.network_manager import NetworkRequestsManager
from oslandia.toolbelt.preferences import PlgOptionsManager


class ViewIssueWidget(QWidget):
    """QWidget to display issue from a selected project

    :param group_id: gitlab group id for project selection
    :type group_id: strl
    :param parent: dialog parent, defaults to None
    :type parent: Optional[QWidget], optional
    """

    def __init__(self, group_id: str, parent: Optional[QWidget] = None):
        super().__init__(parent)
        with OverrideCursor(Qt.CursorShape.WaitCursor):
            ui_path = Path(__file__).resolve(True).parent / "wdg_view_issues.ui"
            uic.loadUi(ui_path, self)
            self.setWindowTitle(self.tr("Project issues"))
            self.plg_settings = PlgOptionsManager.get_plg_settings()
            self.ntwk_requester_blk = NetworkRequestsManager()
            self.project_request = ProjectRequestManager()
            self.group_id = group_id
            self.mdl_project = ProjectListModelManager.get_model_for_group(group_id)
            self.user_request_manager = UserRequestManager()
            self.proxy_mdl = ProjectProxyModel(self)
            self.proxy_mdl.setSourceModel(self.mdl_project)
            self.proxy_mdl.setSortCaseSensitivity(Qt.CaseSensitivity.CaseInsensitive)
            self.proxy_mdl.sort(ProjectListModel.NAME_COL, Qt.SortOrder.AscendingOrder)
            self.cbx_projects.setModel(self.proxy_mdl)
            self.cbx_projects.setModelColumn(ProjectListModel.NAME_COL)
            self.proxy_mdl.show_archived(self.ckb_display_archived.isChecked())

            # Set current project  with last value used if defined in settings
            if self.plg_settings.issue_view_last_project_id:
                self.set_current_project_id(
                    self.plg_settings.issue_view_last_project_id
                )

            self.mdl_issue = IssuesListModel(self)
            self.proxy_mdl_issue = IssueProxyModel(self)
            self.proxy_mdl_issue.setSourceModel(self.mdl_issue)
            self.proxy_mdl_issue.show_closed(self.ckb_display_closed.isChecked())

            self.tbv_issues.setModel(self.proxy_mdl_issue)
            self.tbv_issues.setSelectionMode(
                QAbstractItemView.SelectionMode.SingleSelection
            )
            self.tbv_issues.setSelectionBehavior(
                QAbstractItemView.SelectionBehavior.SelectRows
            )
            self.tbv_issues.setSortingEnabled(True)

            self.ckb_display_archived.clicked.connect(self._project_filter_changed)
            self.ckb_display_closed.clicked.connect(self._issue_filter_changed)
            self.ckb_assign_me.clicked.connect(self._issue_filter_changed)
            self.lne_search.textChanged.connect(self._issue_filter_changed)
            self.cbx_projects.currentIndexChanged.connect(self._project_changed)

            self.tbv_issues.selectionModel().selectionChanged.connect(
                self._selected_issue_changed
            )

            self.tbv_issues.verticalHeader().setVisible(False)
            self.tbv_issues.horizontalHeader().setSectionResizeMode(
                IssuesListModel.NAME_COL, QHeaderView.ResizeMode.Stretch
            )

            delegate = IssueTableViewDelegate(self.tbv_issues)
            self.tbv_issues.setItemDelegate(delegate)

            self.btn_refresh.setIcon(QgsApplication.getThemeIcon("mActionRefresh.svg"))
            self.btn_refresh.clicked.connect(self._refresh_issues)

            self.btn_issue_add.setIcon(QgsApplication.getThemeIcon("mActionAdd.svg"))
            self.btn_issue_add.clicked.connect(self._add_issue)

            self.wdg_issue.set_display_mode(IssueWidget.DisplayMode.READ)
            self.wdg_issue.issueUpdated.connect(self._issue_updated)

            self.lbl_search.setToolTip(
                self.tr("Filter on issue title and description.")
            )

            self.gitlab_notification_manager = None

            # Simulate project changes for project_id definition in widgets
            self._project_changed()

    def set_current_project_id(self, project_id: str) -> None:
        """Define current selected project

        :param project_id: project id
        :type project_id: str
        """
        with OverrideCursor(Qt.CursorShape.WaitCursor):
            self.mdl_project.wait_for_data_load()

            project_row = self.mdl_project.get_project_row(project_id)
            if project_row != -1:
                index = self.proxy_mdl.mapFromSource(
                    self.mdl_project.index(project_row, self.mdl_project.NAME_COL)
                )
                self.cbx_projects.setCurrentIndex(index.row())

    def select_issue(self, issue: Issues) -> None:
        """Select issue in dialog

        :param issue: issue to select
        :type issue: Issues
        """

        # Change selected project if needed
        if issue.project_id != self.selected_project_id():
            self.set_current_project_id(issue.project_id)

        with OverrideCursor(Qt.CursorShape.WaitCursor):
            # Force refresh of issue list
            if project_id := self.selected_project_id():
                self.mdl_issue.set_filter(project_id)

            # Wait for data load to be sure to get issue row
            self.mdl_issue.wait_for_data_load()

            issue_row = self.mdl_issue.get_issue_row(issue)
            if issue_row != -1:
                # Get proxy model index for selection
                index = self.proxy_mdl_issue.mapFromSource(
                    self.mdl_issue.index(issue_row, self.mdl_project.NAME_COL)
                )
                # Update selection model, issue will be display with signal
                self.tbv_issues.selectionModel().select(
                    index,
                    QItemSelectionModel.SelectionFlag.Select
                    | QItemSelectionModel.SelectionFlag.Rows,
                )

    def _project_filter_changed(self) -> None:
        """Update project filter model"""
        with OverrideCursor(Qt.CursorShape.WaitCursor):
            self.proxy_mdl.show_archived(self.ckb_display_archived.isChecked())

    def _issue_filter_changed(self) -> None:
        """Update project filter model"""
        with OverrideCursor(Qt.CursorShape.WaitCursor):
            self.proxy_mdl_issue.show_closed(self.ckb_display_closed.isChecked())
            user_id = (
                [self.user_request_manager.get_current_user().id]
                if self.ckb_assign_me.isChecked()
                else []
            )
            self.proxy_mdl_issue.set_visible_user(user_id)
            search_text = None
            if self.lne_search.text():
                search_text = self.lne_search.text()
            self.proxy_mdl_issue.set_search_txt(search_text)

    def _project_changed(self) -> None:
        """Update issue widget for current project"""
        with OverrideCursor(Qt.CursorShape.WaitCursor):
            if project_id := self.selected_project_id():
                self.mdl_issue.set_filter(project_id)

            # Get project and update notification manager
            if project := self.selected_project():
                self.gitlab_notification_manager = GitLabNotificationManager(
                    parent=self,
                    refresh_period_in_seconds=self.plg_settings.notification_frequency,
                    projects=[project],
                    message_bar=self.msgbar_notification,
                    issue_clicked_callable=self._issue_notification_clicked,
                )
                self.gitlab_notification_manager.check_issues()

    def _issue_notification_clicked(self, issue: Issues) -> None:
        """Select issue when notification is clicked

        :param issue: issue clicked
        :type issue: Issues
        """
        self.select_issue(issue)

    def _issue_updated(self, issue: Issues) -> None:
        """Update current selected row issue

        :param issue: updated issue
        :type issue: Issues
        """
        selected_index = self.tbv_issues.selectionModel().selectedRows()
        if selected_index:
            index = self.proxy_mdl_issue.mapToSource(selected_index[0])
            self.mdl_issue.set_row_object(index.row(), issue)

    def _refresh_issues(self) -> None:
        """Refresh visible issues"""
        # Simulate a project changes
        self._project_changed()

    def _add_issue(self) -> None:
        """Add issue for selected project"""
        project_id = self.selected_project_id()
        dlg_create_issue = CreateIssueDialog(self.group_id, self)
        dlg_create_issue.enable_archived_project(self.ckb_display_archived.isChecked())
        dlg_create_issue.set_current_project_id(project_id)
        dlg_create_issue.setModal(False)
        if dlg_create_issue.exec():
            self._refresh_issues()

    def selected_project_id(self) -> Optional[str]:
        """Get selected project id, return None if no project selected

        :return: selected project id
        :rtype: Optional[str]
        """
        project_index = self.cbx_projects.currentIndex()
        if project_index != -1:
            return self.mdl_project.data(
                self.proxy_mdl.mapToSource(
                    self.proxy_mdl.index(project_index, ProjectListModel.ID)
                )
            )
        return None

    def selected_project(self) -> Optional[Project]:
        """Get selected project, return None if no project selected

        :return: selected project
        :rtype: Optional[Project]
        """
        project_index = self.cbx_projects.currentIndex()
        if project_index != -1:
            return self.mdl_project.data(
                self.proxy_mdl.mapToSource(
                    self.proxy_mdl.index(project_index, ProjectListModel.NAME_COL)
                ),
                Qt.ItemDataRole.UserRole,
            )
        return None

    def _selected_issue_changed(
        self, selected: QItemSelection, deselected: QItemSelection
    ) -> None:
        """Update visible issue"""

        # Check if current issue is updated
        if (
            self.wdg_issue.get_display_mode() == IssueWidget.DisplayMode.UPDATE
            and self.wdg_issue.is_issue_changed()
        ):
            msg_box = QMessageBox(self)
            msg_box.setIcon(QMessageBox.Icon.Information)
            msg_box.setText(
                self.tr(
                    "Updated issue won't be saved. Do you want to change displayed issue ?"
                )
            )
            msg_box.setWindowTitle(self.tr("Changes unsaved"))
            msg_box.setStandardButtons(
                QMessageBox.StandardButton.Ok | QMessageBox.StandardButton.Cancel
            )

            # Restore previous selection
            if msg_box.exec() != QMessageBox.StandardButton.Ok:
                self.tbv_issues.selectionModel().selectionChanged.disconnect(
                    self._selected_issue_changed
                )
                self.tbv_issues.selectionModel().select(
                    deselected, QItemSelectionModel.SelectionFlag.ClearAndSelect
                )
                self.tbv_issues.selectionModel().selectionChanged.connect(
                    self._selected_issue_changed
                )
                return

        with OverrideCursor(Qt.CursorShape.WaitCursor):
            selected_index = self.tbv_issues.selectionModel().selectedRows()
            if selected_index:
                issue = self.mdl_issue.data(
                    self.proxy_mdl_issue.mapToSource(
                        self.proxy_mdl_issue.index(
                            selected_index[0].row(), IssuesListModel.NAME_COL
                        )
                    ),
                    Qt.ItemDataRole.UserRole,
                )
                if issue:
                    self.wdg_issue.set_issue(issue)
                    self.wdg_issue.set_display_mode(IssueWidget.DisplayMode.READ)
