"""A text editor that automatically upload file to gitlab at drop"""

# standard
import tempfile
from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.core import QgsApplication
from qgis.PyQt.QtGui import QDragEnterEvent, QDropEvent, QIcon, QPixmap
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QWidget
from qgis.utils import iface

# project
from oslandia.gitlab_api.upload import UploadRequestManager
from oslandia.gui.auto_resizing_text_edit import AutoResizingTextEdit
from oslandia.toolbelt.qt_utils import current_datetime_str


class GitLabTextEdit(AutoResizingTextEdit):
    def __init__(self, parent: Optional[QWidget] = None) -> None:
        super().__init__(parent)
        self._project_id = None
        self.setAcceptDrops(True)

        self._insert_separator_in_edit_action()
        attach_file_action = QAction(
            QIcon(QgsApplication.iconPath("mActionEditInsertLink.svg")),
            self.tr("Attach file"),
        )
        attach_file_action.setToolTip(self.tr("Attach file or image."))
        attach_file_action.triggered.connect(self._insert_file)
        self._edit_actions.append(attach_file_action)

        canvas_screenshot_action = QAction(
            QgsApplication.getThemeIcon("mActionSaveMapAsImage.svg"),
            self.tr("Canvas screenshot."),
        )
        canvas_screenshot_action.setToolTip(self.tr("Canvas screenshot."))
        canvas_screenshot_action.triggered.connect(self._canvas_screenshot)
        self._edit_actions.append(canvas_screenshot_action)

    def _insert_file(self) -> None:
        """Select file and insert to edit widget"""
        file_path, _ = QFileDialog.getOpenFileName(
            self, self.tr("Select file to upload")
        )
        if file_path:
            self.insert_file(file_path)

    def _canvas_screenshot(self) -> None:
        """Print current canvas to QPixmap and insert it"""
        canvas = iface.mapCanvas()
        if canvas:
            tmp_path = self._temp_file_name(".png")
            canvas.saveAsImage(str(tmp_path))
            self.insert_file(
                tmp_path,
                self.tr("QGIS canvas added {}").format(current_datetime_str("long")),
            )

    @staticmethod
    def _temp_file_name(suffix: str) -> Path:
        """Return temporary file name. File is closed to be used by Qt

        :param suffix: filename suffix
        :type suffix: str
        :return: temporary file name
        :rtype: Path
        """
        # Create temporary file
        tmp_file = tempfile.NamedTemporaryFile(suffix=suffix, delete=False)
        tmp_path = tmp_file.name
        # We are only using tempfile.NamedTemporaryFile to get a temp file path
        # Need to close to be able to save screenshot
        tmp_file.close()
        return Path(tmp_path)

    def _insert_pixmap(self, pixmap: QPixmap, insert_text: Optional[str]) -> None:
        """Insert pixmap as file

        :param pixmap: pixmap
        :type pixmap: QPixmap
        :param insert_text: text to insert before pixmap (optional)
        :type insert_text: Optional[str]
        """
        tmp_path = self._temp_file_name(".png")
        pixmap.save(str(tmp_path), "png")
        self.insert_file(Path(tmp_path), insert_text)

    def set_project_id(self, _project_id: str) -> None:
        """Define project_id needed for image displau

        :param _project_id: project id
        :type _project_id: str
        """
        self._project_id = _project_id

    def dragEnterEvent(self, event: QDragEnterEvent) -> None:
        """Check if we accept drag event, only urls are accepted

        :param event: drag enter event
        :type event: QDragEnterEvent
        """
        if event.mimeData().hasUrls():
            event.acceptProposedAction()
        else:
            event.ignore()

    def dropEvent(self, event: QDropEvent) -> None:
        """Accept drop event for url and create upload in GitLab for current project

        :param event: drop event
        :type event: QDropEvent
        """
        if not self._project_id:
            event.ignore()
            return

        if event.mimeData().hasUrls():
            urls = event.mimeData().urls()
            file_paths = [url.toLocalFile() for url in urls]

            for path in file_paths:
                self.insert_file(path)
            event.acceptProposedAction()
        else:
            event.ignore()

    def insert_file(self, file_path: Path, insert_text: Optional[str] = None) -> None:
        """Insert file in document.
        Upload file to project and insert markdown string

        :param file_path: file path
        :type file_path: Path
        :param insert_text: text to insert before file (optional)
        :type insert_text: Optional[str]
        """
        if not self._project_id:
            return
        manager = UploadRequestManager()
        md_str = manager.create_upload(self._project_id, file_path)
        if md_str:
            if insert_text:
                self.textCursor().insertText(insert_text)
                self.textCursor().insertText("\n\n")

            self.textCursor().insertText(md_str)
            self.textCursor().insertText("\n")
