# standard
import json
from datetime import datetime
from typing import Dict, List, Optional, Tuple

# PyQGIS
from qgis.PyQt.QtCore import QByteArray, QUrl

# plugin
from oslandia.gitlab_api.issue import Issues
from oslandia.toolbelt.log_handler import PlgLogger
from oslandia.toolbelt.network_manager import NetworkRequestsManager
from oslandia.toolbelt.preferences import PlgOptionsManager


class IssuesGraphQLRequestManager:
    """
    Fetches issues for a specific GitLab project using the GraphQL API.
    """

    def __init__(self, page_size: int = 5) -> None:
        """Constructor

        :param page_size: page size, defaults to 5
        :type page_size: int, optional
        """
        self.log = PlgLogger().log
        self.ntwk_requester_blk = NetworkRequestsManager()
        self.plg_settings = PlgOptionsManager.get_plg_settings()
        self.page_size = page_size

        """
        Retrieve all issues for a given GitLab project.

        Args:
            project_id (str): Full path of the GitLab project.
            updated_after (Optional[str]): ISO datetime string to filter issues updated after this date.

        Returns:
            List[Issue]: List of issues as dataclass instances.
        """

    def get_updated_issues_list(
        self, project_path: str, updated_after: datetime
    ) -> List[Issues]:
        """Get issues updated after a datetime

        :param project_path: project full path for GraphQL
        :type project_path: str
        :param updated_after: datetime for update
        :type updated_after: datetime
        :return: list of issue updated after datetime
        :rtype: List[Issues]
        """

        issues, after = self.get_updated_issues_list_for_cursor(
            after=None, project_path=project_path, updated_after=updated_after
        )

        while after is not None:
            page_issue_list, after = self.get_updated_issues_list_for_cursor(
                after=after, project_path=project_path, updated_after=updated_after
            )
            issues.extend(page_issue_list)
        return issues

    def get_updated_issues_list_for_cursor(
        self,
        after: Optional[str],
        project_path: str,
        updated_after: datetime,
    ) -> Tuple[List[Issues], Optional[str]]:
        """Get issues updated after a datetime for a GraphQL cursor
        :param after : GraphQL cursor, None for first page
        :type after :Optional[str]
        :param project_path: project full path for GraphQL
        :type project_path: str
        :param updated_after: datetime for update
        :type updated_after: datetime
        :return: list of issue updated after datetime for the GraphQL cursor
        :rtype: List[Issues]
        """

        # Define GraphQL Query: filter on updatedAfter and get only opened issues
        query = """
        query ($fullPath: ID!, $after: String, $updatedAfter: Time, $pageSize : Int) {
          project(fullPath: $fullPath) {
            id
            issues(first: $pageSize, after: $after, updatedAfter: $updatedAfter, state : opened) {
              pageInfo {
                hasNextPage
                endCursor
              }
              nodes {
                id
                iid
                title
                createdAt
                updatedAt
              }
            }
          }
        }
        """

        # Define GraphQL variable
        variables = {
            "fullPath": project_path,
            "after": after,
            "updatedAfter": updated_after.isoformat(),
            "pageSize": self.page_size,
        }

        issues: List[Issues] = []
        after = None
        if response := self._post_query(query, variables):
            project_data = response["data"]["project"]
            if project_data is not None:
                # Define project id for REST API from id
                project_id = project_data["id"].split("/")[-1]
                issue_data = project_data["issues"]

                for raw in issue_data["nodes"]:
                    if raw["createdAt"] != raw["updatedAt"]:
                        issue = self._issue_from_json(raw, project_id)
                        issues.append(issue)

                after = self._get_next_cursor(issue_data)

        return issues, after

    def get_new_issues_list(
        self, project_path: str, created_after: datetime
    ) -> List[Issues]:
        """Get issues created after a datetime

        :param project_path: project full path for GraphQL
        :type project_path: str
        :param created_after: datetime for creation
        :type created_after: datetime
        :return: list of issue created after datetime
        :rtype: List[Issues]
        """

        issues, after = self.get_new_issues_list_for_cursor(
            after=None, project_path=project_path, created_after=created_after
        )

        while after is not None:
            page_issue_list, after = self.get_new_issues_list_for_cursor(
                after=after, project_path=project_path, created_after=created_after
            )
            issues.extend(page_issue_list)
        return issues

    def get_new_issues_list_for_cursor(
        self,
        after: Optional[str],
        project_path: str,
        created_after: datetime,
    ) -> Tuple[List[Issues], Optional[str]]:
        """Get issues updated after a datetime for a GraphQL cursor
        :param after : GraphQL cursor, None for first page
        :type after :Optional[str]
        :param project_path: project full path for GraphQL
        :type project_path: str
        :param updated_after: datetime for update
        :type updated_after: datetime
        :return: list of issue updated after datetime for the GraphQL cursor
        :rtype: List[Issues]
        """

        # Define GraphQL Query: filter on createdAfter and get only opened issues
        query = """
        query ($fullPath: ID!, $after: String, $createdAfter: Time, $pageSize : Int) {
          project(fullPath: $fullPath) {
            id
            issues(first: $pageSize, after: $after, createdAfter: $createdAfter, state : opened) {
              pageInfo {
                hasNextPage
                endCursor
              }
              nodes {
                id
                iid
                title
                createdAt
              }
            }
          }
        }
        """

        # Define GraphQL variable

        variables = {
            "fullPath": project_path,
            "after": after,
            "createdAfter": created_after.isoformat(),
            "pageSize": self.page_size,
        }
        issues: List[Issues] = []
        after = None
        if response := self._post_query(query, variables):
            project_data = response["data"]["project"]
            if project_data is not None:
                # Define project id for REST API from id
                project_id = project_data["id"].split("/")[-1]
                issue_data = project_data["issues"]

                for raw in issue_data["nodes"]:
                    issue = self._issue_from_json(raw, project_id)
                    issues.append(issue)

                after = self._get_next_cursor(issue_data)

        return issues, after

    @staticmethod
    def _get_next_cursor(issue_data: dict) -> Optional[str]:
        """Get after cursor from issue data

        :param issue_data: issue data dict
        :type issue_data: dict
        :return: after cursor if available, None otherwise
        :rtype: Optional[str]
        """
        if issue_data["pageInfo"]["hasNextPage"]:
            return issue_data["pageInfo"]["endCursor"]
        return None

    @staticmethod
    def _issue_from_json(raw: dict, project_id: str) -> Issues:
        """Create an issue from json value

        :param raw: raw json value
        :type raw: dict
        :param project_id: issue project id (not available in raw data)
        :type project_id: str
        :return: created issue
        :rtype: Issues
        """
        # Define issue id for REST API from id
        issue_id = raw["id"].split("/")[-1]
        return Issues(
            id=issue_id,
            iid=raw["iid"],
            project_id=project_id,
            name=raw["title"],
            created_date=datetime.fromisoformat(raw["createdAt"]),
        )

    def _post_query(self, query: str, variables: Dict) -> Optional[Dict]:
        """Post GraphQL query with wanted variables

        :param query: GraphQL query
        :type query: str
        :param variables: GraphQL variable
        :type variables: Dict
        :return: query result if no error occurs, None otherwise
        :rtype: Optional[Dict]
        """
        # encode data
        data = QByteArray()
        data_map = {
            "query": query,
            "variables": variables,
        }

        data.append(json.dumps(data_map))

        try:
            req_reply = self.ntwk_requester_blk.post_url(
                url=QUrl(self.plg_settings.gitlab_graphql_url),
                config_id=self.plg_settings.authentification_config_id,
                data=data,
                content_type_header="application/json",
            )
            if req_reply:
                return json.loads(req_reply.data().decode("utf-8"))
            return None
        except ConnectionError:
            return None
