# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ogeoraster
        A QGIS plugin to Access Oracle Georaster using GDAL 
        Generated by Plugin Builder3
                              -------------------
        begin                : 2019-01-02
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Oscars S.A.
        email                : ivan.lucena@oscars-sa.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QMessageBox, QWidget

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .oscars_georaster_dialog import ogeorasterDialog
import os.path

from qgis.core import QgsMessageLog, Qgis
import re


class ogeoraster:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ogeoraster_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Oracle Georaster Plugin')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Georaster to load
        self.selected_datasets = []
        self.layer_index_loading = 0
        self.layer_loaded_count = 0

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ogeoraster', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            # Adds action to Layer > 'Add Layer' menu
            self.iface.addLayerMenu().addAction(action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/oscars_georaster/oscars_georaster_plugin.svg'
        self.add_action(
            icon_path,
            text=self.tr(u'Add Oracle Georaster Layer'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()

        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Oracle Georaster Plugin'),
                action)
            self.iface.removeToolBarIcon(action)

    def loadLayer(self, layer):
        layer_name = '_'.join(layer.split(',')[1:])  
        loaded = self.iface.addRasterLayer(layer, layer_name, 'gdal')

        if not loaded:
            QgsMessageLog.logMessage("Layer  {} was not loaded!".format(layer_name), "Georaster loading")
        else:
            self.layer_loaded_count += 1

    def loadNextLayer(self):
        self.layer_index_loading += 1

        try:
            index = self.selected_datasets[self.layer_index_loading]
            self.loadLayer(self.selected_datasets[self.layer_index_loading])
        except IndexError:
            self.iface.mapCanvas().mapCanvasRefreshed.disconnect(self.loadNextLayer)

            # Notification
            self.iface.messageBar().pushMessage("Success", "{} georasters loaded!".format(self.layer_loaded_count), level=Qgis.Success)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = ogeorasterDialog()

        # show the dialog
        self.dlg.show()

        # Clear goeraster selection
        self.dlg.lstResults.clearSelection()

        # Run the dialog event loop
        result = self.dlg.exec_()

        # See if OK was pressed
        if result:
            # Reset seleceted dataset to load
            self.selected_datasets = []
            self.layer_index_loading = 0
            self.layer_loaded_count = 0
        
            # Selected layer to load
            selected_items = [item.text() for item in self.dlg.lstResults.selectedItems()]
            #QgsMessageLog.logMessage(str(selected_items), "name")

            # Check if large dataset
            if len(selected_items[0].split(',')) == 2:
                answer = QMessageBox.question(QWidget(), 'Large dataset', 'You are about to load large dataset. This may take a while. Would you like to continue?', QMessageBox.Yes | QMessageBox.No, QMessageBox.No)

                if answer == QMessageBox.No:
                    self.dlg.show()
                else:
                    # Get subdatasets
                    for item in selected_items:
                        _, subdatasets = self.dlg.loadMetadata(item, True)
                        
                        for subdataset in subdatasets:
                            if not re.match(r'SUBDATASET_\d*_NAME', subdataset) == None:
                                self.selected_datasets.append(self.dlg.baseId + ',' + ','.join(subdataset.split(",")[1:]))

            else:
                self.selected_datasets = [dataset for dataset in selected_items]

            # Load first georaster
            self.loadLayer(self.selected_datasets[self.layer_index_loading])
            
            # Load next layer when finished loading
            self.iface.mapCanvas().mapCanvasRefreshed.connect(self.loadNextLayer)
        