# -*- coding: utf-8 -*-
"""
/***************************************************************************
 WalkabilityTool
                                 A QGIS plugin
 This plugin calculates walkability index of Urban Neighborhoods
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-02-25
        git sha              : $Format:%H$
        copyright            : (C) 2021 by ILS–Institut für Landes- und Stadtentwicklungsforschung gGmbH
        email                : christian.gerten@ils-forschung.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import QgsMapLayerProxyModel, Qgis, QgsApplication

from qgis.core import (
    QgsMessageLog,
    QgsProcessingFeedback,
    QgsProcessingContext,
    QgsVectorLayer,
    QgsProject,
)

# Initialize Qt resources from file resources.py
# Import the code for the dialog
import os.path

from qgis.utils import iface

try:
    from ORStools.utils import configmanager as ORS_configmanager
    from ORStools.proc import DEFAULT_SETTINGS as ORS_DEFAULT_SETTINGS
except ImportError as e:
    iface.messageBar().pushWarning(
        "ORS Tools", "ORS Tools not found. Please install ORS Tools Plugin."
    )
    raise e

from .algorithm.walkability_algorithm import WalkieProcessingAlgorithm, OsWalkEuProvider
from .gui.walkability_plugin_dialog import WalkabilityToolDialog
from .utils.utility import shortHelpString


class WalkabilityTool:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        try:
            locale = QSettings().value("locale/userLocale")[0:2]
        except:
            locale = "en"  # Fallback to a default locale if needed
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "WalkabilityTool_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&OS-WALK-EU")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # variables
        self.population = None
        self.output = None

    def initProcessing(self):
        self.provider = OsWalkEuProvider()
        QgsApplication.processingRegistry().addProvider(self.provider)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("WalkabilityTool", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)

        QgsMessageLog.logMessage(icon_path, tag="OS-WALK-EU", level=0)

        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        self.initProcessing()

        # self.provider = OsWalkEuProvider()
        # QgsApplication.processingRegistry().addProvider(self.provider)

        icon_path = os.path.join(self.plugin_dir, "images/icons/icon2.png")
        self.add_action(
            icon_path,
            text=self.tr("OS-WALK-EU"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&OS-WALK-EU"), action)
            self.iface.removeToolBarIcon(action)

        try:
            QgsApplication.processingRegistry().removeProvider(self.provider)
        except:  # TODO: Remove try-catching...  # noqa: E722
            pass

    def select_out_folder(self):
        """Method to select the output folder for Walkability tool"""
        saving_folder_dir = QFileDialog.getExistingDirectory(
            None, "Select Output Directory"
        )
        self.dlg.lineEditOutPutFolder.setText(saving_folder_dir)
        self.output = self.dlg.lineEditOutPutFolder.text()
        # print(self.dlg.lineEditOutPutFolder.text())

    def update_spin_boxes(self):
        """Method to update spin boxes on Walkability tool"""
        self.dlg.mQgsSpinBoxMaxWalkDist.value()

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = WalkabilityToolDialog()
            try:
                ors_config = ORS_configmanager.read_config()["providers"]
            except:
                ors_config = ORS_DEFAULT_SETTINGS["providers"]
                # [
                #     {
                #         'ENV_VARS': {
                #             'ORS_QUOTA': 'X-Ratelimit-Limit', 
                #             'ORS_REMAINING': 'X-Ratelimit-Remaining'
                #         }, 
                #         'base_url': 'https://api.openrouteservice.org', 
                #         'key': '', 
                #         'name': 'openrouteservice', 
                #         'timeout': 60
                #     }
                # ]
            self.dlg.orsProviderCombobox.addItems(
                [x["name"] for x in ors_config]
            )

            # Push button to save on folder
            self.dlg.pushButtonOutputFolder.clicked.connect(self.select_out_folder)
            # print("After push button : ")

            # print(self.output)

            # Populate the various comboBox with names of all the loaded layers -------#

            self.dlg.comboBoxInputGrid.setFilters(QgsMapLayerProxyModel.PolygonLayer)
            self.dlg.supermarket_dataset_2.setFilters(QgsMapLayerProxyModel.PointLayer)
            self.dlg.education_dataset_2.setFilters(QgsMapLayerProxyModel.PointLayer)
            self.dlg.shopping_dataset_2.setFilters(QgsMapLayerProxyModel.PointLayer)
            self.dlg.errands_dataset_2.setFilters(QgsMapLayerProxyModel.PointLayer)
            self.dlg.leisure_dataset_2.setFilters(QgsMapLayerProxyModel.PointLayer)
            self.dlg.comboBoxDEM.setFilters(QgsMapLayerProxyModel.RasterLayer)
            self.dlg.comboBoxGreen.setFilters(QgsMapLayerProxyModel.PolygonLayer)

        # Walkability Tool Help String
        self.dlg.textBrowserToolDesc.setText(shortHelpString())

        # -----------------------------------------------------------------------------------#

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            iface.messageBar().pushMessage(
                "Great", "Processing Walkability ...", level=Qgis.Info
            )

            # Call walkie class algorithm here
            walkability_instance = WalkieProcessingAlgorithm()
            walkability_instance.initAlgorithm()

            parameters = {
                "debug_mode": self.dlg.DebugModeBoolean.value(),
                "overpassAPIurl": self.dlg.OverpassApiLineEdit.text(),
                "ors_provider": self.dlg.orsProviderCombobox.currentIndex(),
                "gridLayer": self.dlg.comboBoxInputGrid.currentLayer().id(),
                "maxDist": self.dlg.mQgsSpinBoxMaxWalkDist.value(),
                "elevation": self.dlg.comboBoxDEM.currentLayer().id(),
                "output_folder": self.output,
                "amenities_weight": self.dlg.AMENITIES_WEIGHT.value(),
                "pedestrian_shed_weight": self.dlg.PEDESTRIAN_SHED_WEIGHT.value(),
                "green_area_weight": self.dlg.GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT.value(),
                "supermarket_count": self.dlg.SUPERMARKET_COUNT.value(),
                "education_count": self.dlg.EDUCATION_COUNT.value(),
                "shopping_count": self.dlg.SHOPPING_COUNT.value(),
                "other_errands_count": self.dlg.OTHER_ERRANDS_COUNT.value(),
                "leisure_count": self.dlg.LEISURE_COUNT.value(),
                "profile1": self.dlg.Profiles_1.currentIndex(),
                "profile2": self.dlg.Profiles_2.currentIndex(),
                "supermarket_weights1": self.dlg.AMENITY_SUPERMARKET_WEIGHT_1.value(),
                "supermarket_weights2": self.dlg.AMENITY_SUPERMARKET_WEIGHT_2.value(),
                "supermarket_weights3": self.dlg.AMENITY_SUPERMARKET_WEIGHT_3.value(),
                "list_supermarket_weights": [
                    self.dlg.AMENITY_SUPERMARKET_WEIGHT_1.value(),
                    self.dlg.AMENITY_SUPERMARKET_WEIGHT_2.value(),
                    self.dlg.AMENITY_SUPERMARKET_WEIGHT_3.value(),
                ],
                "education_weights1": self.dlg.AMENITY_EDUCATION_WEIGHT_1.value(),
                "education_weights2": self.dlg.AMENITY_EDUCATION_WEIGHT_2.value(),
                "education_weights3": self.dlg.AMENITY_EDUCATION_WEIGHT_3.value(),
                "list_education_weights": [
                    self.dlg.AMENITY_EDUCATION_WEIGHT_1.value(),
                    self.dlg.AMENITY_EDUCATION_WEIGHT_2.value(),
                    self.dlg.AMENITY_EDUCATION_WEIGHT_3.value(),
                ],
                "list_shopping_weights": [
                    self.dlg.AMENITY_SHOPPING_WEIGHT_1.value(),
                    self.dlg.AMENITY_SHOPPING_WEIGHT_2.value(),
                    self.dlg.AMENITY_SHOPPING_WEIGHT_3.value(),
                ],
                "other_errands_weights1": self.dlg.AMENITY_OTHER_ERRANDS_WEIGHT_1.value(),
                "other_errands_weights2": self.dlg.AMENITY_OTHER_ERRANDS_WEIGHT_2.value(),
                "other_errands_weights3": self.dlg.AMENITY_OTHER_ERRANDS_WEIGHT_3.value(),
                "list_other_errands_weights": [
                    self.dlg.AMENITY_OTHER_ERRANDS_WEIGHT_1.value(),
                    self.dlg.AMENITY_OTHER_ERRANDS_WEIGHT_2.value(),
                    self.dlg.AMENITY_OTHER_ERRANDS_WEIGHT_3.value(),
                ],
                "leisure_weights1": self.dlg.AMENITY_LEISURE_WEIGHT_1.value(),
                "leisure_weights2": self.dlg.AMENITY_LEISURE_WEIGHT_2.value(),
                "leisure_weights3": self.dlg.AMENITY_LEISURE_WEIGHT_3.value(),
                "list_leisure_weights": [
                    self.dlg.AMENITY_LEISURE_WEIGHT_1.value(),
                    self.dlg.AMENITY_LEISURE_WEIGHT_2.value(),
                    self.dlg.AMENITY_LEISURE_WEIGHT_3.value(),
                ],
                "expertMode": self.dlg.expertMode.isChecked(),
                "supermarket_dataset": self.dlg.supermarket_dataset.isChecked(),
                "education_dataset": self.dlg.education_dataset.isChecked(),
                "shopping_dataset": self.dlg.shopping_dataset.isChecked(),
                "errands_dataset": self.dlg.errands_dataset.isChecked(),
                "leisure_dataset": self.dlg.leisure_dataset.isChecked(),
                "supermarketLayer": self.dlg.supermarket_dataset_2.currentLayer().id(),
                "educationLayer": self.dlg.education_dataset_2.currentLayer().id(),
                "shoppingLayer": self.dlg.shopping_dataset_2.currentLayer().id(),
                "errandsLayer": self.dlg.errands_dataset_2.currentLayer().id(),
                "leisureLayer": self.dlg.leisure_dataset_2.currentLayer().id(),
                "elevation_checkbox": self.dlg.checkBox_2.isChecked(),
                "green_checkbox": self.dlg.checkBox_3.isChecked(),
                "greenLayer": self.dlg.comboBoxGreen.currentLayer().id(),
            }

            context = QgsProcessingContext()
            context.setProject(QgsProject.instance())

            processing_result = walkability_instance.processAlgorithm(
                parameters,
                context=context,
                feedback=QgsProcessingFeedback(),
            )

            # wait for 1 min 30 sec before deleting temporary directory
            # time.sleep(90)
            # walkability_instance.delete_temp_dir()
            # print("Done")

            walkability_instance.upload_walkability_result()

            self.dlg.progressBar.setValue(100)

            # Show Done message to user
            iface.messageBar().pushMessage(
                "Done", "Walkability processing done ", level=Qgis.Success
            )
