# -*- coding: utf-8 -*-

"""
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

import os
# import sys

# import numpy as np
# import json
import pandas as pd
import numpy as np
import sqlite3
from textwrap import dedent
import datetime
import random

from pathlib import PurePath  # , PurePosixPath, PureWindowsPath

from qgis.utils import iface
# from qgis.core import QgsMessageLog

try:
    from ORStools.utils import configmanager as ORS_configmanager
    from ORStools.proc import DEFAULT_SETTINGS as ORS_DEFAULT_SETTINGS
except ImportError as e:
    iface.messageBar().pushWarning(
        "ORS Tools", "ORS Tools not found. Please install ORS Tools Plugin."
    )
    raise e

from ..gui.walkability_plugin_dialog import WalkabilityToolDialog

from qgis.PyQt.QtCore import QCoreApplication, QVariant

from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsVectorLayer,
    QgsVectorFileWriter,
    QgsProcessing,
    QgsProcessingProvider,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFolderDestination,
    QgsProcessingParameterVectorLayer,
    QgsProcessingParameterDistance,
    QgsProcessingParameterNumber,
    QgsProcessingParameterString,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterEnum,
    QgsProcessingException,
    QgsFeature,
    QgsFeatureRequest,
    QgsField,
    QgsWkbTypes,
    QgsProject,
    QgsDataSourceUri,
)

import processing
from osgeo import ogr


class OsWalkEuProvider(QgsProcessingProvider):
    def loadAlgorithms(self):
        self.addAlgorithm(WalkieProcessingAlgorithm())

    def id(self):
        return "os_walk_eu"

    def name(self):
        return "OS-WALK-EU"

    def longName(self):
        return "OS-WALK-EU Processing Provider"


class WalkieProcessingAlgorithm(QgsProcessingAlgorithm):
    """
    This is a walkability algorithm
    """

    # DIST_FIELD = "DIST_KM"  # ORS distance field in matrix

    def name(self):
        return "os_walk_eu_walkability_algorithm"

    def displayName(self):
        return "OS-WALK-EU - Walkability Algorithm"

    def group(self):
        return "OS-WALK-EU"

    def groupId(self):
        return "os_walk_eu"

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("WalkabilityTool", message)

    def initAlgorithm(self, config={}):
        # main parameters
        try:
            ors_config = ORS_configmanager.read_config()["providers"]
        except:
            ors_config = ORS_DEFAULT_SETTINGS["providers"]
            # [
            #     {
            #         'ENV_VARS': {
            #             'ORS_QUOTA': 'X-Ratelimit-Limit', 
            #             'ORS_REMAINING': 'X-Ratelimit-Remaining'
            #         }, 
            #         'base_url': 'https://api.openrouteservice.org', 
            #         'key': '', 
            #         'name': 'openrouteservice', 
            #         'timeout': 60
            #     }
            # ]
        
        self.addParameter(
            QgsProcessingParameterBoolean(
                "debug_mode",  # checkBox_2
                self.tr("Check Checkbox if you want to limit your Features to 500 and give additional debug output"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                "ors_provider",  # ???
                self.tr("ORS Provider"),
                options=[
                    x["name"] for x in ors_config
                ],
                allowMultiple=False,
                defaultValue=ors_config[0]["name"],
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                "overpassAPIurl",  # OverpassApiLineEdit
                self.tr("Overpass-API-URL (for download of OSM-Data)"),
                defaultValue="https://overpass.kumi.systems/api/interpreter?data=",
                # URL = 'https://z.overpass-api.de/api/interpreter?data='
                # URL = 'https://lz4.overpass-api.de/api/interpreter?data='
            )
        )

        ## Geometry
        self.addParameter(
            # QgsProcessingParameterFeatureSource(
            QgsProcessingParameterVectorLayer(
                "gridLayer",  # comboBoxInputGrid
                self.tr("Geometry"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )
        ## Pedestrian profiles
        self.addParameter(
            QgsProcessingParameterEnum(
                "profile1",  # Profiles_1
                self.tr("Age group"),
                options=[
                    self.tr("All"),
                    self.tr("under 17"),
                    self.tr("between 17 and 30"),
                    self.tr("between 30 and 70"),
                    self.tr("Over 70"),
                ],
                allowMultiple=False,
                defaultValue=0,
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                "profile2",  # Profiles_2
                self.tr("Socio-economic status"),
                options=[
                    self.tr("All"),
                    self.tr("Low"),
                    self.tr("Medium"),
                    self.tr("High"),
                ],
                allowMultiple=False,
                defaultValue=0,
            )
        )
        ## Slope
        self.addParameter(
            QgsProcessingParameterBoolean(
                "elevation_checkbox",  # checkBox_2
                self.tr("Optional: slope reduction"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "elevation",  # comboBoxDEM
                self.tr("Digital elevation model"),
                [QgsProcessing.TypeRaster],
                optional=True,
            )
        )
        ## green
        self.addParameter(
            QgsProcessingParameterBoolean(
                "green_checkbox",  # checkBox_3
                self.tr("Optional: Use individual data for urban green"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "greenLayer",  # comboBoxGreen
                self.tr("Urban green"),
                [QgsProcessing.TypeVectorPolygon],
                optional=True,
            )
        )
        ##Output
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                "output_folder",  # lineEditOutPutFolder
                self.tr("Output folder"),
            )
        )

        # Expert settings
        self.addParameter(
            QgsProcessingParameterBoolean(
                "expertMode",  # expertMode
                self.tr(
                    "Use expert mode with settings below (deactive to use predefined profiles)"
                ),
                defaultValue=False,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                "amenities_weight",  # AMENITIES_WEIGHT
                self.tr("Ameneties"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=1.0,
                defaultValue=1.0,
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "pedestrian_shed_weight",  # PEDESTRIAN_SHED_WEIGHT
                self.tr("Pedestrian network"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=1.0,
                defaultValue=0.4,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "green_area_weight",  # GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT
                self.tr("Green infrastructure"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=1.0,
                defaultValue=0.6,
            )
        )
        self.addParameter(
            QgsProcessingParameterDistance(  # or something from QgsProcessingParameterDefinition?
                "maxDist",  # mQgsSpinBoxMaxWalkDist
                self.tr("Distance"),
                defaultValue=500,
                minValue=250,
                maxValue=2500,
            )
        )

        ## supermarket
        self.addParameter(
            QgsProcessingParameterBoolean(
                "supermarket_dataset",  # supermarket_dataset
                self.tr("Supermarket"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "supermarketLayer",  # supermarket_dataset_2
                self.tr("Point layer with supermarkets"),
                [QgsProcessing.TypeVectorPoint],
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "supermarket_count",  # SUPERMARKET_COUNT
                self.tr("Supermarket count"),
                type=QgsProcessingParameterNumber.Integer,
                minValue=0,
                defaultValue=1,
            )
        )
        # list_supermarket_weights
        self.addParameter(
            QgsProcessingParameterNumber(
                "supermarket_weight1",  # AMENITY_SUPERMARKET_WEIGHT_1
                self.tr("Supermarket weight 1"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=4.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "supermarket_weight2",  # AMENITY_SUPERMARKET_WEIGHT_2
                self.tr("Supermarket weight 2"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=0.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "supermarket_weight3",  # AMENITY_SUPERMARKET_WEIGHT_3
                self.tr("Supermarket weight 3"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=0.0,
            )
        )

        ## Education
        self.addParameter(
            QgsProcessingParameterBoolean(
                "education_dataset",  # education_dataset
                self.tr("Education"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "educationLayer",  # education_dataset_2
                self.tr("Point layer with education amenities"),
                [QgsProcessing.TypeVectorPoint],
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "education_count",  # EDUCATION_COUNT
                self.tr("Education count"),
                type=QgsProcessingParameterNumber.Integer,
                minValue=0,
                defaultValue=2,
            )
        )
        # list_education_weights
        self.addParameter(
            QgsProcessingParameterNumber(
                "education_weight1",  # AMENITY_EDUCATION_WEIGHT_1
                self.tr("Education weight 1"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=2.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "education_weight2",  # AMENITY_EDUCATION_WEIGHT_2
                self.tr("Education weight 2"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=1.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "education_weight3",  # AMENITY_EDUCATION_WEIGHT_3
                self.tr("Education weight 3"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=0.0,
            )
        )

        ## Shopping
        self.addParameter(
            QgsProcessingParameterBoolean(
                "shopping_dataset",  # shopping_dataset
                self.tr("Shopping"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "shoppingLayer",  # shopping_dataset_2
                self.tr("Point layer with shopping amenities"),
                [QgsProcessing.TypeVectorPoint],
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "shopping_count",  # SHOPPING_COUNT
                self.tr("Shopping count"),
                type=QgsProcessingParameterNumber.Integer,
                minValue=0,
                defaultValue=2,
            )
        )
        # list_shopping_weights
        self.addParameter(
            QgsProcessingParameterNumber(
                "shopping_weight1",  # AMENITY_SHOPPING_WEIGHT_1
                self.tr("Shopping weight 1"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=2.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "shopping_weight2",  # AMENITY_SHOPPING_WEIGHT_2
                self.tr("Shopping weight 2"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=1.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "shopping_weight3",  # AMENITY_SHOPPING_WEIGHT_3
                self.tr("Shopping weight 3"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=0.0,
            )
        )

        ## Other Errands
        self.addParameter(
            QgsProcessingParameterBoolean(
                "errands_dataset",  # errands_dataset
                self.tr("Other Errands"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "errandsLayer",  # errands_dataset_2
                self.tr("Point layer with other errends"),
                [QgsProcessing.TypeVectorPoint],
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "other_errands_count",  # OTHER_ERRANDS_COUNT
                self.tr("Other Errands count"),
                type=QgsProcessingParameterNumber.Integer,
                minValue=0,
                defaultValue=3,
            )
        )
        # list_other_errands_weights
        self.addParameter(
            QgsProcessingParameterNumber(
                "other_errands_weight1",  # AMENITY_OTHER_ERRANDS_WEIGHT_1
                self.tr("Other Errands weight 1"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=2.5,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "other_errands_weight2",  # AMENITY_OTHER_ERRANDS_WEIGHT_2
                self.tr("Other Errands weight 2"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=1.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "other_errands_weight3",  # AMENITY_OTHER_ERRANDS_WEIGHT_3
                self.tr("Other Errands weight 3"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=1.0,
            )
        )

        ## Leisure
        self.addParameter(
            QgsProcessingParameterBoolean(
                "leisure_dataset",  # leisure_dataset
                self.tr("Leisure"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "leisureLayer",  # leisure_dataset_2
                self.tr("Point layer with leisure amenities"),
                [QgsProcessing.TypeVectorPoint],
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "leisure_count",  # LEISURE_COUNT
                self.tr("Leisure count"),
                type=QgsProcessingParameterNumber.Integer,
                minValue=0,
                defaultValue=1,
            )
        )
        # list_leisure_weights
        self.addParameter(
            QgsProcessingParameterNumber(
                "leisure_weight1",  # AMENITY_LEISURE_WEIGHT_1
                self.tr("Leisure weight 1"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=2.5,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "leisure_weight2",  # AMENITY_LEISURE_WEIGHT_2
                self.tr("Leisure weight 2"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=1.5,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                "leisure_weight3",  # AMENITY_LEISURE_WEIGHT_3
                self.tr("Leisure weight 3"),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                maxValue=10.0,
                defaultValue=0.0,
            )
        )
        # self.addParameter(
        #     QgsProcessingParameterNumber(
        #         name="max_ors_workers",
        #         description= self.tr("Number of parallel requests to ORS (1-8) | ONLY FOR CUSTOM ORS DEPLOYMENT!"),
        #         defaultValue=1,
        #         minValue=1,
        #         maxValue=8,
        #         optional=True,
        #     )
        # )

    def createInstance(self):
        return WalkieProcessingAlgorithm()

    def __init__(self):
        super().__init__()
        try:
            if hasattr(iface, "mainWindow") and iface.mainWindow() is not None:
                self.is_cli = False
            else:
                self.is_cli = True
        except NameError:
            self.is_cli = True
        if self.is_cli: print(f"run in cli: {self.is_cli}", flush=True)
        self.starttime = None
        self.DEBUG_MODE = False

    def log_time(self):
        if self.starttime is None:
            self.starttime = datetime.datetime.now()
        return f"{datetime.datetime.now().isoformat()} - {str(datetime.datetime.now() - self.starttime)}"

    def custom_feedback(self, message, level="info"):
        if level == "debug":
            if self.DEBUG_MODE:
                if self.RAN_FROM_GUI:
                    iface.messageBar().pushMessage("Debug", message, level="debug")
                else:
                    self.feedback.pushDebugInfo(message)
                    if self.is_cli: print( f"[{self.log_time()}] [DEBUG] {message}", flush=True )
        elif level == "info":
            if self.RAN_FROM_GUI:
                iface.messageBar().pushMessage("Info", message, level="info")
            else:
                self.feedback.setProgressText(message)
                self.feedback.pushInfo(message)
                if self.is_cli: print(f"[{self.log_time()}] [INFO] {message}", flush=True)
        elif level == "warning":
            if self.RAN_FROM_GUI:
                iface.messageBar().pushMessage("Warning", message, level="warning")
            else:
                self.feedback.setProgressText(message)
                self.feedback.pushWarning(message)
                if self.is_cli: print(f"[{self.log_time()}] [WARNING] {message}", flush=True)
        elif level == "critical":
            if self.RAN_FROM_GUI:
                iface.messageBar().pushMessage("Error", message, level="critical")
            else:
                self.feedback.setProgressText(message)
                self.feedback.reportError(message)
                if self.is_cli: print(f"[{self.log_time()}] [ERROR] {message}", flush=True)

    def load_parameters(self, parameters, context, feedback):
        self.feedback = feedback
        self.context = context

        self.RAN_FROM_GUI = parameters.get("ran_from_gui", False)

        # self.MAX_ORS_WORKERS = self.parameterAsInt(parameters, "max_ors_workers", context)
        # if self.MAX_ORS_WORKERS is None or self.MAX_ORS_WORKERS < 1:
        #     self.MAX_ORS_WORKERS = 1  # Fallback default
        # feedback.pushDebugInfo(f"Using {self.MAX_ORS_WORKERS} parallel workers")
        # if self.is_cli: print(f"Using {self.MAX_ORS_WORKERS} parallel workers", flush=True)

        self.custom_feedback("loading defaults for use of profiles...")

        profileList = [
            "P00",
            "P01",
            "P02",
            "P03",
            "P10",
            "P20",
            "P30",
            "P40",
            "P10",
            "P11",
            "P12",
            "P13",
            "P20",
            "P21",
            "P22",
            "P23",
            "P30",
            "P31",
            "P32",
            "P33",
            "P40",
            "P41",
            "P42",
            "P43",
        ]
        vals = [
            [
                1,
                0.4,
                0.6,
                1,
                2,
                2,
                3,
                2,
                4,
                0,
                0,
                2,
                1,
                0,
                2,
                1,
                0,
                2.5,
                1,
                1,
                2.5,
                1.5,
                0,
                500,
            ],  # P00
            [
                1,
                0.4,
                0.6,
                1,
                2,
                2,
                3,
                2,
                4.5,
                0,
                0,
                2,
                1.5,
                0,
                2,
                1,
                0,
                3,
                1,
                1,
                2.5,
                1,
                0,
                500,
            ],  # P01
            [
                1,
                0.4,
                0.6,
                1,
                2,
                2,
                3,
                3,
                9,
                0,
                0,
                2,
                1.5,
                0,
                4,
                2,
                0,
                5,
                3,
                2,
                4,
                2.5,
                1.5,
                500,
            ],  # P02
            [
                1,
                0.4,
                0.6,
                1,
                2,
                2,
                3,
                3,
                7,
                0,
                0,
                5,
                3,
                0,
                4,
                2,
                0,
                4,
                2,
                1.5,
                4,
                3,
                1.5,
                500,
            ],  # P03
            [
                1,
                0.4,
                0.6,
                1,
                2,
                1,
                1,
                2,
                1,
                0,
                0,
                4.5,
                2.5,
                0,
                1.5,
                0,
                0,
                1.5,
                0,
                0,
                3,
                1,
                0,
                500,
            ],  # P10
            [
                1,
                0.2,
                0.6,
                1,
                2,
                2,
                2,
                3,
                3,
                0,
                0,
                3,
                1,
                0,
                2,
                1,
                0,
                2.5,
                1.5,
                0,
                2,
                1.5,
                1,
                700,
            ],  # P20
            [
                1,
                0.3,
                0.6,
                1,
                0,
                2,
                3,
                2,
                5,
                0,
                0,
                0,
                0,
                0,
                2,
                1.5,
                0,
                3,
                1.5,
                1,
                3,
                1,
                0,
                600,
            ],  # P30
            [
                1,
                0.5,
                0.6,
                1,
                0,
                2,
                3,
                2,
                5,
                0,
                0,
                0,
                0,
                0,
                2,
                1,
                0,
                3.5,
                2,
                1,
                2.5,
                1,
                0,
                400,
            ],  # P40
            [
                1,
                0.4,
                0.6,
                1,
                2,
                1,
                1,
                2,
                1,
                0,
                0,
                7,
                3,
                0,
                1.5,
                0,
                0,
                1.5,
                0,
                0,
                2.5,
                1,
                0,
                500,
            ],  # P11
            [
                1,
                0.4,
                0.6,
                1,
                2,
                1,
                1,
                2,
                1,
                0,
                0,
                7,
                3,
                0,
                1,
                0,
                0,
                1.5,
                0,
                0,
                3,
                1,
                0,
                500,
            ],  # P12
            [
                1,
                0.4,
                0.6,
                1,
                2,
                1,
                1,
                2,
                1,
                0,
                0,
                7,
                3,
                0,
                1.5,
                0,
                0,
                1,
                0,
                0,
                3,
                1,
                0,
                500,
            ],  # P13
            [
                1,
                0.2,
                0.6,
                1,
                2,
                2,
                2,
                2,
                3.5,
                0,
                0,
                3.5,
                2,
                0,
                1.5,
                1,
                0,
                2.5,
                1.5,
                0,
                3,
                1,
                0,
                700,
            ],  # P21
            [
                1,
                0.2,
                0.6,
                1,
                2,
                2,
                2,
                3,
                4,
                0,
                0,
                2,
                1.5,
                0,
                2,
                1,
                0,
                2.5,
                1.5,
                0,
                2,
                1.5,
                1,
                700,
            ],  # P22
            [
                1,
                0.2,
                0.6,
                1,
                2,
                2,
                2,
                3,
                2.5,
                0,
                0,
                3,
                1.5,
                0,
                1.5,
                1,
                0,
                2.5,
                1.5,
                0,
                2,
                1.5,
                1,
                700,
            ],  # P23
            [
                1,
                0.3,
                0.6,
                1,
                0,
                2,
                3,
                2,
                6,
                0,
                0,
                0,
                0,
                0,
                3,
                1,
                0,
                3,
                2,
                1,
                2.5,
                1,
                0,
                600,
            ],  # P31
            [
                1,
                0.3,
                0.6,
                1,
                0,
                2,
                3,
                2,
                5,
                0,
                0,
                0,
                0,
                0,
                2,
                1.5,
                0,
                3,
                1.5,
                1,
                3,
                1,
                0,
                600,
            ],  # P32
            [
                1,
                0.3,
                0.6,
                1,
                0,
                2,
                3,
                3,
                5,
                0,
                0,
                0,
                0,
                0,
                3,
                1,
                0,
                3,
                1,
                1,
                2,
                1.5,
                1,
                600,
            ],  # P33
            [
                1,
                0.5,
                0.6,
                1,
                0,
                2,
                3,
                2,
                5.5,
                0,
                0,
                0,
                0,
                0,
                2,
                1,
                0,
                3.5,
                2,
                1,
                2.5,
                1,
                0,
                400,
            ],  # P41
            [
                1,
                0.5,
                0.6,
                1,
                0,
                2,
                3,
                2,
                5,
                0,
                0,
                0,
                0,
                0,
                2,
                1,
                0,
                3.5,
                2,
                1,
                2.5,
                1,
                0,
                400,
            ],  # P42
            [
                1,
                0.5,
                0.6,
                1,
                0,
                2,
                3,
                2,
                5.5,
                0,
                0,
                0,
                0,
                0,
                2,
                1,
                0,
                3,
                2,
                1,
                3,
                1,
                0,
                400,
            ],  # P43
        ]

        self.custom_feedback("loading parameters...")

        self.PROFILE_1 = self.parameterAsInt(parameters, "profile1", context)
        self.PROFILE_2 = self.parameterAsInt(parameters, "profile2", context)
        Profile = "P" + str(self.PROFILE_1) + str(self.PROFILE_2)
        index = profileList.index(Profile)
        self.custom_feedback(
            str(self.PROFILE_1) + str(self.PROFILE_2) + "check", "debug"
        )

        self.DEBUG_MODE = self.parameterAsBoolean(parameters, "debug_mode", context)

        self.ORS_PROVIDER = self.parameterAsInt(parameters, "ors_provider", context)

        self.SUPERMARKET_DATASET = self.parameterAsBoolean(
            parameters, "supermarket_dataset", context
        )
        self.EDUCATION_DATASET = self.parameterAsBoolean(
            parameters, "education_dataset", context
        )
        self.SHOPPING_DATASET = self.parameterAsBoolean(
            parameters, "shopping_dataset", context
        )
        self.ERRANDS_DATASET = self.parameterAsBoolean(
            parameters, "errands_dataset", context
        )
        self.LEISURE_DATASET = self.parameterAsBoolean(
            parameters, "leisure_dataset", context
        )

        self.MAX_DIST = vals[index][23]

        # Weightings for the calculation of the final score
        self.AMENITIES_WEIGHT = vals[index][0]
        self.PEDESTRIAN_SHED_WEIGHT = vals[index][1]
        self.GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT = vals[index][2]

        self.SUPERMARKET_COUNT = vals[index][3]
        self.AMENITY_SUPERMARKET_WEIGHT_1 = vals[index][8]
        self.AMENITY_SUPERMARKET_WEIGHT_2 = vals[index][9]
        self.AMENITY_SUPERMARKET_WEIGHT_3 = vals[index][10]

        self.EDUCATION_COUNT = vals[index][3]
        self.AMENITY_EDUCATION_WEIGHT_1 = vals[index][11]
        self.AMENITY_EDUCATION_WEIGHT_2 = vals[index][12]
        self.AMENITY_EDUCATION_WEIGHT_3 = vals[index][13]

        self.SHOPPING_COUNT = vals[index][5]
        self.AMENITY_SHOPPING_WEIGHT_1 = vals[index][14]
        self.AMENITY_SHOPPING_WEIGHT_2 = vals[index][15]
        self.AMENITY_SHOPPING_WEIGHT_3 = vals[index][16]

        self.OTHER_ERRANDS_COUNT = vals[index][6]
        self.AMENITY_OTHER_ERRANDS_WEIGHT_1 = vals[index][17]
        self.AMENITY_OTHER_ERRANDS_WEIGHT_2 = vals[index][18]
        self.AMENITY_OTHER_ERRANDS_WEIGHT_3 = vals[index][19]

        self.LEISURE_COUNT = vals[index][7]
        self.AMENITY_LEISURE_WEIGHT_1 = vals[index][20]
        self.AMENITY_LEISURE_WEIGHT_2 = vals[index][21]
        self.AMENITY_LEISURE_WEIGHT_3 = vals[index][22]

        self.EXPERT_MODE = self.parameterAsBoolean(parameters, "expertMode", context)
        if self.EXPERT_MODE is True:
            self.custom_feedback(
                "overwriting parameters with expert parameters where configured..."
            )
            # Define maxDistances in GUI
            self.MAX_DIST = self.get(
                self.parameterAsInt(parameters, "maxDist", context), vals[index][23]
            )

            # Weightings for the calculation of the final score
            self.AMENITIES_WEIGHT = self.get(
                self.parameterAsDouble(parameters, "amenities_weight", context),
                vals[index][0],
            )
            self.PEDESTRIAN_SHED_WEIGHT = self.get(
                self.parameterAsDouble(parameters, "pedestrian_shed_weight", context),
                vals[index][1],
            )
            self.GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT = self.get(
                self.parameterAsDouble(parameters, "green_area_weight", context),
                vals[index][2],
            )

            if self.SUPERMARKET_DATASET is True:
                self.SUPERMARKET_LAYER = self.parameterAsVectorLayer(
                    parameters, "supermarketLayer", context
                )
                self.SUPERMARKET_COUNT = self.get(
                    self.parameterAsDInteger(parameters, "supermarket_count", context),
                    vals[index][3],
                )
                self.AMENITY_SUPERMARKET_WEIGHT_1 = self.get(
                    self.parameterAsDouble(parameters, "supermarket_weights1", context),
                    vals[index][8],
                )
                self.AMENITY_SUPERMARKET_WEIGHT_2 = self.get(
                    self.parameterAsDouble(parameters, "supermarket_weights2", context),
                    vals[index][9],
                )
                self.AMENITY_SUPERMARKET_WEIGHT_3 = self.get(
                    self.parameterAsDouble(parameters, "supermarket_weights3", context),
                    vals[index][10],
                )
            if self.EDUCATION_DATASET is True:
                self.EDUCATION_LAYER = self.parameterAsVectorLayer(
                    parameters, "educationLayer", context
                )
                self.EDUCATION_COUNT = self.get(
                    self.parameterAsInt(parameters, "education_count", context),
                    vals[index][3],
                )
                self.AMENITY_EDUCATION_WEIGHT_1 = self.get(
                    self.parameterAsDouble(parameters, "education_weights1", context),
                    vals[index][11],
                )
                self.AMENITY_EDUCATION_WEIGHT_2 = self.get(
                    self.parameterAsDouble(parameters, "education_weights2", context),
                    vals[index][12],
                )
                self.AMENITY_EDUCATION_WEIGHT_3 = self.get(
                    self.parameterAsDouble(parameters, "education_weights3", context),
                    vals[index][13],
                )
            if self.SHOPPING_DATASET is True:
                self.SHOPPING_LAYER = self.parameterAsVectorLayer(
                    parameters, "shoppingLayer", context
                )
                self.SHOPPING_COUNT = self.get(
                    self.parameterAsInt(parameters, "shopping_count", context),
                    vals[index][5],
                )
                self.AMENITY_SHOPPING_WEIGHT_1 = self.get(
                    self.parameterAsDouble(parameters, "shopping_weights1", context),
                    vals[index][14],
                )
                self.AMENITY_SHOPPING_WEIGHT_2 = self.get(
                    self.parameterAsDouble(parameters, "shopping_weights2", context),
                    vals[index][15],
                )
                self.AMENITY_SHOPPING_WEIGHT_3 = self.get(
                    self.parameterAsDouble(parameters, "shopping_weights3", context),
                    vals[index][16],
                )
            if self.ERRANDS_DATASET is True:
                self.ERRANDS_LAYER = self.parameterAsVectorLayer(
                    parameters, "errandsLayer", context
                )
                self.OTHER_ERRANDS_COUNT = self.get(
                    self.parameterAsInt(parameters, "other_errands_count", context),
                    vals[index][6],
                )
                self.AMENITY_OTHER_ERRANDS_WEIGHT_1 = self.get(
                    self.parameterAsDouble(
                        parameters, "other_errands_weights1", context
                    ),
                    vals[index][17],
                )
                self.AMENITY_OTHER_ERRANDS_WEIGHT_2 = self.get(
                    self.parameterAsDouble(
                        parameters, "other_errands_weights2", context
                    ),
                    vals[index][18],
                )
                self.AMENITY_OTHER_ERRANDS_WEIGHT_3 = self.get(
                    self.parameterAsDouble(
                        parameters, "other_errands_weights3", context
                    ),
                    vals[index][19],
                )
            if self.LEISURE_DATASET is True:
                self.LEISURE_LAYER = self.parameterAsDouble(
                    parameters, "leisureLayer", context
                )
                self.LEISURE_COUNT = self.get(
                    self.parameterAsInt(parameters, "leisure_count", context),
                    vals[index][7],
                )
                self.AMENITY_LEISURE_WEIGHT_1 = self.get(
                    self.parameterAsDouble(parameters, "leisure_weights1", context),
                    vals[index][20],
                )
                self.AMENITY_LEISURE_WEIGHT_2 = self.get(
                    self.parameterAsDouble(parameters, "leisure_weights2", context),
                    vals[index][21],
                )
                self.AMENITY_LEISURE_WEIGHT_3 = self.get(
                    self.parameterAsDouble(parameters, "leisure_weights3", context),
                    vals[index][22],
                )

        self.GRID_LAYER = self.parameterAsVectorLayer(parameters, "gridLayer", context)

        self.ELEVATION_CHECKBOX = self.parameterAsBoolean(
            parameters, "elevation_checkbox", context
        )
        self.ELEVATION = self.parameterAsRasterLayer(parameters, "elevation", context)

        self.GREEN_CHECKBOX = self.parameterAsBoolean(
            parameters, "green_checkbox", context
        )

        self.GREEN_LAYER = self.parameterAsVectorLayer(
            parameters, "greenLayer", context
        )

        self.OUTPUT_FOLDER = self.parameterAsFile(parameters, "output_folder", context)
        if not self.OUTPUT_FOLDER:
            iface.messageBar().pushWarning(
                "Critical",
                "No output directory specified. Please specify an output directory.",
            )
            raise QgsProcessingException(
                self.invalidSourceError(parameters, "output_folder")
            )
        else:
            self.custom_feedback("Output directory: " + self.OUTPUT_FOLDER, "debug")

        self.OVERPASS_API_URL = self.parameterAsString(
            parameters, "overpassAPIurl", context
        )

        self.custom_feedback("loading parameters...done")

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        self.load_parameters(parameters, context, feedback)

        self.custom_feedback(
            "From __dir__():"
            + str(
                {
                    x: eval("self." + str(x))
                    for x in self.__dir__()[: self.__dir__().index("__module__")]
                    if not isinstance(eval("self." + str(x)), QgsVectorLayer)
                }
                | {
                    x: [eval("self." + str(x)).type(), eval("self." + str(x)).id()]
                    for x in self.__dir__()[: self.__dir__().index("__module__")]
                    if isinstance(eval("self." + str(x)), QgsVectorLayer)
                }
            ),
            "debug",
        )
        self.custom_feedback(
            "From parameterDefinitions():"
            + str(
                {param.name(): param.type() for param in self.parameterDefinitions()}
            ),
            "debug",
        )

        try:
            os.remove(self.OUTPUT_FOLDER + r"/tempDir")
            self.custom_feedback("cleaned up tempDir", "debug")
        except PermissionError as e:
            raise e
        except FileNotFoundError:
            pass
        except Exception as e:
            raise e

        try:
            os.makedirs(self.OUTPUT_FOLDER + r"/tempDir", exist_ok=True)
            self.custom_feedback("Created tempDir", "debug")
        except FileExistsError as e:
            raise e
        except Exception as e:
            raise e

        gpkgs = [
            "WalkEU.gpkg",
            "tempDir/WalkEU_process.gpkg",
            # "tempDir/amenities.gpkg",
            # "tempDir/temp.gpkg",
        ]
        for gpkg in gpkgs:
            gpkg_path = os.path.join(self.OUTPUT_FOLDER, gpkg)
            if not os.path.exists(gpkg_path):
                empty_layer = QgsVectorLayer("Point?crs=EPSG:4326", "dummy", "memory")
                QgsVectorFileWriter.writeAsVectorFormat(
                    empty_layer, gpkg_path, "UTF-8", empty_layer.crs(), "GPKG"
                )

        try:  # fails for non-expertmode
            poiLayer = [
                self.SUPERMARKET_LAYER,
                self.EDUCATION_LAYER,
                self.SHOPPING_LAYER,
                self.ERRANDS_LAYER,
                self.LEISURE_LAYER,
            ]
            poiName = ["SU", "ED", "SH", "OE", "LE"]
            c = 0
            for layer in poiLayer:
                processing.run(
                    "gdal:convertformat",
                    {
                        "INPUT": layer,
                        "OPTIONS": "",
                        "OUTPUT": "ogr:dbname='"
                        + os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                        )
                        + "' table="
                        + poiName[c]
                        + "",
                    },
                )
                c += 1
            # Add an incremental ID to the layer for later processing.algorithmHelp("native:addautoincrementalfield")
        except:  # noqa: E722
            pass

        





        self.custom_feedback("Starting processing...", "info")
        # self.custom_feedback("Adding Walk_ID as incrementing field", "debug")
        # autoIncrement = processing.run(
        #     "native:addautoincrementalfield",
        #     {
        #         "INPUT": self.GRID_LAYER,
        #         "FIELD_NAME": "Walk_ID",
        #         "START": 1,
        #         "GROUP_FIELDS": [],
        #         "SORT_EXPRESSION": "",
        #         "SORT_ASCENDING": True,
        #         "SORT_NULLS_FIRST": False,
        #         "OUTPUT": "ogr:dbname='"
        #         + os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
        #         + "' table=Grid_500",
        #     },
        # )["OUTPUT"]


        # self.custom_feedback("Assigning CRS to the reference layer", "debug")
        # autoIncrementproj = processing.run(
        #     "native:assignprojection",
        #     {
        #         "INPUT": autoIncrement,
        #         "CRS": QgsCoordinateReferenceSystem("EPSG:3035"),
        #         "OUTPUT": "ogr:dbname='"
        #         + os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
        #         + "' table=increment",
        #     },
        # )["OUTPUT"]

        self.custom_feedback(
            "Adding Walk_ID as incrementing field, this time exporting "
            + "it to geojson not keep it only in memory",
            "info",
        )
        grid_id_layer_path = processing.run(
            "native:addautoincrementalfield",
            {
                "INPUT": self.GRID_LAYER,
                "FIELD_NAME": "Walk_ID",
                "START": 1,
                "MODULUS": 0,
                "GROUP_FIELDS": [],
                "SORT_EXPRESSION": "",
                "SORT_ASCENDING": True,
                "SORT_NULLS_FIRST": False,
                "OUTPUT": "ogr:dbname='"
                + os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
                + "' table=grid500_id",
            },
        )["OUTPUT"]
        grid_id_layer = QgsVectorLayer(grid_id_layer_path, "grid500_id", "ogr")
        grid_id = QgsDataSourceUri(grid_id_layer_path).uri()
        self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")
        
        self.custom_feedback("Extracting centroids for routing purposes", "info")
        polyCentroids = processing.run(
            "native:centroids",
            {
                #"INPUT": autoIncrement,
                #"INPUT": autoIncrementproj,
                "INPUT": grid_id_layer,
                "ALL_PARTS": False,
                "OUTPUT": "ogr:dbname='" + os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg") + "' table=polycentroids",
            },
        )["OUTPUT"]

        polyCentroids4326 = processing.run(
            "native:reprojectlayer", 
            {
                'INPUT':polyCentroids,
                'TARGET_CRS':QgsCoordinateReferenceSystem('EPSG:4326'),
                'CONVERT_CURVED_GEOMETRIES':False,
                #'OPERATION':'+proj=pipeline +step +inv +proj=utm +zone=32 +ellps=GRS80 +step +proj=unitconvert +xy_in=rad +xy_out=deg',
                'OUTPUT':"ogr:dbname='"
                + os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
                + "' table=polyCentroids4326",
            }
        )["OUTPUT"]
        
        polyCentroids4326_layer = QgsVectorLayer(polyCentroids4326, "grid500_id_centroids4326", "ogr")
        polyCentroids4326_id = QgsDataSourceUri(polyCentroids4326).uri()

        if self.DEBUG_MODE:
            self.custom_feedback("We are limiting our features to be processed to 500 due to debug-mode:")

            
            # Zufälliges Feature auswählen
            features = list(polyCentroids4326_layer.getFeatures())
            random_feature = random.choice(features)
            self.custom_feedback(f"{random_feature}", "debug")

            # Temporären Layer mit dem zufälligen Feature erstellen
            temp_layer = QgsVectorLayer("Point?crs=" + polyCentroids4326_layer.crs().authid(), "temp", "memory")
            temp_layer.dataProvider().addAttributes(polyCentroids4326_layer.fields())
            temp_layer.updateFields()
            temp_layer.dataProvider().addFeature(random_feature)
            temp_layer.updateExtents()
            
            options = QgsVectorFileWriter.SaveVectorOptions()
            options.layerName="samplepoint"
            options.format="GPKG"
            options.encoding='UTF-8'
            options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer 

            writeresult = QgsVectorFileWriter.writeAsVectorFormatV3(
                temp_layer,
                f"{os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")}",
                QgsProject.instance().transformContext(),
                options=options
            )
            self.custom_feedback(f"Was writing of sample successful? -> {writeresult}", "debug")

            # Distanzmatrix berechnen
            polyCentroids4326_distances_layer_path = processing.run("qgis:distancematrix", {
                'INPUT': polyCentroids4326_layer,
                'INPUT_FIELD': 'Walk_ID',
                'TARGET': temp_layer,
                'TARGET_FIELD': 'Walk_ID',
                'MATRIX_TYPE': 0,  # Nur Entfernungen
                'NEAREST_POINTS': 0,
                'OUTPUT': "ogr:dbname='"
                + os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
                + "' table=polyCentroids4326_distances",
            })['OUTPUT']
            polyCentroids4326_distances_layer = QgsVectorLayer(polyCentroids4326_distances_layer_path, "polyCentroids4326_n500", "ogr")
            
            # IDs der 500 nächsten Features extrahieren
            distances = []
            for f in polyCentroids4326_distances_layer.getFeatures():
                distances.append((f['InputID'], f['Distance']))
            distances.sort(key=lambda x: x[1])
            nearest_ids = [fid for fid, _ in distances[:500]]
            
            

            # # Centroids extrahieren
            request = QgsFeatureRequest().setFilterFids(nearest_ids)
            selected_features = [f for f in polyCentroids4326_layer.getFeatures(request)]

            # Temporären Memory-Layer mit ausgewählten Features erstellen
            temp_layer = QgsVectorLayer(f"Point?crs={polyCentroids4326_layer.crs().authid()}", "temp_subset", "memory")
            temp_layer.dataProvider().addAttributes(polyCentroids4326_layer.fields())
            temp_layer.updateFields()
            temp_layer.dataProvider().addFeatures(selected_features)

            # Ausgabe-Pfad definieren
            output_path = os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
            output_layer_name = "polyCentroids4326_n500"
            
            # In GeoPackage speichern
            options = QgsVectorFileWriter.SaveVectorOptions()
            options.driverName = "GPKG"
            options.layerName = output_layer_name
            options.fileEncoding = "UTF-8"
            options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer  # Nur den Layer überschreiben, nicht die ganze Datei

            writeresult = QgsVectorFileWriter.writeAsVectorFormatV3(
                layer=temp_layer,
                fileName=output_path,
                transformContext=QgsProject.instance().transformContext(),
                options=options
            )

            print(f"Layer '{output_layer_name}' wurde erfolgreich gespeichert? -> {writeresult}")

            polyCentroids4326_layer_path = f"{output_path}|layername={output_layer_name}"
            polyCentroids4326_layer = QgsVectorLayer(
                polyCentroids4326_layer_path,
                "n500",
                "ogr"
            )
            polyCentroids4326_id = QgsDataSourceUri(polyCentroids4326_layer_path).uri()


            # # Gridcells extrahieren
            request = QgsFeatureRequest().setFilterFids(nearest_ids)
            selected_features = [f for f in grid_id_layer.getFeatures(request)]

            # Temporären Memory-Layer mit ausgewählten Features erstellen
            temp_layer = QgsVectorLayer(f"Polygon?crs={grid_id_layer.crs().authid()}", "temp_subset", "memory")
            temp_layer.dataProvider().addAttributes(grid_id_layer.fields())
            temp_layer.updateFields()
            temp_layer.dataProvider().addFeatures(selected_features)

            # Ausgabe-Pfad definieren
            output_path = os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
            output_layer_name = "Grid_500_n500"
            
            # In GeoPackage speichern
            options = QgsVectorFileWriter.SaveVectorOptions()
            options.driverName = "GPKG"
            options.layerName = output_layer_name
            options.fileEncoding = "UTF-8"
            options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer  # Nur den Layer überschreiben, nicht die ganze Datei

            writeresult = QgsVectorFileWriter.writeAsVectorFormatV3(
                layer=temp_layer,
                fileName=output_path,
                transformContext=QgsProject.instance().transformContext(),
                options=options
            )

            print(f"Layer '{output_layer_name}' wurde erfolgreich gespeichert? -> {writeresult}")

            # grid_id_layer_path = ("ogr:dbname='"
            # + os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
            # + f"' table={output_layer_name}")
            #grid_id_layer_path = f"{output_path}|layername={output_layer_name}"
            grid_id_layer_path = str(
                PurePath(
                    os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
                    + f"|layername={output_layer_name}"
                )
            )
            grid_id_layer = QgsVectorLayer(
                grid_id_layer_path,
                "n500",
                "ogr"
            )
            grid_id = QgsDataSourceUri(grid_id_layer_path).uri()
            self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")
            
            self.custom_feedback("DEBUGMODE! We have overwritten our grid_id-Variables (_,_layer,_layer_path) and polyCentroids4326 so that we continue with reduced dataset:")

        # Create Walking Area for the analysis
        self.custom_feedback("Creating maximum walking area for the analysis", "info")
        self.custom_feedback(str(self.ORS_PROVIDER), "debug")
        isochrones_max = processing.run(
            "ORS Tools:isochrones_from_layer",
            {
                "INPUT_PROVIDER": self.ORS_PROVIDER,
                "INPUT_PROFILE": 6,
                "INPUT_POINT_LAYER": polyCentroids4326_layer,
                "INPUT_FIELD": "Walk_ID",
                "INPUT_METRIC": 1,
                "INPUT_RANGES": str(self.MAX_DIST),
                "INPUT_SMOOTHING": None,
                "LOCATION_TYPE": 0,  # start - is that correct?
                "INPUT_AVOID_FEATURES": [],
                "INPUT_AVOID_BORDERS": None,
                "INPUT_AVOID_COUNTRIES": "",
                "INPUT_AVOID_POLYGONS": None,
                # "MAX_WORKERS": self.MAX_ORS_WORKERS,
                "OUTPUT": "ogr:dbname='"
                + os.path.join(self.OUTPUT_FOLDER, "WalkEU.gpkg")
                + "' table=walkarea",
            },
        )
        walkarea = isochrones_max['OUTPUT']
        self.custom_feedback(f"Wrote Isochrones (step1) to {walkarea}", "debug")

        self.custom_feedback(
            "get isochrones for reference geometries centers in 250m steps until 2000m ",
            "info",
        )
        isochrones_stacked = processing.run(
            "ORS Tools:isochrones_from_layer",
            {
                "INPUT_PROVIDER": self.ORS_PROVIDER,
                "INPUT_PROFILE": 6,
                "INPUT_POINT_LAYER": polyCentroids4326_layer,
                "INPUT_FIELD": "Walk_ID",
                "INPUT_METRIC": 1,
                "INPUT_RANGES": "250,500,750,1000,1250,1500,1750,2000",
                "INPUT_SMOOTHING": None,
                "LOCATION_TYPE": 0,  # start - is that correct?
                "INPUT_AVOID_FEATURES": [],
                "INPUT_AVOID_BORDERS": None,
                "INPUT_AVOID_COUNTRIES": "",
                "INPUT_AVOID_POLYGONS": None,
                # "MAX_WORKERS": self.MAX_ORS_WORKERS,
                "OUTPUT": "ogr:dbname='"
                + os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
                + "' table=walkareaAmenities_raw", # update=true
            },
        )
        walkareaAmenities_raw = isochrones_stacked['OUTPUT']
        try:
            excluded_feature_ids = isochrones_stacked['no_isochrones_for']
            self.custom_feedback(f"For these Features stacked isochrones couldn't be created: {excluded_feature_ids}", "warning")
        except Exception as e:
            pass
            
        self.custom_feedback(f"Wrote Isochrones (step2) to {walkareaAmenities_raw}", "debug")

        self.custom_feedback("Fixing geometries of those raw isochrones", "info")
        walkareaAmenities = processing.run(
            "native:fixgeometries",
            {
                "INPUT": walkareaAmenities_raw,
                "OUTPUT": "ogr:dbname='"
                + os.path.join(self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg")
                + "' table=walkarea_amenities",
            },
        )["OUTPUT"]
        self.custom_feedback(f"Fixed geometries and Saved to {walkareaAmenities}", "debug")

        def _amenities_download():  # formerly downloadOsm
            self.custom_feedback("Downloading OSM data from Overpass API", "info")
            self.custom_feedback(
                "Getting Extent - maximal walking distance (hardcoded 2000m)",
                "debug",
            )

            walkarea_amenities_layer, walkarea_amenities_fields = (
                _get_vectorlayer_and_fieldnames(
                    os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
                    + "|layername=walkarea_amenities",
                    "walkarea_amenities",
                )
            )
            walkarea_amenities_dp = walkarea_amenities_layer.dataProvider()
            self.custom_feedback(
                "dataProvider retrieval successful: "
                + str(walkarea_amenities_dp.extent()),
                "debug",
            )
            # walkarea_amenities_layer.updateExtents()
            qgis_extent = walkarea_amenities_dp.extent()

            overpass_extent = [ymin, xmin, ymax, xmax] = [
                round(qgis_extent.yMinimum(), 2),
                round(qgis_extent.xMinimum(), 2),
                round(qgis_extent.yMaximum(), 2),
                round(qgis_extent.xMaximum(), 2),
            ]
            extent = overpass_extent

            self.custom_feedback("creating QL for Overpass", "info")
            Supermarket = (
                'SU[out:xml][timeout:200];(node["shop"="supermarket"]('
                + str(extent)[1:-1]
                + ');way["shop"="supermarket"]('
                + str(extent)[1:-1]
                + ');relation["shop"="supermarket"]('
                + str(extent)[1:-1]
                + ");  ); (._;>;); out body;"
            )
            Education = (
                'EA[out:xml][timeout:200];(node["amenity"="childcare"]('
                + str(extent)[1:-1]
                + ');node["amenity"="university"]('
                + str(extent)[1:-1]
                + ');node["amenity"="college"]('
                + str(extent)[1:-1]
                + ');node["amenity"="prep_school"]('
                + str(extent)[1:-1]
                + ');way["amenity"="childcare"]('
                + str(extent)[1:-1]
                + ');way["amenity"="university"]('
                + str(extent)[1:-1]
                + ');way["amenity"="college"]('
                + str(extent)[1:-1]
                + ');way["amenity"="prep_school"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="childcare"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="university"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="college"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="prep_school"]('
                + str(extent)[1:-1]
                + ");  ); (._;>;); out body;"
            )
            Shopping = (
                'SH[out:xml] [timeout:300]; (node["shop"="clothes"]('
                + str(extent)[1:-1]
                + ');     node["shop"="boutique"]('
                + str(extent)[1:-1]
                + ');     node["shop"="fashion"]('
                + str(extent)[1:-1]
                + ');     node["shop"="shoes"]('
                + str(extent)[1:-1]
                + ');     node["shop"="electronics"]('
                + str(extent)[1:-1]
                + ');     node["shop"="hairdresser"]('
                + str(extent)[1:-1]
                + ');     node["shop"="shoe_repair"]('
                + str(extent)[1:-1]
                + ');     node["shop"="books"]('
                + str(extent)[1:-1]
                + ');     way["shop"="clothes"]('
                + str(extent)[1:-1]
                + ');     way["shop"="boutique"]('
                + str(extent)[1:-1]
                + ');     way["shop"="fashion"]('
                + str(extent)[1:-1]
                + ');     way["shop"="shoes"]('
                + str(extent)[1:-1]
                + ');     way["shop"="computer"]('
                + str(extent)[1:-1]
                + ');     way["shop"="electronics"]('
                + str(extent)[1:-1]
                + ');     way["shop"="hairdresser"]('
                + str(extent)[1:-1]
                + ');     way["shop"="shoe_repair"]('
                + str(extent)[1:-1]
                + ');     way["shop"="books"]('
                + str(extent)[1:-1]
                + ');     relation["shop"="clothes"]('
                + str(extent)[1:-1]
                + ');     relation["shop"="boutique"]('
                + str(extent)[1:-1]
                + ');     relation["shop"="fashion"]('
                + str(extent)[1:-1]
                + ');     relation["shop"="shoes"]('
                + str(extent)[1:-1]
                + ');     relation["shop"="electronics"]('
                + str(extent)[1:-1]
                + ');     relation["shop"="hairdresser"]('
                + str(extent)[1:-1]
                + ');     relation["shop"="shoe_repair"]('
                + str(extent)[1:-1]
                + ');     relation["shop"="books"]('
                + str(extent)[1:-1]
                + "); ); (._;>;); out body;"
            )
            OtherErrands = (
                'OE[out:xml][timeout:300];(node["healthcare"="hospital"]('
                + str(extent)[1:-1]
                + ');node["healthcare"="clinic"]('
                + str(extent)[1:-1]
                + ');node["amenity"="doctors"]('
                + str(extent)[1:-1]
                + ');node["healthcare"="doctor"]('
                + str(extent)[1:-1]
                + ');node["healthcare"="dentist"]('
                + str(extent)[1:-1]
                + ');node["healthcare:speciality"="orthodontics"]('
                + str(extent)[1:-1]
                + ');node["healthcare"="pharmacy"]('
                + str(extent)[1:-1]
                + ');node["healthcare"="physiotherapist"]('
                + str(extent)[1:-1]
                + ');node["amenity"="post_box"]('
                + str(extent)[1:-1]
                + ');node["amenity"="post_office"]('
                + str(extent)[1:-1]
                + ');node["amenity"="bank"]('
                + str(extent)[1:-1]
                + ');node["amenity"="atm"]('
                + str(extent)[1:-1]
                + ');node["amenity"="public_building"]('
                + str(extent)[1:-1]
                + ');node["amenity"="townhall"]('
                + str(extent)[1:-1]
                + ');node["amenity"="community_centre"]('
                + str(extent)[1:-1]
                + ');way["healthcare"="hospital"]('
                + str(extent)[1:-1]
                + ');way["healthcare"="clinic"]('
                + str(extent)[1:-1]
                + ');way["amenity"="doctors"]('
                + str(extent)[1:-1]
                + ');way["healthcare"="doctor"]('
                + str(extent)[1:-1]
                + ');way["healthcare"="dentist"]('
                + str(extent)[1:-1]
                + ');way["healthcare:speciality"="orthodontics"]('
                + str(extent)[1:-1]
                + ');way["healthcare"="pharmacy"]('
                + str(extent)[1:-1]
                + ');way["healthcare"="physiotherapist"]('
                + str(extent)[1:-1]
                + ');way["amenity"="post_box"]('
                + str(extent)[1:-1]
                + ');way["amenity"="post_office"]('
                + str(extent)[1:-1]
                + ');way["amenity"="bank"]('
                + str(extent)[1:-1]
                + ');way["amenity"="atm"]('
                + str(extent)[1:-1]
                + ');way["amenity"="public_building"]('
                + str(extent)[1:-1]
                + ');way["amenity"="townhall"]('
                + str(extent)[1:-1]
                + ');way["amenity"="community_centre"]('
                + str(extent)[1:-1]
                + ');relation["healthcare"="hospital"]('
                + str(extent)[1:-1]
                + ');relation["healthcare"="clinic"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="doctors"]('
                + str(extent)[1:-1]
                + ');relation["healthcare"="doctor"]('
                + str(extent)[1:-1]
                + ');relation["healthcare"="dentist"]('
                + str(extent)[1:-1]
                + ');relation["healthcare:speciality"="orthodontics"]('
                + str(extent)[1:-1]
                + ');relation["healthcare"="pharmacy"]('
                + str(extent)[1:-1]
                + ');relation["healthcare"="physiotherapist"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="post_box"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="post_office"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="bank"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="atm"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="public_building"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="townhall"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="community_centre"]('
                + str(extent)[1:-1]
                + "););(._;>;);out body;"
            )
            Leisure = (
                'LE[out:xml] [timeout:400]; (node["amenity"="cinema"]('
                + str(extent)[1:-1]
                + ');     node["tourism"="zoo"]('
                + str(extent)[1:-1]
                + ');     node["leisure"="swimming_pool"]('
                + str(extent)[1:-1]
                + ');     node["leisure"="swimming_area"]('
                + str(extent)[1:-1]
                + ');     node["leisure"="fitness_station"]('
                + str(extent)[1:-1]
                + ');     node["amenity"="public_bath"]('
                + str(extent)[1:-1]
                + ');     node["leisure"="dog_park"]('
                + str(extent)[1:-1]
                + ');     node["leisure"="playground"]('
                + str(extent)[1:-1]
                + ');     node["amenity"="nightclub"]('
                + str(extent)[1:-1]
                + ');     node["amenity"="restaurant"]('
                + str(extent)[1:-1]
                + ');     node["amenity"="bar"]('
                + str(extent)[1:-1]
                + ');     way["amenity"="cinema"]('
                + str(extent)[1:-1]
                + ');     way["tourism"="zoo"]('
                + str(extent)[1:-1]
                + ');     way["leisure"="swimming_pool"]('
                + str(extent)[1:-1]
                + ');     way["leisure"="swimming_area"]('
                + str(extent)[1:-1]
                + ');     way["leisure"="fitness_station"]('
                + str(extent)[1:-1]
                + ');     way["amenity"="public_bath"]('
                + str(extent)[1:-1]
                + ');     way["leisure"="dog_park"]('
                + str(extent)[1:-1]
                + ');     way["leisure"="playground"]('
                + str(extent)[1:-1]
                + ');     way["amenity"="nightclub"]('
                + str(extent)[1:-1]
                + ');     way["amenity"="restaurant"]('
                + str(extent)[1:-1]
                + ');     way["amenity"="bar"]('
                + str(extent)[1:-1]
                + ');     relation["amenity"="cinema"]('
                + str(extent)[1:-1]
                + ');     relation["tourism"="zoo"]('
                + str(extent)[1:-1]
                + ');     relation["leisure"="swimming_pool"]('
                + str(extent)[1:-1]
                + ');     relation["leisure"="swimming_area"]('
                + str(extent)[1:-1]
                + ');     relation["leisure"="fitness_station"]('
                + str(extent)[1:-1]
                + ');     relation["amenity"="public_bath"]('
                + str(extent)[1:-1]
                + ');     relation["leisure"="dog_park"]('
                + str(extent)[1:-1]
                + ');     relation["leisure"="playground"]('
                + str(extent)[1:-1]
                + ');     relation["amenity"="nightclub"]('
                + str(extent)[1:-1]
                + ');     relation["amenity"="restaurant"]('
                + str(extent)[1:-1]
                + ');     relation["amenity"="bar"]('
                + str(extent)[1:-1]
                + "); ); (._;>;); out body;"
            )
            School = (
                'SC[out:xml][timeout:200];(node["amenity"="school"]('
                + str(extent)[1:-1]
                + ');node["amenity"="kindergarten"]('
                + str(extent)[1:-1]
                + ');way["amenity"="school"]('
                + str(extent)[1:-1]
                + ');way["amenity"="kindergarten"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="kindergarten"]('
                + str(extent)[1:-1]
                + ');relation["amenity"="kindergarten"]('
                + str(extent)[1:-1]
                + ");  ); (._;>;); out body;"
            )
            checkDatasets = [
                self.SUPERMARKET_DATASET,
                self.EDUCATION_DATASET,
                self.SHOPPING_DATASET,
                self.ERRANDS_DATASET,
                self.LEISURE_DATASET,
            ]

            datasetFalse = [i for i, x in enumerate(checkDatasets) if x is False]
            datasetTrue = [i for i, x in enumerate(checkDatasets) if x is not False]

            datasetFalse.append(5)

            Amenity_Cat = [
                Supermarket,
                Education,
                Shopping,
                OtherErrands,
                Leisure,
                School,
            ]

            self.custom_feedback(
                "These are the ql-snippets for amenities: " + str(Amenity_Cat), "debug"
            )

            st = 6
            for val in datasetFalse:  # online downloads if no custom dataset in that variable is chosen. Wild programming! :D
                st += 4
                results = Amenity_Cat[val]
                self.custom_feedback(
                    f"Downloading data ({results[:2]}) from Overpass API",
                    "info",
                )
                processing.run(
                    "native:filedownloader",
                    {
                        "URL": self.OVERPASS_API_URL + results[2:],
                        "OUTPUT": self.OUTPUT_FOLDER
                        + r"//tempDir//"
                        + str(results[:2])
                        + ".geojson",
                    },
                )
                # time.sleep(5)

            if self.EDUCATION_DATASET is False:
                self.custom_feedback(
                    "Repair geometries of overpass response regarding schools",
                    "info",
                )
                school_repair = processing.run(
                    "native:fixgeometries",
                    {
                        "INPUT": self.OUTPUT_FOLDER
                        + r"//tempDir//SC.geojson|layername=multipolygons",
                        "OUTPUT": "ogr:dbname='"
                        + os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                        )
                        + "' table=SC_repaired",
                    },
                )["OUTPUT"]

                schools = processing.run(
                    "native:centroids",
                    {
                        "INPUT": school_repair,
                        "ALL_PARTS": False,
                        "OUTPUT": "ogr:dbname='"
                        + os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                        )
                        + "' table=tempSchool",
                    },
                )["OUTPUT"]

                self.custom_feedback(
                    "Merging school centroids with the other educational amenities so we don't count them twice",
                    "info",
                )
                ED = processing.run(
                    "native:mergevectorlayers",
                    {
                        "LAYERS": [
                            schools,
                            self.OUTPUT_FOLDER
                            + r"//tempDir//EA.geojson|layername=points",
                        ],
                        "CRS": None,
                        "OUTPUT": os.path.join(
                            self.OUTPUT_FOLDER, "tempDir", "ED.geojson"
                        ),
                    },
                )["OUTPUT"]
            else:
                pass

            Amenity_Cat = ["SU", "ED", "SH", "OE", "LE"]
            amenities_tables = []
            for fac in Amenity_Cat:

                self.custom_feedback(
                    "Counting points in polygon for {}".format(fac), "info"
                )
                
                # Convert GeoJSON points to GPKG
                try:
                    processing.run(
                        "gdal:convertformat",
                        {
                            "INPUT": os.path.join(
                                self.OUTPUT_FOLDER, r"tempDir", fac + ".geojson"
                            ) + "|layername=points",
                            "OPTIONS": "-update -nln " + fac + "_points",
                            "OUTPUT": os.path.join(
                                self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                            ),
                        },
                    )
                    self.custom_feedback(
                        f"Converted {fac} points to GPKG", "debug"
                    )
                    skippolygons = False
                except Exception as e:
                    try:
                        just_points_assumed = processing.run(
                            "gdal:convertformat",
                            {
                                "INPUT": os.path.join(
                                    self.OUTPUT_FOLDER, r"tempDir", fac + ".geojson"
                                ),
                                "OPTIONS": "-update -nln " + fac + "_points",
                                "OUTPUT": os.path.join(
                                    self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                                ),
                            },
                        )
                        points_for_counting = os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                        ) + "|layername=" + fac + "_points"
                    except Exception as e:
                        if "Unable to execute algorithm" in str(e):
                            self.custom_feedback(f"input geojson: {os.path.join(
                                    self.OUTPUT_FOLDER, r"tempDir", fac + ".geojson"
                                )}", "debug")
                            self.custom_feedback(f"output gpkg: {os.path.join(
                                    self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                                )}", "debug")
                            self.custom_feedback(f'output layername: {fac + "_points"}', "debug")
                            self.custom_feedback(f"return algorithm: {just_points_assumed}", "debug")
                            
                        self.custom_feedback(
                            f"No points layer found for {fac}: {str(e)}", "debug"
                        )
                        # Fall back to original points only
                        points_for_counting = os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir", fac + ".geojson"
                        ) + "|layername=points"
                        skippolygons = True
                    
                
                # Convert GeoJSON polygons to GPKG if they exist
                if not skippolygons:
                    try:
                        processing.run(
                            "gdal:convertformat",
                            {
                                "INPUT": os.path.join(
                                    self.OUTPUT_FOLDER, r"tempDir", fac + ".geojson"
                                ) + "|layername=multipolygons",
                                "OPTIONS": "-update -nln " + fac + "_polygons",
                                "OUTPUT": os.path.join(
                                    self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                                ),
                            },
                        )
                        self.custom_feedback(
                            f"Converted {fac} polygons to GPKG", "debug"
                        )
                        
                        # Convert polygons to points (pole of inaccessibility - guaranteed inside polygon)
                        polygon_points = processing.run(
                            "native:poleofinaccessibility",
                            {
                                "INPUT": os.path.join(
                                    self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                                ) + "|layername=" + fac + "_polygons",
                                "TOLERANCE": 1.0,
                                "OUTPUT": "ogr:dbname='"
                                + os.path.join(
                                    self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                                )
                                + "' table=" + fac + "_polygon_points",
                            },
                        )["OUTPUT"]
                        
                        self.custom_feedback(
                            f"Converted {fac} polygons to points inside polygons", "debug"
                        )
                        
                        # Merge original points with polygon-derived points
                        merged_points = processing.run(
                            "native:mergevectorlayers",
                            {
                                "LAYERS": [
                                    os.path.join(
                                        self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                                    ) + "|layername=" + fac + "_points",
                                    polygon_points
                                ],
                                "CRS": None,
                                "OUTPUT": "ogr:dbname='"
                                + os.path.join(
                                    self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                                )
                                + "' table=" + fac + "_all_points",
                            },
                        )["OUTPUT"]
                        
                        self.custom_feedback(
                            f"Merged {fac} points and polygon-derived points", "debug"
                        )
                        
                        # Use merged points for counting
                        points_for_counting = merged_points
                    except Exception as e:
                        self.custom_feedback(
                            f"No polygons layer found for {fac} or merge failed: {str(e)}", "debug"
                        )


                # Use the merged points (or fallback to original) for counting
                # If we have merged points, use them; otherwise fall back to original geojson
                if 'points_for_counting' in locals() and points_for_counting != os.path.join(
                    self.OUTPUT_FOLDER, r"tempDir", fac + ".geojson"
                ) + "|layername=points":
                    count_input = points_for_counting
                else:
                    count_input = os.path.join(
                        self.OUTPUT_FOLDER, r"tempDir", fac + ".geojson"
                    )
                self.custom_feedback(f"Count-Input: {count_input}", "debug")
                Fac1 = processing.run(
                    "native:countpointsinpolygon",
                    {
                        "POLYGONS": walkareaAmenities,
                        "POINTS": count_input,
                        "WEIGHT": "",
                        "CLASSFIELD": "",
                        "FIELD": fac + "_count",
                        "OUTPUT": os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir", fac + "_count.csv"
                        ),
                    },
                )["OUTPUT"]
                
                amenities_tables.append(Fac1)

            return amenities_tables

        def _ogr_path_modifier(grid_id_layer_path): 
            self.custom_feedback(f"Ask for modification of {grid_id_layer_path}")
            gpkg_path = grid_id_layer_path.rsplit('|',1)[0]
            layername = grid_id_layer_path.rsplit('|',1)[1].split("layername=",1)[1]
            if layername[0] in ('"',"'"):
                layername = layername[1:][:-1]
            ogr_layerpath = f"ogr:dbname='{gpkg_path}' table={layername}"
            self.custom_feedback(f"Modified to {ogr_layerpath}")
            return ogr_layerpath
            
        def _delete_from_geopackage(layername, gpkgpath):
            self.custom_feedback(
                f"Deleting layer {layername} from GeoPackage {gpkgpath} using sqlite3-hack",
                "debug",
            )
            try:
                conn = sqlite3.connect(str(PurePath(gpkgpath.strip())))
            except Exception as e:
                self.custom_feedback(
                    "Verbindung zum GeoPackage konnte nicht hergestellt werden."
                )
                raise e
            try:
                cursor = conn.cursor()
                try:
                    cursor.execute(f'DROP TABLE "{layername}"')
                except Exception as e:
                    if "no such table" in str(e):
                        self.custom_feedback(
                            "Tabelle "
                            + layername
                            + " existiert nicht, löschen nicht nötig."
                        )
                    conn.close()
                    return
                cursor.execute(
                    "DELETE FROM gpkg_contents WHERE table_name=?", (layername,)
                )
                cursor.execute(
                    "DELETE FROM gpkg_geometry_columns WHERE table_name=?", (layername,)
                )
                cursor.execute("DELETE FROM gpkg_extensions WHERE table_name=?", (layername,))
                cursor.execute("DELETE FROM gpkg_metadata_reference WHERE table_name=?", (layername,))
                try:
                    cursor.execute("DELETE FROM gpkg_geometry_columns_statistics WHERE table_name=?", (layername,))
                except:
                    pass
                try:
                    cursor.execute(f'DROP TABLE "rtree_{layername}_geom"')
                except Exception:
                    pass
                try:
                    cursor.execute(f'DROP TABLE "gpkg_{layername}_node"')
                    cursor.execute(f'DROP TABLE "gpkg_{layername}_rowid"')
                    cursor.execute(f'DROP TABLE "gpkg_{layername}_parent"')
                except Exception:
                    pass

                conn.commit()
                cursor.execute("VACUUM")
                conn.close()
            except Exception as e:
                self.custom_feedback(
                    "Verbindung zum GeoPackage konnte nicht hergestellt werden."
                )
                raise e

        def _include_in_grid_id(
            pathtodataset, 
            columns=[], 
            r_index="Walk_ID", 
            grid_id_layer=grid_id_layer, 
            grid_id_layer_path=grid_id_layer_path,
            grid_id=grid_id
        ):
        
            self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")
            if False and not self.DEBUG_MODE:
                grid_id_layer_path = (
                    "ogr:dbname='"
                    + os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
                    + "' table=grid500_id"
                )
                grid_id_defined_not_retrieved = str(
                    PurePath(
                        os.path.join(self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg")
                        + "|layername=grid500_id"
                    )
                )  # wild, why is this necessary????
                grid_id_layer = QgsVectorLayer(
                    grid_id_defined_not_retrieved, "grid500_id", "ogr"
                )
                if not grid_id_layer.isValid():
                    self.custom_feedback(
                        "Layer grid500_id konnte nicht geladen werden", "warning"
                    )
                    raise Exception("Layer grid500_id konnte nicht geladen werden")

            self.custom_feedback(
                "Layer grid500_id_old wird gelöscht (via sqlite)", "info"
            )
            _delete_from_geopackage("grid500_id_old", grid_id.rsplit("|", 1)[0])

            grid_id_old = processing.run(
                "gdal:convertformat",
                {
                    "INPUT": grid_id_layer,  # str(PureWindowsPath(grid_id)).replace("'", ""),
                    "CONVERT_ALL_LAYERS": False,
                    "OPTIONS": "-update -nln grid500_id_old",
                    "OUTPUT": os.path.join(
                        self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                    ),
                },
            )["OUTPUT"]
            if not grid_id_old:
                self.custom_feedback(
                    str(grid_id_old) + "|layername=grid500_id_old", "debug"
                )
                raise Exception(
                    "Could not convert the grid_id layer to a new name grid500_id_old."
                )

            self.custom_feedback(
                f"Die gewünschten Columns ({columns}) werden an grid_id angehängt", "info"
            )
            gpkg_path = grid_id_layer_path.rsplit("|", 1)[0]
            grid_id_layer_path = _ogr_path_modifier(grid_id_layer_path)

            self.custom_feedback(f"Layer {grid_id_layer_path.rsplit('=',1)[-1]} wird gelöscht (via sqlite)", "info")
            _delete_from_geopackage(grid_id_layer_path.rsplit('=',1)[-1], gpkg_path )

            grid_id_layer_path = processing.run(
                "native:joinattributestable",
                {
                    "INPUT": str(
                        PurePath(
                            os.path.join(
                                self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                            )
                        )
                    )
                    + "|layername=grid500_id_old",
                    "FIELD": "Walk_ID",
                    "INPUT_2": pathtodataset,
                    "FIELD_2": r_index,
                    "FIELDS_TO_COPY": columns,
                    "METHOD": 1,
                    "DISCARD_NONMATCHING": False,
                    "PREFIX": "",
                    "OUTPUT": grid_id_layer_path,
                },
            )["OUTPUT"]
            grid_id_layer = QgsVectorLayer(
                grid_id_layer_path, "grid500_id", "ogr"
            )
            grid_id = QgsDataSourceUri(grid_id_layer_path).uri()
            self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")

            # grid_id_local_layer = QgsVectorLayer(grid_id_local)
            # if grid_id_local_layer.isValid():
            #     self.custom_feedback(
            #         "current fields in grid_id:"
            #         + str(grid_id_local_layer.dataProvider().fields().names())
            #         + "["
            #         + str(grid_id_local)
            #         + "]",
            #         "info",
            #     )
            # else:
            #     raise Exception("grid_id_local is not valid: " + str(grid_id_local))
            return grid_id, grid_id_layer, grid_id_layer_path 
            #grid_id_local

        def _get_vectorlayer_and_fieldnames(pathtodataset, layername):
            self.custom_feedback("Processing layer {}".format(pathtodataset), "debug")
            if not isinstance(pathtodataset, QgsVectorLayer):
                layer = QgsVectorLayer(pathtodataset, layername, "ogr")
                pathtodataset = layer.dataProvider().dataSourceUri()
            else:
                layer = pathtodataset
            if not layer.isValid():
                self.custom_feedback("Layer konnte nicht geladen werden", "warning")
                raise Exception("Layer konnte nicht geladen werden")
            provider = layer.dataProvider()
            fields = provider.fields()
            fieldnames = [field.name() for field in fields]
            self.custom_feedback(
                "Fields in layer {}: {}".format(pathtodataset, fieldnames), "debug"
            )
            return layer, fieldnames

        def _df_to_memorylayer(df, layername):
            layer = QgsVectorLayer("None", layername, "memory")
            provider = layer.dataProvider()

            # 2. Attributfelder hinzufügen
            fields = []
            for col in df.columns:
                if col == "Walk_ID" or df[col].dtype == "int64":
                    fields.append(QgsField(col, QVariant.Int))
                elif df[col].dtype == "float64":
                    fields.append(QgsField(col, QVariant.Double))
                else:
                    fields.append(QgsField(col, QVariant.String))
            provider.addAttributes(fields)
            layer.updateFields()

            # 3. Features (Datenzeilen) hinzufügen
            features = []
            for _, row in df.iterrows():
                feature = QgsFeature()
                feature.setFields(layer.fields())
                for i, value in enumerate(row):
                    if i == 0:
                        feature.setAttribute(i, int(value))
                    else:
                        feature.setAttribute(i, value)
                features.append(feature)
                # self.custom_feedback(str(feature.attributeMap()), "debug")
            provider.addFeatures(features)
            layer.updateExtents()

            # for feature in features:
            #     self.custom_feedback(str(feature.attributes()), "debug")

            return layer

        def checkSlope(
            grid_id_layer_path=grid_id_layer_path, 
            grid_id_layer=grid_id_layer,
            grid_id=grid_id
        ):
            """This functions calculates slope in the walking area to reduce the radius. If the elevation has no input
            in the GUI, this function is skipped."""
            if self.ELEVATION_CHECKBOX is True:
                if not walkarea:  # but should be set in parent function  # noqa: F823
                    walkarea = (
                        "ogr:dbname='"
                        + os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg"
                        )
                        + "' table=walkarea"
                    )
                self.custom_feedback("Clipping elevation to walking area", "info")
                processing.run(
                    "gdal:cliprasterbymasklayer",
                    {
                        "INPUT": self.ELEVATION,
                        "MASK": walkarea,
                        "SOURCE_CRS": None,
                        "TARGET_CRS": None,
                        "NODATA": None,
                        "ALPHA_BAND": False,
                        "CROP_TO_CUTLINE": True,
                        "KEEP_RESOLUTION": False,
                        "SET_RESOLUTION": False,
                        "X_RESOLUTION": None,
                        "Y_RESOLUTION": None,
                        "MULTITHREADING": False,
                        "OPTIONS": "",
                        "DATA_TYPE": 0,
                        "EXTRA": "",
                        "OUTPUT": self.OUTPUT_FOLDER + r"/tempDir/elev_clip.tif",
                    },
                )
                self.custom_feedback("Calculating slope in the walking area", "info")
                processing.run(
                    "qgis:slope",
                    {
                        "INPUT": self.OUTPUT_FOLDER + r"/tempDir/elev_clip.tif",
                        "Z_FACTOR": 1,
                        "OUTPUT": self.OUTPUT_FOLDER + r"/tempDir/slope.tif",
                    },
                )
                self.custom_feedback(
                    "Calculating mean slope in the walking area", "info"
                )
                processing.run(
                    "qgis:zonalstatistics",
                    {
                        "INPUT_RASTER": self.OUTPUT_FOLDER + r"/tempDir/slope.tif",
                        "RASTER_BAND": 1,
                        "INPUT_VECTOR": walkarea,
                        "COLUMN_PREFIX": "slope_",
                        "STATISTICS": [2],
                        "OUTPUT": "ogr:dbname='"
                        + os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                        )
                        + "' table=slope_tempFile",
                    },
                )
                self.custom_feedback(
                    "Reducing the walking area based on slope", "info"
                )
                slope_tempFile1 = processing.run(
                    "native:joinattributestable",
                    {
                        "INPUT": grid_id_layer,
                        "FIELD": "Walk_ID",
                        "INPUT_2": walkarea,
                        "FIELD_2": "Walk_ID",
                        "FIELDS_TO_COPY": [],
                        "METHOD": 1,
                        "DISCARD_NONMATCHING": False,
                        "PREFIX": "",
                        "OUTPUT": "ogr:dbname='"
                        + os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                        )
                        + "' table=slope_tempFile update=true",
                    },
                )["OUTPUT"]

                slope_tempFile2 = processing.run(
                    "qgis:advancedpythonfieldcalculator",
                    {
                        "INPUT": slope_tempFile1,
                        "FIELD_NAME": "Slope_red",
                        "FIELD_TYPE": 1,
                        "FIELD_LENGTH": 10,
                        "FIELD_PRECISION": 3,
                        "GLOBAL": "def rec (input):\n if input <= 7:\n  return 0\n elif input >7 and input <=15:\n  return 2.5\n elif input >15 and input <=25:\n  return 5.0\n elif input >25 and input <=45:\n  return 7.5\n elif input >45:\n  return 10.0",
                        "FORMULA": "value = rec(<slope_mean>)",
                        "OUTPUT": "ogr:dbname='"
                        + os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                        )
                        + "' table=slope_tempFile2",
                    },
                )["OUTPUT"]

                return slope_tempFile2

            return None

        def calculateAmenities(
            grid_id_layer_path=grid_id_layer_path, 
            grid_id_layer=grid_id_layer,
            grid_id=grid_id
        ):
            """
            calculates the index value for the amenity indicator
            output is written to intermediate output file <Grid_Complete.gpkg>
            """
            save = []
            #featureCount = self.GRID_LAYER.featureCount()
            featureCount = grid_id_layer.featureCount()

            pd.options.mode.chained_assignment = None  # default='warn'

            # fetch data from overpass-api
            # URL = "https://overpass.kumi.systems/api/interpreter?data="
            # URL = 'https://z.overpass-api.de/api/interpreter?data='
            # URL = 'https://lz4.overpass-api.de/api/interpreter?data='
            URL = self.OVERPASS_API_URL

            self.custom_feedback(
                "Overpass-Info",
                "We are using " + URL + " as the Overpass API URL.",
            )
            # Download OSM from Overpass API
            # # downloadOsm()
            amenities_tables = _amenities_download()

            def calculate_distance_point(
                infra_name, 
                infra_count, 
                infra_weight
            ):
                self.custom_feedback(
                    "Calculating distance between points for {}".format(infra_name),
                    "info",
                )
                values = []

                ## disable this uglyness:
                # value = []
                # featureList = []

                # inputCSV = os.path.join(
                #     self.OUTPUT_FOLDER, r"tempDir", infra_name + "_count.csv"
                # )
                # df = pd.read_csv(inputCSV)

                # # all feature IDs
                # for i in range(1, featureCount+1):
                #     featureList.append(i)

                # #
                # count_row = int(df.shape[0])
                # for i in range(count_row):
                #     value.append((df["Walk_ID"][i]))

                # id_list = []
                # [id_list.append(x) for x in value if x not in id_list]
                # insertPos = set(featureList) - set(id_list)

                ## do this instead in pandas
                inputCSV = os.path.join(
                    self.OUTPUT_FOLDER, r"tempDir", infra_name + "_count.csv"
                )
                df = pd.read_csv(inputCSV)
                # self.custom_feedback(df.to_string(), "debug")

                # All feature IDs
                featureList = list(range(1, featureCount + 1))

                # Unique Walk_IDs from the CSV
                id_list = df["Walk_ID"].unique().tolist() #this keeps order! (in contrast to use of set())

                # Calculate missing feature IDs
                # insertPos = set(featureList) - set(id_list)

                self.custom_feedback("ids:" + str(id_list), "debug")
                # self.custom_feedback("insertPos:" + str(insertPos), "debug")

                for n, i in enumerate(id_list):
                    if n%1000==0:
                        self.custom_feedback(f"Done {n} of {len(id_list)}")
                    current_infra_count = infra_count
                    # val = infra_weight

                    filtered_df = df[df["Walk_ID"] == i]
                    # self.custom_feedback(filtered_df.to_string(), "debug")

                    filtered_df = filtered_df.join(
                        pd.Series(
                            [
                                (filtered_df.iloc[0, -1] - (filtered_df.iloc[1, -1])),
                                (filtered_df.iloc[1, -1] - (filtered_df.iloc[2, -1])),
                                (filtered_df.iloc[2, -1] - (filtered_df.iloc[3, -1])),
                                (filtered_df.iloc[3, -1] - (filtered_df.iloc[4, -1])),
                                (filtered_df.iloc[4, -1] - (filtered_df.iloc[5, -1])),
                                (filtered_df.iloc[5, -1] - (filtered_df.iloc[6, -1])),
                                (filtered_df.iloc[6, -1] - (filtered_df.iloc[7, -1])),
                                (filtered_df.iloc[7, -1]),
                            ],
                            index=filtered_df.index,
                            name="count_in_holed_isochrone",
                        )
                    )
                    # self.custom_feedback(filtered_df.to_string(), "debug")
                    filtered_df = filtered_df.drop(
                        columns=[
                            "CENTER_LON",
                            "CENTER_LAT",
                            "AA_MODE",
                            "TOTAL_POP",
                            infra_name + "_count",
                        ]
                    )  # cols are now: "Walk_ID", "AA_METERS", "count_in_holed_isochrone"
                    # self.custom_feedback(filtered_df.to_string(), "debug")

                    filtered_df.loc[
                        filtered_df["count_in_holed_isochrone"] > 2,
                        "count_in_holed_isochrone",
                    ] = 3
                    filtered_df["count_in_holed_isochrone"] = filtered_df[
                        "count_in_holed_isochrone"
                    ].fillna(0)

                    # filtered_df["count_category"] = filtered_df.loc[:, "AA_METERS"].apply(check)
                    filtered_df = filtered_df.assign(
                        AA_METERS_STR=filtered_df["AA_METERS"].apply(lambda x: str(x))
                    )

                    # filtered_df["count_category"]
                    abschmelzer_series = pd.Series(
                        {
                            "250": 1,
                            "500": 0.9,
                            "750": 0.7,
                            "1000": 0.4,
                            "1250": 0.2,
                            "1500": 0.05,
                            "1750": 0.025,
                            "2000": 0.005,
                        },
                        name="weighing",
                    )

                    filtered_df = filtered_df.join(
                        abschmelzer_series, on="AA_METERS_STR"
                    )
                    filtered_df = filtered_df.drop(columns=["AA_METERS_STR"])
                    # self.custom_feedback(filtered_df.to_string(), "debug")

                    # and now itereate from 250 to 2000!
                    for i, row in filtered_df.sort_values(
                        by="AA_METERS", ascending=True
                    ).iterrows():
                        # self.custom_feedback(
                        #     f"current_infra_count: {str(current_infra_count)} - {str(row['AA_METERS'])} - {str(i)}",
                        #     "debug",
                        # )
                        current_infra_count = current_infra_count - row["count_in_holed_isochrone"]
                        # self.custom_feedback(
                        #     f"current_infra_count - after: {str(current_infra_count)} - {str(row['AA_METERS'])}",
                        #     "debug",
                        # )
                        if (current_infra_count < 1 
                            #and current_infra_count >= 0.0 #bullshit! ??
                        ):
                            filtered_df.loc[
                                filtered_df.index == i,
                                "distance_point",
                            ] = (row["count_in_holed_isochrone"] + current_infra_count) * row[
                                "weighing"
                            ]
                            break
                            # continue  # TODO: is this correct? - nope, wrong.
                        # elif current_infra_count < 0:  # is this correct and neccassary? #nope, wrong!
                        #     filtered_df.loc[
                        #         filtered_df.index == i,
                        #         "distance_point",
                        #     ] = 0.0
                        #     break
                        #     # continue  # TODO: is this correct? - nope, wrong.
                        else:
                            filtered_df.loc[
                                filtered_df.index == i,
                                "distance_point",
                            ] = row["count_in_holed_isochrone"] * row["weighing"]
                    # self.custom_feedback(filtered_df.to_string(), "debug")

                    filtered_df["distance_point"] = filtered_df[
                        "distance_point"
                    ].fillna(0)

                    filtered_df["current_infra_count_cleaned"] = (
                        (filtered_df["distance_point"].apply(lambda x: round(x, 1))) / (filtered_df["weighing"].apply(lambda x: round(x, 1)))
                    ).apply(lambda x: x if not np.isinf(x) else np.nan).apply(lambda x: int(x) if not np.isnan(x) else x)
                    if True:
                        # self.custom_feedback("rounded distance_point:", "debug")
                        # self.custom_feedback(
                        #     filtered_df["distance_point"]
                        #     .apply(lambda x: int(round(x, 1)))
                        #     .to_string(),
                        #     "debug",
                        # )
                        # self.custom_feedback("rounded weighing:", "debug")
                        # self.custom_feedback(
                        #     (
                        #         filtered_df["weighing"].apply(lambda x: round(x, 1))
                        #     ).to_string(),
                        #     "debug",
                        # )
                        # self.custom_feedback("current_infra_count_cleaned calculated:", "debug")
                        #self.custom_feedback(filtered_df.to_string(), "debug")
                        pass
                        
                    filtered_df["current_infra_count_cleaned"] = filtered_df[
                        "current_infra_count_cleaned"
                    ].fillna(0)

                    # self.custom_feedback(filtered_df.to_string(), "debug")

                    # def check(x):
                    #     a = 3 if x > 2 else 2 if x == 2 else 1 if x == 1 else 0
                    #     return a

                    # dist250 = check(filtered_df.iloc[7, 2])
                    # dist500 = check(filtered_df.iloc[6, 2])
                    # dist750 = check(filtered_df.iloc[5, 2])
                    # dist1000 = check(filtered_df.iloc[4, 2])
                    # dist1250 = check(filtered_df.iloc[3, 2])
                    # dist1500 = check(filtered_df.iloc[2, 2])
                    # dist1750 = check(filtered_df.iloc[1, 2])
                    # dist2000 = check(filtered_df.iloc[0, 2])

                    # liste = [
                    #     dist250,
                    #     dist500,
                    #     dist750,
                    #     dist1000,
                    #     dist1250,
                    #     dist1500,
                    #     dist1750,
                    #     dist2000,
                    # ]

                    # empty_list = []
                    # for l_no, item in enumerate(liste):
                    #     dist_values = [1, 0.9, 0.7, 0.4, 0.2, 0.05, 0.025, 0.005]
                    #     infra_count = infra_count - item
                    #     if infra_count < 1:
                    #         empty_list.append((item + infra_count) * dist_values[l_no])
                    #         break
                    #     else:
                    #         empty_list.append(item * dist_values[l_no])

                    # infra_count_list = []

                    # for i in range(8):
                    #     try:
                    #         infra_count_list.append(
                    #             int(round(empty_list[i], 1) / round(dist_values[i], 1))
                    #         )
                    #     except:  # noqa: E722
                    #         infra_count_list.append(0)

                    rawPos = []
                    for i, value in filtered_df["current_infra_count_cleaned"].items():
                        if value == 3:
                            rawPos.append(i)
                            rawPos.append(i)
                            rawPos.append(i)
                        elif value == 2:
                            rawPos.append(i)
                            rawPos.append(i)
                        elif value == 1:
                            rawPos.append(i)
                    # self.custom_feedback(f"rawPos: {str(rawPos)}", "debug")
                    # self.custom_feedback(f"infra_weight: {str(infra_weight)}", "debug")
                    weight = 0
                    filtered_df = filtered_df.assign(walk_points=np.nan)
                    for pos in rawPos:
                        # self.custom_feedback(f"weight: {str(weight)}", "debug")
                        filtered_df.loc[filtered_df.index == pos, "walk_points"] = (
                            filtered_df.loc[
                                filtered_df.index == pos, "current_infra_count_cleaned"
                            ]
                            * infra_weight[weight]
                        )
                        weight += 1

                    # self.custom_feedback(filtered_df.to_string(), "debug")
                    values.append(filtered_df["walk_points"].sum())
                    # rawPos = []
                    # for i in range(8):
                    #     if infra_count_list[i] == 3.0:
                    #         rawPos.append(i + 1)
                    #         rawPos.append(i + 1)
                    #         rawPos.append(i + 1)
                    #     elif infra_count_list[i] == 2.0:
                    #         rawPos.append(i + 1)
                    #         rawPos.append(i + 1)
                    #     elif infra_count_list[i] == 1.0:
                    #         rawPos.append(i + 1)
                    # L = []
                    # weight = 0

                    # for pos in rawPos:
                    #     L.append(float(dist_values[int(pos - 1)]) * infra_weight[weight])
                    #     weight += 1

                    # values.append(round(sum(L), 1))
                
                # IDs der Grid-Zellen holen
                walk_ids = [f['Walk_ID'] for f in grid_id_layer.getFeatures()]
                # Dictionary: Walk_ID -> Wert
                values_dict = dict(zip(id_list, values))
                # Werte-Liste in der Reihenfolge der Grid-IDs, fehlende als 0
                values = [values_dict.get(wid, 0) for wid in walk_ids]

                # for nullPosition in insertPos:
                #     values.insert(int(nullPosition - 1), 0)
                self.custom_feedback(
                    f"distance_points calculated for {infra_name}:" + str(values),
                    "debug",
                )
                return values

            SU_point = calculate_distance_point(
                "SU",
                self.SUPERMARKET_COUNT,
                [
                    self.AMENITY_SUPERMARKET_WEIGHT_1,
                    self.AMENITY_SUPERMARKET_WEIGHT_2,
                    self.AMENITY_SUPERMARKET_WEIGHT_3,
                ],
            )
            ED_point = calculate_distance_point(
                "ED",
                self.EDUCATION_COUNT,
                [
                    self.AMENITY_EDUCATION_WEIGHT_1,
                    self.AMENITY_EDUCATION_WEIGHT_2,
                    self.AMENITY_EDUCATION_WEIGHT_3,
                ],
            )
            SH_point = calculate_distance_point(
                "SH",
                self.SHOPPING_COUNT,
                [
                    self.AMENITY_SHOPPING_WEIGHT_1,
                    self.AMENITY_SHOPPING_WEIGHT_2,
                    self.AMENITY_SHOPPING_WEIGHT_3,
                ],
            )
            OE_point = calculate_distance_point(
                "OE",
                self.OTHER_ERRANDS_COUNT,
                [
                    self.AMENITY_OTHER_ERRANDS_WEIGHT_1,
                    self.AMENITY_OTHER_ERRANDS_WEIGHT_2,
                    self.AMENITY_OTHER_ERRANDS_WEIGHT_3,
                ],
            )
            LE_point = calculate_distance_point(
                "LE",
                self.LEISURE_COUNT,
                [
                    self.AMENITY_LEISURE_WEIGHT_1,
                    self.AMENITY_LEISURE_WEIGHT_2,
                    self.AMENITY_LEISURE_WEIGHT_3,
                ],
            )
            sum_list = list(zip(SU_point, ED_point, SH_point, OE_point, LE_point))
            # self.custom_feedback(f"sumlist: {sum_list}", "debug")
            maxVal = []

            def checkMax(Val1, Val2):
                if Val1 == 3:
                    maxVal.append(Val2[0] + Val2[1] + Val2[2])
                elif Val1 == 2:
                    maxVal.append(Val2[0] + Val2[1])
                elif Val1 == 1:
                    maxVal.append(Val2[0])

            checkMax(
                self.SUPERMARKET_COUNT,
                [
                    self.AMENITY_SUPERMARKET_WEIGHT_1,
                    self.AMENITY_SUPERMARKET_WEIGHT_2,
                    self.AMENITY_SUPERMARKET_WEIGHT_3,
                ],
            )
            checkMax(
                self.EDUCATION_COUNT,
                [
                    self.AMENITY_EDUCATION_WEIGHT_1,
                    self.AMENITY_EDUCATION_WEIGHT_2,
                    self.AMENITY_EDUCATION_WEIGHT_3,
                ],
            )
            checkMax(
                self.SHOPPING_COUNT,
                [
                    self.AMENITY_SHOPPING_WEIGHT_1,
                    self.AMENITY_SHOPPING_WEIGHT_2,
                    self.AMENITY_SHOPPING_WEIGHT_3,
                ],
            )
            checkMax(
                self.OTHER_ERRANDS_COUNT,
                [
                    self.AMENITY_OTHER_ERRANDS_WEIGHT_1,
                    self.AMENITY_OTHER_ERRANDS_WEIGHT_2,
                    self.AMENITY_OTHER_ERRANDS_WEIGHT_3,
                ],
            )
            checkMax(
                self.LEISURE_COUNT,
                [
                    self.AMENITY_LEISURE_WEIGHT_1,
                    self.AMENITY_LEISURE_WEIGHT_2,
                    self.AMENITY_LEISURE_WEIGHT_3,
                ],
            )

            weighting = 100 / sum(maxVal)
            amenity_sum = [
                (a + b + c + d + e) * weighting for (a, b, c, d, e) in sum_list
            ]

            def recalculateAmenities(oldval):
                if oldval >= 90:
                    return 10
                elif oldval >= 80 and oldval < 90:
                    return 9
                elif oldval >= 70 and oldval < 80:
                    return 8
                elif oldval >= 60 and oldval < 70:
                    return 7
                elif oldval >= 50 and oldval < 60:
                    return 6
                elif oldval >= 40 and oldval < 50:
                    return 5
                elif oldval >= 30 and oldval < 40:
                    return 4
                elif oldval >= 20 and oldval < 30:
                    return 3
                elif oldval >= 10 and oldval < 20:
                    return 2
                elif oldval >= 1 and oldval < 10:
                    return 1
                elif oldval == 0:
                    return 0

            amenity_rec = []
            for item in amenity_sum:
                amenity_rec.append(recalculateAmenities(item))

            #featureCount = self.GRID_LAYER.featureCount()
            featureCount = grid_id_layer.featureCount()
            
            try:
                df_amenitypoints = pd.DataFrame().from_dict(
                    {
                        # "Walk_ID": list(range(1, featureCount + 1)),
                        "Walk_ID": [f['Walk_ID'] for f in grid_id_layer.getFeatures()],
                        "amenity_rec": amenity_rec,
                        "amenity_sum": amenity_sum,
                    }
                )
            except ValueError:
                self.custom_feedback(dedent(f"""
                An error occured, possible due to no ammenities in the area.
                {maxVal}
                {weighting}
                {[f['Walk_ID'] for f in grid_id_layer.getFeatures()]} - {len([f['Walk_ID'] for f in grid_id_layer.getFeatures()])}
                {amenity_rec} - {len(amenity_rec)}
                {amenity_sum} - {len(amenity_sum)}
                """), "warning")

            # self.custom_feedback(df_amenitypoints.to_string(), "debug")
            amenity_points = _df_to_memorylayer(df_amenitypoints, "amenity_points")

            self.custom_feedback(
                "Convert amenity_points (memory-layer) to geopackage", "debug"
            )
            result = processing.run(
                "gdal:convertformat",
                {
                    "INPUT": amenity_points,
                    "OPTIONS": "-update -nln amenity_points",
                    "OUTPUT": os.path.join(
                        self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                    ),
                },
            )
            if not result:
                self.custom_feedback(
                    "Could not convert the amenity_points layer to a new name amenity_points."
                )
                raise Exception(
                    "Could not convert the amenity_points layer to a new name amenity_points."
                )

            return amenity_points

        def calculatePedshed( 
            grid_id_layer_path=grid_id_layer_path, 
            grid_id_layer=grid_id_layer,
            grid_id=grid_id
        ):
            """function to calculate the pedestrian shed."""

            radius = self.MAX_DIST
            radius2 = int(radius) * int(radius)

            self.custom_feedback(
                "Calculating pedestrian shed, first fieldcalculator, then join it to original.",
                "info",
            )
            pedshed_calculated = processing.run(
                "qgis:fieldcalculator",
                {
                    "INPUT": walkarea,
                    "FIELD_NAME": "PedShed",
                    "FIELD_TYPE": 0,
                    "FIELD_LENGTH": 10,
                    "FIELD_PRECISION": 3,
                    "NEW_FIELD": True,
                    "FORMULA": "$area/(pi()*" + str(radius2) + ")*100",
                    "OUTPUT": "ogr:dbname='"
                    + os.path.join(self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg")
                    + "' table=pedshed_calculated",
                },
            )["OUTPUT"]

            pedshed_on_gridid = processing.run(
                "native:joinattributestable",
                {
                    "INPUT": grid_id_layer,
                    "FIELD": "Walk_ID",
                    "INPUT_2": pedshed_calculated,
                    "FIELD_2": "Walk_ID",
                    "FIELDS_TO_COPY": ["PedShed"],
                    "METHOD": 1,
                    "DISCARD_NONMATCHING": False,
                    "PREFIX": "",
                    "OUTPUT": "ogr:dbname='"
                    + os.path.join(self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg")
                    + "' table=pedshed_on_gridid",
                },
            )["OUTPUT"]

            self.custom_feedback(
                "classifying pedshed via advancedpythonfieldcalculator (replacing geojson operation)",
                "debug",
            )
            pedshed_file_pointed = processing.run(
                "qgis:advancedpythonfieldcalculator",
                {
                    "INPUT": pedshed_on_gridid,
                    "FIELD_NAME": "PedShed_point",
                    "FIELD_TYPE": 1,
                    "FIELD_LENGTH": 10,
                    "FIELD_PRECISION": 3,
                    "GLOBAL": dedent("""
                        def recalculatePedShed(oldval):
                            try:
                                if oldval >= 90:
                                    return 10
                                elif oldval >= 80 and oldval < 90:
                                    return 9
                                elif oldval >= 70 and oldval < 80:
                                    return 8
                                elif oldval >= 60 and oldval < 70:
                                    return 7
                                elif oldval >= 50 and oldval < 60:
                                    return 6
                                elif oldval >= 40 and oldval < 50:
                                    return 5
                                elif oldval >= 30 and oldval < 40:
                                    return 4
                                elif oldval >= 20 and oldval < 30:
                                    return 3
                                elif oldval >= 10 and oldval < 20:
                                    return 2
                                elif oldval < 10:
                                    return 1
                            except:  # noqa: E722
                                return 0
                    """),
                    "FORMULA": "value = recalculatePedShed(<PedShed>)",
                    "OUTPUT": "ogr:dbname='"
                    + os.path.join(self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg")
                    + "' table=pedshed_pointed",
                },
            )["OUTPUT"]

            return pedshed_file_pointed

        def calcGreen( 
            grid_id_layer_path=grid_id_layer_path, 
            grid_id_layer=grid_id_layer,
            grid_id=grid_id
        ):
            """Function to calculate the percentage of green area in the walking radius"""

            self.custom_feedback(
                "Calculating green area in the walking radius", "info"
            )
            if self.GREEN_CHECKBOX is False:
                walkarea_layer, walkarea_fieldnames = _get_vectorlayer_and_fieldnames(
                    walkarea, "walkarea"
                )
                # walkarea_layer.updateExtents()
                qgis_extent = walkarea_layer.dataProvider().extent()

                overpass_extent = [ymin, xmin, ymax, xmax] = [
                    round(qgis_extent.yMinimum(), 2),
                    round(qgis_extent.xMinimum(), 2),
                    round(qgis_extent.yMaximum(), 2),
                    round(qgis_extent.xMaximum(), 2),
                ]
                extent = overpass_extent

                URL = (
                    self.OVERPASS_API_URL
                )  # "https://overpass.kumi.systems/api/interpreter?data="
                Green = (
                    'GR[out:xml] [timeout:200]; ( node["landuse"="meadow"]('
                    + str(extent)[1:-1]
                    + '); node["landuse"="orchard"]('
                    + str(extent)[1:-1]
                    + '); node["landuse"="allotments"]('
                    + str(extent)[1:-1]
                    + '); node["landuse"="flowerbed"]('
                    + str(extent)[1:-1]
                    + '); node["leisure"="garden"]('
                    + str(extent)[1:-1]
                    + '); node["landuse"="grass"]('
                    + str(extent)[1:-1]
                    + '); node["landuse"="village_green"]('
                    + str(extent)[1:-1]
                    + '); node["leisure"="park"]('
                    + str(extent)[1:-1]
                    + '); node["landuse"="recreation_ground"]('
                    + str(extent)[1:-1]
                    + '); node["amenity"="grave_yard"]('
                    + str(extent)[1:-1]
                    + '); node["landuse"="cemetery"]('
                    + str(extent)[1:-1]
                    + '); node["cemetery"="sector"]('
                    + str(extent)[1:-1]
                    + '); node["landuse"="greenfield"]('
                    + str(extent)[1:-1]
                    + '); way["landuse"="meadow"]('
                    + str(extent)[1:-1]
                    + '); way["landuse"="orchard"]('
                    + str(extent)[1:-1]
                    + '); way["landuse"="allotments"]('
                    + str(extent)[1:-1]
                    + '); way["landuse"="flowerbed"]('
                    + str(extent)[1:-1]
                    + '); way["leisure"="garden"]('
                    + str(extent)[1:-1]
                    + '); way["landuse"="grass"]('
                    + str(extent)[1:-1]
                    + '); way["landuse"="village_green"]('
                    + str(extent)[1:-1]
                    + '); way["leisure"="park"]('
                    + str(extent)[1:-1]
                    + '); way["landuse"="recreation_ground"]('
                    + str(extent)[1:-1]
                    + '); way["amenity"="grave_yard"]('
                    + str(extent)[1:-1]
                    + '); way["landuse"="cemetery"]('
                    + str(extent)[1:-1]
                    + '); way["cemetery"="sector"]('
                    + str(extent)[1:-1]
                    + '); way["landuse"="greenfield"]('
                    + str(extent)[1:-1]
                    + '); relation["landuse"="meadow"]('
                    + str(extent)[1:-1]
                    + '); relation["landuse"="orchard"]('
                    + str(extent)[1:-1]
                    + '); relation["landuse"="allotments"]('
                    + str(extent)[1:-1]
                    + '); relation["landuse"="flowerbed"]('
                    + str(extent)[1:-1]
                    + '); relation["leisure"="garden"]('
                    + str(extent)[1:-1]
                    + '); relation["landuse"="grass"]('
                    + str(extent)[1:-1]
                    + '); relation["landuse"="village_green"]('
                    + str(extent)[1:-1]
                    + '); relation["leisure"="park"]('
                    + str(extent)[1:-1]
                    + '); relation["landuse"="recreation_ground"]('
                    + str(extent)[1:-1]
                    + '); relation["amenity"="grave_yard"]('
                    + str(extent)[1:-1]
                    + '); relation["landuse"="cemetery"]('
                    + str(extent)[1:-1]
                    + '); relation["cemetery"="sector"]('
                    + str(extent)[1:-1]
                    + '); relation["landuse"="greenfield"]('
                    + str(extent)[1:-1]
                    + "); ); (._;>;); out body;"
                )
                self.custom_feedback(
                    "Downloading green area data from Overpass API", "info"
                )
                greenArea_raw = processing.run(
                    "native:filedownloader",
                    {
                        "URL": URL + Green[2:],
                        "OUTPUT": os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir/greenarea_raw.xml"
                        ),
                    },
                )["OUTPUT"]
                greenArea_layer = QgsVectorLayer(
                    os.path.join(
                        self.OUTPUT_FOLDER, r"tempDir/greenarea_raw.xml"
                    ) + "|layername=multipolygons", 
                    "greenarea_raw", 
                    "ogr"
                )
                self.custom_feedback(
                    "Repair geometries of overpass response regarding green areas",
                    "info",
                )
                green_fixed = processing.run(
                    "native:fixgeometries",
                    {
                        "INPUT": greenArea_layer, #greenArea_raw + "|layername=multipolygons",
                        "OUTPUT": "ogr:dbname='"
                        + os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg"
                        )
                        + "' table=greenarea",
                    },
                )["OUTPUT"]
            else:
                self.custom_feedback(
                    "Repair geometries of custom green area dataset", "info"
                )
                green_fixed = processing.run(
                    "native:fixgeometries",
                    {
                        "INPUT": self.GREEN_LAYER,
                        "OUTPUT": "ogr:dbname='"
                        + os.path.join(
                            self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg"
                        )
                        + "' table=greenarea",
                    },
                )["OUTPUT"]

            self.custom_feedback(
                "Calculate the overlap of green areas and walking radius", "info"
            )
            green_overlaps = processing.run(
                "native:calculatevectoroverlaps",
                {
                    "INPUT": walkarea,
                    "LAYERS": green_fixed,
                    "OUTPUT": "ogr:dbname='"
                    + os.path.join(self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg")
                    + "' table=green_overlaps",
                },
            )["OUTPUT"]

            green_overlaps_layer, green_overlaps_fields = (
                _get_vectorlayer_and_fieldnames(green_overlaps, "green_overlaps")
            )
            green_overlaps_dp = green_overlaps_layer.dataProvider()
            renamed = green_overlaps_dp.renameAttributes(
                renamedAttributes={
                    green_overlaps_dp.fieldNameIndex(
                        "WalkEU_process_area"
                    ): "green_area",
                    green_overlaps_dp.fieldNameIndex("WalkEU_process_pc"): "green_pc",
                }
            )
            if not renamed:
                raise Exception("green_pc couldn't properly renamed")

            self.custom_feedback(
                "Join the green area values to the reference geometries", "info"
            )
            green_score = processing.run(
                "native:joinattributestable",
                {
                    "INPUT": grid_id_layer,
                    "FIELD": "Walk_ID",
                    "INPUT_2": green_overlaps,
                    "FIELD_2": "Walk_ID",
                    "FIELDS_TO_COPY": ["green_pc"],
                    "METHOD": 1,
                    "DISCARD_NONMATCHING": False,
                    "PREFIX": "",
                    "OUTPUT": "ogr:dbname='"
                    + os.path.join(self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg")
                    + "' table=green_score",
                },
            )["OUTPUT"]
            self.custom_feedback(green_score)

            self.custom_feedback(
                "classifying greenshed via advancedpythonfieldcalculator (replacing geojson operation)",
                "debug",
            )
            greenshed_pointed = processing.run(
                "qgis:advancedpythonfieldcalculator",
                {
                    "INPUT": green_score,
                    "FIELD_NAME": "green_point",
                    "FIELD_TYPE": 1,
                    "FIELD_LENGTH": 10,
                    "FIELD_PRECISION": 3,
                    "GLOBAL": dedent("""
                        def recalculateGreen(oldval):
                            try:
                                if oldval >= 22.5:
                                    return 10
                                elif oldval >= 20 and oldval < 22.5:
                                    return 9
                                elif oldval >= 17.5 and oldval < 20.0:
                                    return 8
                                elif oldval >= 15 and oldval < 17.5:
                                    return 7
                                elif oldval >= 12.5 and oldval < 15:
                                    return 6
                                elif oldval >= 10.0 and oldval < 12.5:
                                    return 5
                                elif oldval >= 7.5 and oldval < 10:
                                    return 4
                                elif oldval >= 5 and oldval < 7.5:
                                    return 3
                                elif oldval >= 2.5 and oldval < 5.0:
                                    return 2
                                elif oldval < 2.5:
                                    return 1
                            except:  # noqa: E722
                                return 0
                    """),
                    "FORMULA": "value = recalculateGreen(<green_pc>)",
                    "OUTPUT": "ogr:dbname='"
                    + os.path.join(self.OUTPUT_FOLDER, r"tempDir/WalkEU_process.gpkg")
                    + "' table=greenshed_pointed",
                },
            )["OUTPUT"]

            return greenshed_pointed

        def calc( 
            grid_id_layer_path=grid_id_layer_path, 
            grid_id_layer=grid_id_layer,
            grid_id=grid_id
        ):
            """Calculate the OS-Walk-EU Index from the indicators. Multiplying the summed values with correction factor set the value range 0 - 100"""
            AMENITIES_WEIGHT = self.AMENITIES_WEIGHT
            PEDESTRIAN_SHED_WEIGHT = self.PEDESTRIAN_SHED_WEIGHT
            GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT = (
                self.GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT
            )
            cor_fac = 100 / (
                (
                    PEDESTRIAN_SHED_WEIGHT
                    + GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT
                    + AMENITIES_WEIGHT
                )
                * 10
            )
            self.custom_feedback("Calculate the OS-Walk-EU Index", "info")
            
            self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")
            try:
                
                # globalcode = dedent(f"""
                #     AMENITIES_WEIGHT = {self.AMENITIES_WEIGHT}
                #     PEDESTRIAN_SHED_WEIGHT = {self.PEDESTRIAN_SHED_WEIGHT}
                #     GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT = {self.GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT}
                #     cor_fac = {cor_fac}
                #     """)
                #     # def safe(val):
                #     #     try:
                #     #         if val is None:
                #     #             return 0
                #     #         elif str(type(val)) == "<class 'QVariant'>":
                #     #             return 0 if val.isNull() else float(val)
                #     #         else:
                #     #             return float(val)
                #     #     except:
                #     #         return 0
                # self.custom_feedback(f"globalcode: {globalcode}", "debug")
                if self.ELEVATION_CHECKBOX is True:
                    self.custom_feedback(
                        "delete irrelevant columns from slope_tempFile2.geojson", "debug"
                    )
                    temp_result = processing.run(
                        "native:deletecolumn",
                        {
                            "INPUT": "ogr:dbname='"
                            + os.path.join(
                                self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                            )
                            + "' table=slope_tempFile2",
                            "COLUMN": [
                                "Walk_ID_2",
                                "CENTER_LON",
                                "CENTER_LAT",
                                "AA_METERS",
                                "AA_MODE",
                                "TOTAL_POP",
                            ],
                            "OUTPUT": "ogr:dbname='"
                            + os.path.join(
                                self.OUTPUT_FOLDER, r"tempDir", "WalkEU_process.gpkg"
                            )
                            + "' table=result_tempFile3",
                        },
                    )["OUTPUT"]
                    
                    self.custom_feedback(
                        "Calculate the OS-Walk-EU Index in advanced field calculator (with Slope)",
                        "debug",
                    )
                    # formulacode = dedent("""
                    #     value = (
                    #         (
                    #             <amenity_rec> * AMENITIES_WEIGHT
                    #             + <green_point> * GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT 
                    #             + <PedShed_point>* PEDESTRIAN_SHED_WEIGHT
                    #         ) * cor_fac
                    #     )-(
                    #         (
                    #             <amenity_rec> * AMENITIES_WEIGHT
                    #             + <green_point> * GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT
                    #             + <PedShed_point> * PEDESTRIAN_SHED_WEIGHT
                    #         ) * cor_fac * ( <Slope_red> / 100 )
                    #     )
                    #     """)
                    # self.custom_feedback(f"fomulacode: {formulacode}", "debug")
                    walkEU = processing.run(
                        "qgis:advancedpythonfieldcalculator",
                        {
                            "INPUT": temp_result,
                            "FIELD_NAME": "Result",
                            "FIELD_TYPE": 1,
                            "FIELD_LENGTH": 10,
                            "FIELD_PRECISION": 3,
                            "GLOBAL": dedent(f"""
                                AMENITIES_WEIGHT = {self.AMENITIES_WEIGHT}
                                PEDESTRIAN_SHED_WEIGHT = {self.PEDESTRIAN_SHED_WEIGHT}
                                GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT = {self.GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT}
                                cor_fac = {cor_fac}
                                """
                            ),
                            "FORMULA": dedent("""
                                value = (
                                    (
                                        <amenity_rec> * AMENITIES_WEIGHT + <green_point> * GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT + <PedShed_point>* PEDESTRIAN_SHED_WEIGHT
                                    ) * cor_fac
                                )-(
                                    (
                                        <amenity_rec> * AMENITIES_WEIGHT + <green_point> * GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT + <PedShed_point> * PEDESTRIAN_SHED_WEIGHT
                                    ) * cor_fac * ( <Slope_red> / 100 )
                                )
                                """
                            ),
                            "OUTPUT": "ogr:dbname='"
                            + os.path.join(self.OUTPUT_FOLDER, r"WalkEU.gpkg")
                            + "' table=WalkEU",
                        },
                    )
                else:
                    self.custom_feedback(
                        "Calculate the OS-Walk-EU Index in advanced field calculator (without Slope)",
                        "debug",
                    )
                    # formulacode = dedent("""
                    #     value = (
                    #         <amenity_rec> * AMENITIES_WEIGHT
                    #         + <green_point> * GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT
                    #         + <PedShed_point> * PEDESTRIAN_SHED_WEIGHT
                    #     ) * cor_fac
                    #     """)
                    # self.custom_feedback(
                    #     f"""{
                    #         "INPUT": {grid_id_layer},
                    #         "FIELD_NAME": "Result",
                    #         "FIELD_TYPE": 1,
                    #         "FIELD_LENGTH": 10,
                    #         "FIELD_PRECISION": 3,
                    #         "GLOBAL": dedent(f```
                    #             AMENITIES_WEIGHT = {self.AMENITIES_WEIGHT};
                    #             PEDESTRIAN_SHED_WEIGHT = {self.PEDESTRIAN_SHED_WEIGHT};
                    #             GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT = {self.GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT};
                    #             cor_fac = {cor_fac}
                    #         ```),
                    #         "FORMULA": dedent(v
                    #             value = (<amenity_rec> * AMENITIES_WEIGHT + <green_point> * GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT + <PedShed_point> * PEDESTRIAN_SHED_WEIGHT) * cor_fac
                    #             ```),
                    #         "OUTPUT": "ogr:dbname='"
                    #         + os.path.join(self.OUTPUT_FOLDER, r"WalkEU.gpkg")
                    #         + "' table=WalkEU",
                    #     }"""
                    # )
                    walkEU = processing.run(
                        "qgis:advancedpythonfieldcalculator",
                        {
                            "INPUT": grid_id_layer,
                            "FIELD_NAME": "Result",
                            "FIELD_TYPE": 1,
                            "FIELD_LENGTH": 10,
                            "FIELD_PRECISION": 3,
                            "GLOBAL": dedent(f"""
                                AMENITIES_WEIGHT = {self.AMENITIES_WEIGHT};
                                PEDESTRIAN_SHED_WEIGHT = {self.PEDESTRIAN_SHED_WEIGHT};
                                GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT = {self.GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT};
                                cor_fac = {cor_fac}
                            """),
                            "FORMULA": dedent("""
                                value = (<amenity_rec> * AMENITIES_WEIGHT + <green_point> * GREEN_AND_BLUE_INFRASTRUCTURE_WEIGHT + <PedShed_point> * PEDESTRIAN_SHED_WEIGHT) * cor_fac
                                """),
                            "OUTPUT": "ogr:dbname='"
                            + os.path.join(self.OUTPUT_FOLDER, r"WalkEU.gpkg")
                            + "' table=WalkEU",
                        },
                    )
            except TypeError as e:
                self.custom_feedback(
                    dedent(f"""
                        There is a TypeError raised in FieldCalculator:
                        {e}
                        trying to do this:
                        and in global:
                    """), 
                    "warning"
                )
            return walkEU

        # these two functions are run in parallel in QgsTasks. Uncomment of QgsTasks crashes Qgis more than twice.
        if self.RAN_FROM_GUI:
            self.dlg = WalkabilityToolDialog()
            self.dlg.progressBar.setValue(25)
        else:
            pass

        amenity_points = calculateAmenities()
        self.custom_feedback(
            "Amenity points calculated: " + str(amenity_points), "info"
        )
        
        self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")
        grid_id, grid_id_layer, grid_id_layer_path = _include_in_grid_id(
            amenity_points, 
            columns=["amenity_rec", "amenity_sum"], 
            grid_id_layer_path=grid_id_layer_path, 
            grid_id_layer=grid_id_layer,
            grid_id=grid_id
        )

        # rest of the functions which run sequentially
        if self.RAN_FROM_GUI:
            self.dlg.progressBar.setValue(50)

        pedshed_file_pointed = calculatePedshed()
        self.custom_feedback(
            "Pedshed points calculated: " + str(pedshed_file_pointed), "info"
        )
        
        self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")
        grid_id, grid_id_layer, grid_id_layer_path  = _include_in_grid_id(
            pedshed_file_pointed, 
            columns=["PedShed", "PedShed_point"], 
            grid_id_layer_path=grid_id_layer_path, 
            grid_id_layer=grid_id_layer,
            grid_id=grid_id
        )

        greenshed_pointed = calcGreen()
        self.custom_feedback(
            "Greenshed points calculated: " + str(greenshed_pointed), "info"
        )
        
        self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")
        grid_id, grid_id_layer, grid_id_layer_path  = _include_in_grid_id(
            greenshed_pointed, 
            columns=["green_pc", "green_point"], 
            grid_id_layer_path=grid_id_layer_path, 
            grid_id_layer=grid_id_layer,
            grid_id=grid_id
        )

        if self.RAN_FROM_GUI:
            self.dlg.progressBar.setValue(75)
        slope_tempFile2 = checkSlope()
        if slope_tempFile2:
            self.custom_feedback(
                "Slope points calculated: " + str(slope_tempFile2), "info"
            )
            
            self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")
            grid_id, grid_id_layer, grid_id_layer_path  = _include_in_grid_id(
                slope_tempFile2, 
                columns=["slope_mean", "Slope_red"], 
                grid_id_layer_path=grid_id_layer_path, 
                grid_id_layer=grid_id_layer,
                grid_id=grid_id
            )

        self.custom_feedback(f"grid_id_tuple: {grid_id, grid_id_layer, grid_id_layer_path}", "debug")
        walkEU = calc(grid_id_layer=grid_id_layer, grid_id=grid_id, grid_id_layer_path=grid_id_layer_path)

        # shutil.rmtree(self.OUTPUT_FOLDER + r'/tempDir', ignore_errors=True)
        if self.RAN_FROM_GUI:
            self.dlg.progressBar.setValue(100)

        return {
            "OUTPUT": self.OUTPUT_FOLDER,
            "OUTPUT_FILE": self.OUTPUT_FOLDER + r"/WalkEU.gpkg",
            "OUTPUT_LAYER": walkEU,
        }
        # ---->Adding the final walkability score into the QGIS instance

    # Delete Temporary Directory
    def delete_temp_dir(self):
        """Delete tempDir and all its contents."""
        tempDir = self.OUTPUT_FOLDER + r"/tempDir"
        """
        try:
            shutil.rmtree(tempDir)
        except OSError as e:
            print("Error: %s - %s." % (e.filename, e.strerror))
        """

    # Upload walkability result on the iface
    def upload_walkability_result(self):
        walkability_gpkg = self.OUTPUT_FOLDER + r"/WalkEU.gpkg"
        gpkg_layers = [layer.GetName() for layer in ogr.Open(walkability_gpkg)]
        walkability_layer = "WalkEU"
        if walkability_layer in gpkg_layers:
            iface.addVectorLayer(
                walkability_gpkg + "|layername=" + walkability_layer,
                walkability_layer,
                "ogr",
            )

        else:
            print('Error: there is no layer named "{}" '.format(walkability_layer))
