from qgis.PyQt.QtWidgets import QGridLayout, QLabel, QSizePolicy, QWidget, QWizardPage

from openlog.datamodel.connection.openlog_connection import OpenLogConnection
from openlog.gui.utils.dataset_selection_widget import DatasetSelectionWidget
from openlog.gui.utils.person_selection_widget import PersonSelectionWidget


class PersonDatasetSelectionPageWizard(QWizardPage):
    def __init__(self, parent: QWidget, openlog_connection: OpenLogConnection) -> None:
        """
        QWizard to select/create person and dataset for xplordb import.

        Args:
            openlog_connection: OpenLogConnection used to import person
            parent : QWidget parent
        """
        super().__init__(parent)

        self._openlog_connection = openlog_connection
        self.setTitle("Person and dataset definition")

        # PERSON
        label = QLabel("Select a person and a dataset for import.")
        label.setWordWrap(True)

        layout = QGridLayout()
        layout.addWidget(label, 0, 0, 1, 2)
        person_label = QLabel("Person")
        person_label.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
        layout.addWidget(person_label, 1, 0)

        self.person_selection = PersonSelectionWidget()
        self.person_selection.set_openlog_connection(openlog_connection)
        layout.addWidget(self.person_selection, 1, 1)
        self.person_cb = self.person_selection.get_person_combobox()
        self.setLayout(layout)

        # Register person combobox and use text instead of index
        self.registerField(
            "person", self.person_cb, "currentText", self.person_cb.currentTextChanged
        )

        # DATASET
        dataset_label = QLabel("Dataset")
        dataset_label.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
        layout.addWidget(dataset_label, 3, 0)

        self.dataset_selection = DatasetSelectionWidget()
        self.dataset_selection.set_openlog_connection(openlog_connection)
        layout.addWidget(self.dataset_selection, 3, 1)
        self.dataset_cb = self.dataset_selection.get_dataset_combobox()
        self.setLayout(layout)

        # Register dataset combobox and use text instead of index
        self.registerField(
            "dataset",
            self.dataset_cb,
            "currentText",
            self.dataset_cb.currentTextChanged,
        )

        # update DatasetSelectionWidget.person attribute whenever person change
        self.person_selection.person_changed.connect(
            lambda code: self.dataset_selection.set_loaded_by_person(code)
        )

    def initializePage(self):

        self.dataset_selection.set_loaded_by_person(self.field("person"))

    def data_label(self) -> str:
        """
        Returns label to be used in confirmation dialog

        Returns: imported data label

        """
        return "Persons", "Datasets"

    def data_count(self) -> int:
        """
        Returns expected imported data count to be displayed in confirmation dialog

        Returns: expected imported data count

        """
        return len(self.person_selection.added_persons), len(
            self.dataset_selection.added_datasets
        )

    def import_data(self) -> bool:
        """
        Import data into openlog database.

        OpenLogConnection.ImportData exception can be raised.

        """
        self.person_selection.import_data()
        self.dataset_selection.import_data()

    def validatePage(self) -> bool:
        """
        Validate page by checking that a person is selected

        Returns: True if a person is selected, False otherwise

        """
        return (
            len(self.person_cb.currentText()) != 0
            and len(self.dataset_cb.currentText()) != 0
        )
