from pathlib import Path

from qgis.core import QgsDataSourceUri, QgsVectorLayer

from openlog.core.barchart_vector_layer import BarChartVectorLayer
from openlog.datamodel.assay.generic_assay import (
    AssayDataExtent,
    AssayDefinition,
    AssayDomainType,
)
from openlog.datamodel.connection.interfaces.layers_interface import LayersInterface


class SpatialiteLayersInterface(LayersInterface):
    def __init__(self, file_path: Path):
        """
        Implements LayersInterface for SpatialiteConnection

        Args:
            file_path: spatialite file path
        """
        super().__init__()
        self._file_path = file_path

    def _get_assay_query(self, assay_def: AssayDefinition) -> str:
        """
        Generate SQL queries for assay views.
        """

        if assay_def.domain == AssayDomainType.TIME:
            q = f"SELECT * FROM {assay_def.variable}"
            return q

        column_names = list(assay_def.columns.keys())
        # add uncertainties and detection limits
        extra_columns = [
            assay_def.columns.get(col).uncertainty.get_uncertainty_columns()
            for col in column_names
        ] + [
            assay_def.columns.get(col).detection_limit.get_detection_columns()
            for col in column_names
        ]
        extra_columns = [item for sublist in extra_columns for item in sublist]
        column_names = column_names + extra_columns
        column_names = sorted(column_names)
        column_sql = ",".join(column_names)

        # TODO : use st_3dlineinterpolatepoint instead, but doesn't work with custom function
        if assay_def.data_extent == AssayDataExtent.DISCRETE:
            q = f"""
                SELECT hole, dataset, person, import_date, x AS depth, {column_sql} FROM
                (SELECT hole,o.*, max(ifnull(eoh, 0), ifnull(planned_eoh, 0))
                FROM {assay_def.variable} AS o
                LEFT JOIN collar AS c
                ON o.hole = c.hole_id) AS tmp
            """

        else:
            q = f"""
                SELECT hole, dataset, person, import_date, x AS top, x_end as bottom, {column_sql}
                FROM
                (SELECT hole,o.*, max(ifnull(eoh, 0), ifnull(planned_eoh, 0))
                FROM {assay_def.variable} AS o
                LEFT JOIN collar AS c
                ON o.hole = c.hole_id) AS tmp
            """

        return q

    def get_assay_layers(
        self, assay_definitions: list[AssayDefinition]
    ) -> list[QgsVectorLayer]:
        """
        Return list of assays as QgsVectorLayers.
        """
        layers = []
        for assay_def in assay_definitions:
            try:
                uri = self._get_datasource_uri()
                query = self._get_assay_query(assay_def)
                uri.setDataSource("", f"({query})", "", "")
                layer_name = f"{assay_def.display_name} - [{self._file_path}]"
                layer = QgsVectorLayer(uri.uri(False), layer_name, "spatialite")
                layers.append(layer)
            except:
                continue

        return layers

    def get_collar_layer(self) -> QgsVectorLayer:
        """
        Return collar QgsVectorLayer

        In spatialite collar geometry is available in collar geom column

        """
        if self.collar_layer is None:
            uri = self._get_datasource_uri()
            uri.setDataSource(
                "",
                "(SELECT * FROM collar LEFT JOIN metadata USING (hole_id))",
                "geom",
                "",
                "",
            )
            self.collar_layer = QgsVectorLayer(
                uri.uri(False), self.get_collar_layer_name(), "spatialite"
            )
            # probably a QGIS bug when layer is empty
            if not self.collar_layer.isValid():
                uri = self._get_datasource_uri()
                uri.setDataSource(
                    "",
                    "collar",
                    "geom",
                    "",
                    "",
                )
                self.collar_layer = QgsVectorLayer(
                    uri.uri(False), self.get_collar_layer_name(), "spatialite"
                )
            self._set_clamping(self.collar_layer)
        return self.collar_layer

    def get_planned_collar_layer(self) -> QgsVectorLayer:
        """
        Return planned_collar QgsVectorLayer

        In spatialite planned collar geometry is available in collar planned_loc column

        """
        if self.planned_collar_layer is None:
            uri = self._get_datasource_uri()
            uri.setDataSource(
                "",
                "(SELECT * FROM collar LEFT JOIN metadata USING (hole_id))",
                "planned_loc",
                "",
                "",
            )
            self.planned_collar_layer = QgsVectorLayer(
                uri.uri(False), self.get_planned_collar_layer_name(), "spatialite"
            )
            # probably a QGIS bug when layer is empty
            if not self.planned_collar_layer.isValid():
                uri = self._get_datasource_uri()
                uri.setDataSource(
                    "",
                    "collar",
                    "planned_loc",
                    "",
                    "",
                )
                self.planned_collar_layer = QgsVectorLayer(
                    uri.uri(False), self.get_planned_collar_layer_name(), "spatialite"
                )
            self._set_clamping(self.planned_collar_layer)
        return self.planned_collar_layer

    def get_collar_layer_name(self) -> str:
        """
        Get collar layer name

        Returns: (str) collar layer name

        """
        return "Collar - [{0}]".format(self._file_path)

    def get_planned_collar_layer_name(self) -> str:
        """
        Get planned collar layer name

        Returns: (str) collar layer name

        """
        return "Planned collar - [{0}]".format(self._file_path)

    def get_collar_trace_layer(self) -> QgsVectorLayer:
        """
        Return collar trace QgsVectorLayer

        In spatialite collar trace geometry is available in collar effective_geom column

        """
        if self.collar_trace_layer is None:
            uri = self._get_datasource_uri()
            uri.setDataSource("", "(SELECT * FROM collar)", "effective_geom")
            self.collar_trace_layer = QgsVectorLayer(
                uri.uri(False), self.get_collar_trace_layer_name(), "spatialite"
            )
            # probably a QGIS bug when layer is empty
            if not self.collar_trace_layer.isValid():
                uri = self._get_datasource_uri()
                uri.setDataSource(
                    "",
                    "collar",
                    "effective_geom",
                    "",
                    "",
                )
                self.collar_trace_layer = QgsVectorLayer(
                    uri.uri(False), self.get_collar_trace_layer_name(), "spatialite"
                )
            self._set_clamping(self.collar_trace_layer)
        return self.collar_trace_layer

    def get_planned_trace_layer(self) -> QgsVectorLayer:
        """
        Return plannedf trace QgsVectorLayer

        In spatialite collar trace geometry is available in collar planned_geom column

        """
        if self.planned_trace_layer is None:
            uri = self._get_datasource_uri()
            uri.setDataSource("", "(SELECT * FROM collar)", "planned_geom")
            self.planned_trace_layer = QgsVectorLayer(
                uri.uri(False), self.get_planned_trace_layer_name(), "spatialite"
            )
            # probably a QGIS bug when layer is empty
            if not self.planned_trace_layer.isValid():
                uri = self._get_datasource_uri()
                uri.setDataSource(
                    "",
                    "collar",
                    "planned_geom",
                    "",
                    "",
                )
                self.planned_trace_layer = QgsVectorLayer(
                    uri.uri(False), self.get_planned_trace_layer_name(), "spatialite"
                )

            self._set_clamping(self.planned_trace_layer)
        return self.planned_trace_layer

    def splitted_trace_layer_available(self) -> bool:
        """Define if splitted trace layer can be used

        :return: True is splitted trace layer can be used, False otherwise
        :rtype: bool
        """
        return True

    def get_splitted_trace_layer(
        self, column_config, planned: bool = False, barcharts: bool = False
    ) -> QgsVectorLayer:
        """
        Return splitted trace QgsVectorLayer

        In spatialite collar trace geometry is available in collar geom_trace column

        """
        layer_name = self.get_splitted_trace_layer_name(
            column_config, planned, barcharts
        )
        id_name = f"{column_config.assay_name}"
        suffix = "planned_trace" if planned else "trace"

        uri = self._get_datasource_uri()
        uri.setDataSource("", f"{id_name}_{suffix}", "geom_interval")

        if barcharts:
            splitted_trace_layer = BarChartVectorLayer(
                uri.uri(False), layer_name, "spatialite"
            )
        else:
            splitted_trace_layer = QgsVectorLayer(
                uri.uri(False), layer_name, "spatialite"
            )
        # check if layer need updates: if so, make it unvalid
        valid = True
        field_names = splitted_trace_layer.fields().names()
        for ftr in splitted_trace_layer.getFeatures():
            attr = ftr.attributes()
            if len(attr) > 0 and attr[field_names.index("need_update")] == 1:
                valid = False
                break
        if valid is False:
            uri = self._get_datasource_uri()
            uri.setDataSource("", f"{id_name}_{suffix}", "fake_column")
            splitted_trace_layer = QgsVectorLayer(
                uri.uri(False), layer_name, "spatialite"
            )

        self._set_clamping(splitted_trace_layer)

        # add to list
        self.splitted_trace_layer.append(splitted_trace_layer)
        self.splitted_trace_layer = [
            lyr
            for lyr in self.splitted_trace_layer
            if self._is_layer_exist(lyr) and lyr.isValid()
        ]

        return splitted_trace_layer

    def get_collar_trace_layer_name(self) -> str:
        return "Trace - [{0}]".format(self._file_path)

    def get_planned_trace_layer_name(self) -> str:
        return "Planned trace - [{0}]".format(self._file_path)

    def _get_datasource_uri(self) -> QgsDataSourceUri:
        """
        Get a QgsDataSourceUri from spatialite file

        Returns: QgsDataSourceUri with current connection parameters

        """
        uri = QgsDataSourceUri()
        uri.setDatabase(str(self._file_path))
        return uri
