# -*- coding: utf-8 -*-

"""
/***************************************************************************
 LandCoverDownloader
                                 A QGIS plugin
 Downloads Sentinel-2 Land Cover Data from Esri's Living Atlas
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2026-01-31
        copyright            : (C) 2026 by CustomCartographix
        email                : john@customcartographix.com
        version              : 2.0.0
        version date         : 2026-01-31
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

 /***************************************************************************
 *                                                                         *
 *   This script contains the necessary functions for the OpenHLZ          *
 *   processing algorithm to work.                                         *
 *                                                                         *
 ***************************************************************************/
"""

# Import statements
from qgis.core import (QgsRasterLayer,
                       QgsNetworkAccessManager,
                       QgsVectorLayer,
                       QgsVectorFileWriter)
from qgis import processing
from pyproj import Proj, transform
from PyQt5.QtNetwork import QNetworkRequest
from PyQt5.QtCore import QUrl
from json import load
from os import remove


def downloadCopernicusGlo30Data(aoi_layer, scratch_folder, project_coordinate_system_name):
    """
    Function to download Copernicus GLO-30 data based on input AOI using OpenTopography API.

    Returns output filename
    """
    # Get bounding box
    extent_var = aoi_layer.extent()
    extent_array = [extent_var.xMinimum(), extent_var.yMinimum(), extent_var.xMaximum(), extent_var.yMaximum()]
    input_projection = Proj(project_coordinate_system_name)
    output_projection = Proj('epsg:4326')
    min_x_proj, min_y_proj = transform(input_projection, output_projection, extent_array[0], extent_array[1])
    max_x_proj, max_y_proj = transform(input_projection, output_projection, extent_array[2], extent_array[3])
    url_bbox = ('south=' + str(min_x_proj) + '&north=' + str(max_x_proj) + '&west=' + str(min_y_proj) +
                '&east=' + str(max_y_proj))

    # Download Data
    download_url = ('https://portal.opentopography.org/API/globaldem?demtype=COP30&' + url_bbox +
                    '&outputFormat=GTiff&API_Key=8422c95f5e3490ecca69be478123007c')
    print(download_url)
    download_url = QUrl(download_url)
    request = QNetworkRequest(download_url)
    output_filename = scratch_folder + '/srtm_dem.tif'
    network_access_manager = QgsNetworkAccessManager()
    response = network_access_manager.blockingGet(request)
    content = response.content()
    open(output_filename, "wb").write(content)
    temp_dem = open(output_filename, )
    temp_dem.close()

    # Return output filename
    return output_filename


def getJson(aoi_layer, scratch_folder, project_coordinate_system_name):
    """
    Function to get JSON file return for either DEM or LULC

    Returns JSON files for 1m, 10m, and 1/9 arc-second data
    """

    # Get bounding box
    extent_var = aoi_layer.extent()
    extent_array = [extent_var.xMinimum(), extent_var.yMinimum(), extent_var.xMaximum(), extent_var.yMaximum()]
    input_projection = Proj(project_coordinate_system_name)
    output_projection = Proj('epsg:4326')
    min_x_proj, min_y_proj = transform(input_projection, output_projection, extent_array[0], extent_array[1])
    max_x_proj, max_y_proj = transform(input_projection, output_projection, extent_array[2], extent_array[3])
    url_bbox = str(min_y_proj) + ',' + str(min_x_proj) + ',' + str(max_y_proj) + ',' + str(max_x_proj)

    # Get json return for 1m DEM
    url = 'https://tnmaccess.nationalmap.gov/api/v1/products?bbox=' + url_bbox + \
          '&datasets=Digital+Elevation+Model+(DEM)+1+meter&prodFormats=GeoTIFF&outputFormat=json'
    print(url)
    qurl = QUrl(url)
    output_json_filename_1m = scratch_folder + '\\json_1m.json'
    request = QNetworkRequest(qurl)
    network_access_manager = QgsNetworkAccessManager()
    response_1m = network_access_manager.blockingGet(request)
    content = response_1m.content()
    open(output_json_filename_1m, "wb").write(content)
    temp_file = open(output_json_filename_1m, )
    json_1m = load(temp_file)
    temp_file.close()

    # Get json return for 1/3 arc-second (10m) data
    url = 'https://tnmaccess.nationalmap.gov/api/v1/products?bbox=' + url_bbox + \
          ('&datasets=National%20Elevation%20Dataset%20('
           'NED)%201%20arc-second&prodFormats=GeoTIFF&outputFormat=json')
    qurl = QUrl(url)
    output_json_filename_10m = scratch_folder + '/json_10m.json'
    request = QNetworkRequest(qurl)
    network_access_manager = QgsNetworkAccessManager()
    response_10m = network_access_manager.blockingGet(request)
    content = response_10m.content()
    open(output_json_filename_10m, "wb").write(content)
    temp_file_10m = open(output_json_filename_10m, )
    json_10m = load(temp_file_10m)
    temp_file_10m.close()

    # Get json return for 1/9 arc-second (1m) DEM
    url = 'https://tnmaccess.nationalmap.gov/api/v1/products?bbox=' + url_bbox + \
          ('&datasets = National % 20Elevation % 20Dataset % 20(NED) % 201 / 9 % 20arc - second&prodFormats=GeoTIFF&'
           'outputFormat=json')
    print(url)
    qurl = QUrl(url)
    output_json_filename_19as = scratch_folder + '/json_19as.json'
    request = QNetworkRequest(qurl)
    network_access_manager = QgsNetworkAccessManager()
    response_19as = network_access_manager.blockingGet(request)
    content = response_19as.content()
    open(output_json_filename_19as, "wb").write(content)
    temp_file = open(output_json_filename_19as, )
    json_19as = load(temp_file)
    temp_file.close()

    return json_1m, json_10m, json_19as


def downloadDem(i, json_file, scratch_folder, input_data):
    """
    Function to download DEM data based on input location. Will first try 1m, then will try 10m if no 1m

    Returns filename of saved DEM raster
    """

    # Download 1m data
    if input_data == '1m':
        temp_dict = json_file['items'][i]
        download_url = QUrl(temp_dict['downloadURL'])
        request = QNetworkRequest(download_url)
        output_filename = scratch_folder + '/dem_1m_' + str(i) + '.tif'
        network_access_manager = QgsNetworkAccessManager()
        response = network_access_manager.blockingGet(request)
        content = response.content()
        open(output_filename, "wb").write(content)
        temp_dem = open(output_filename, )
        temp_dem.close()
        return output_filename

    # Download 1/9m data
    if input_data == '19m':
        temp_dict = json_file['items'][i]
        download_url = QUrl(temp_dict['downloadURL'])
        request = QNetworkRequest(download_url)
        output_filename = scratch_folder + '/dem_19m_' + str(i) + '.tif'
        network_access_manager = QgsNetworkAccessManager()
        response = network_access_manager.blockingGet(request)
        content = response.content()
        open(output_filename, "wb").write(content)
        temp_dem = open(output_filename, )
        temp_dem.close()
        return output_filename

    # Else download 10m data
    else:
        temp_dict = json_file['items'][i]
        download_url = QUrl(temp_dict['downloadURL'])
        request = QNetworkRequest(download_url)
        output_filename = scratch_folder + '/dem_10m_' + str(i) + '.tif'
        network_access_manager = QgsNetworkAccessManager()
        response = network_access_manager.blockingGet(request)
        content = response.content()
        open(output_filename, "wb").write(content)
        temp_dem = open(output_filename, )
        temp_dem.close()
        return output_filename


def downloadLandCoverRaster(i, utm_zones, scratch_folder):
    """
    Function to download land cover data based on current location

    Returns filename of saved land cover raster
    """

    # Download data based on whichever UTM zone the aoi is within
    # Get UTM zone and create output filename
    zone = utm_zones[i]
    zone_id = str(zone[0]) + zone[1]
    output_filename = scratch_folder + '/' + zone_id + ".tif"

    # Download data from Living Atlas and write to scratch folder
    url = ("https://lulctimeseries.blob.core.windows.net/lulctimeseriesv003/lc2024/" + zone_id +
           "_20240101-20241231.tif")
    qurl = QUrl(url)
    request = QNetworkRequest(qurl)
    network_access_manager = QgsNetworkAccessManager()
    response = network_access_manager.blockingGet(request)
    content = response.content()
    open(output_filename, "wb").write(content)
    temp_lc = open(output_filename, )
    temp_lc.close()

    # Return filename of downloaded land cover raster
    return output_filename


def mosaicAndClipRasters(input_raster_filenames, aoi_layer, scratch_folder, output_raster, instance):
    """
    Function to mosaic and clip rasters based on input AOI.

    Outputs to specified location
    """

    # Clip/mosaic data
    # Slightly different methodology if greater than one tile (i.e. need to mosaic after clipping)
    if len(input_raster_filenames) == 1:
        input_raster = input_raster_filenames[0]
        input_raster_layer = QgsRasterLayer(input_raster, 'temp layer')
        temp_data_type = int(input_raster_layer.dataProvider().dataType(0))
        processing.run("gdal:cliprasterbymasklayer", {'INPUT': input_raster,
                                                      'MASK': aoi_layer,
                                                      'SOURCE_CRS': None,
                                                      'TARGET_CRS': aoi_layer.crs().authid(),
                                                      'TARGET_EXTENT': None,
                                                      'NODATA': None,
                                                      'ALPHA_BAND': False,
                                                      'CROP_TO_CUTLINE': True,
                                                      'KEEP_RESOLUTION': False,
                                                      'SET_RESOLUTION': False,
                                                      'X_RESOLUTION': None,
                                                      'Y_RESOLUTION': None,
                                                      'MULTITHREADING': False,
                                                      'OPTIONS': '',
                                                      'DATA_TYPE': temp_data_type,
                                                      'EXTRA': '',
                                                      'OUTPUT': output_raster})
        instance.removeMapLayer(input_raster_layer)
    else:
        clipped_rasters = []
        for i in range(len(input_raster_filenames)):
            temp_name = scratch_folder + '/clipped_' + str(i) + '.tif'
            clipped_rasters.append(temp_name)
            input_raster = input_raster_filenames[i]
            input_raster_layer = QgsRasterLayer(input_raster, 'temp layer')
            temp_data_type = int(input_raster_layer.dataProvider().dataType(0))
            processing.run("gdal:cliprasterbymasklayer",
                           {'INPUT': input_raster,
                            'MASK': aoi_layer,
                            'SOURCE_CRS': None,
                            'TARGET_CRS': aoi_layer.crs().authid(),
                            'TARGET_EXTENT': None,
                            'NODATA': None,
                            'ALPHA_BAND': False,
                            'CROP_TO_CUTLINE': True,
                            'KEEP_RESOLUTION': False,
                            'SET_RESOLUTION': False,
                            'X_RESOLUTION': None,
                            'Y_RESOLUTION': None,
                            'MULTITHREADING': False,
                            'OPTIONS': '',
                            'DATA_TYPE': temp_data_type,
                            'EXTRA': '',
                            'OUTPUT': temp_name})
            instance.removeMapLayer(input_raster_layer)
        clipped_rasters_temp_layer = QgsRasterLayer(clipped_rasters[0], 'temp layer')
        temp_data_type = int(clipped_rasters_temp_layer.dataProvider().dataType(0))
        processing.run("gdal:merge", {'INPUT': clipped_rasters,
                                      'PCT': False,
                                      'SEPARATE': False,
                                      'NODATA_INPUT': None,
                                      'NODATA_OUTPUT': None,
                                      'OPTIONS': '',
                                      'EXTRA': '',
                                      'DATA_TYPE': temp_data_type,
                                      'OUTPUT': output_raster})
        instance.removeMapLayer(clipped_rasters_temp_layer)


def generateHlsRaster(output, projected_dem, projected_land_cover, slope_caution, slope_limit, model_coordinate_system,
                      style_path, instance, scratch_folder, suitable_land_cover_classes='5,8,11'):
    """
    Function to generate HLS raster based on slope and LULC constraints (see README.md on GitHub for more info on
    methodology).
    """
    # Calculate slope raster
    processing.run("gdal:slope", {'INPUT': projected_dem, 'BAND': 1, 'SCALE': 1, 'AS_PERCENT': False,
                                  'COMPUTE_EDGES': False, 'ZEVENBERGEN': False, 'OPTIONS': '', 'EXTRA': '',
                                  'OUTPUT': scratch_folder + '/slope_raster.tif'})

    # Reclassify slope and LULC rasters
    processing.run("qgis:rastercalculator", {'EXPRESSION': '"slope_raster@1" < ' + str(slope_caution),
                                             'LAYERS': [scratch_folder + '/slope_raster.tif'],
                                             'CELLSIZE': 0,
                                             'EXTENT': None,
                                             'CRS': model_coordinate_system,
                                             'OUTPUT': scratch_folder + '/slope_caution.tif'})
    processing.run("qgis:rastercalculator", {'EXPRESSION': '"slope_raster@1" < ' + str(slope_limit),
                                             'LAYERS': [scratch_folder + '/slope_raster.tif'],
                                             'CELLSIZE': 0,
                                             'EXTENT': None,
                                             'CRS': model_coordinate_system,
                                             'OUTPUT': scratch_folder + '/slope_limit.tif'})
    processing.run("qgis:rastercalculator", {'EXPRESSION': '"slope_limit@1" + "slope_caution@1"',
                                             'LAYERS': [scratch_folder + '/slope_limit.tif',
                                                        scratch_folder + '/slope_caution.tif'],
                                             'CELLSIZE': 0,
                                             'EXTENT': None,
                                             'CRS': model_coordinate_system,
                                             'OUTPUT': scratch_folder + '/slope_reclass.tif'})
    remove(projected_dem)
    remove(scratch_folder + '/slope_limit.tif')
    remove(scratch_folder + '/slope_caution.tif')

    # Reclassify land cover based on possible input classes
    input_classes = suitable_land_cover_classes.split(',')
    input_classes_len = len(input_classes)
    expression = ''
    for i in range(input_classes_len):
        value = str(input_classes[i])
        if i == (input_classes_len - 1):
            expression += '("clipped_lc@1" = ' + value + ')'
        else:
            expression += '("clipped_lc@1" = ' + value + ') or '
    processing.run("qgis:rastercalculator", {'EXPRESSION': expression,
                                             'LAYERS': [projected_land_cover],
                                             'CELLSIZE': 0,
                                             'EXTENT': None,
                                             'CRS': model_coordinate_system,
                                             'OUTPUT': scratch_folder + '/lc_reclass.tif'})
    remove(projected_land_cover)

    # Generate HLS
    hls_raster = processing.run("qgis:rastercalculator", {'EXPRESSION': '"lc_reclass@1" * "slope_reclass@1',
                                                          'LAYERS': [scratch_folder + '/lc_reclass.tif',
                                                                     scratch_folder + '/slope_reclass.tif'],
                                                          'CELLSIZE': 0,
                                                          'EXTENT': None,
                                                          'CRS': model_coordinate_system,
                                                          'OUTPUT': output})['OUTPUT']
    remove(scratch_folder + '/lc_reclass.tif')
    remove(scratch_folder + '/slope_reclass.tif')

    # Add raster layer to map
    hls_layer = QgsRasterLayer(hls_raster, 'HLS Overlay')
    instance.addMapLayer(hls_layer)

    # Add layer style and trigger repaint
    hls_layer.loadNamedStyle(style_path)
    hls_layer.triggerRepaint()

    return hls_raster


def identifyHlzs(output, hls_raster, tdp_diameter, model_coordinate_system, style_path,
                 instance, scratch_folder):
    """
    Function to identify possible HLZs based on criteria and HLS raster
    """
    # Convert raster (only values of 1 or 2) to polygons
    processing.run("gdal:polygonize", {'INPUT': hls_raster,
                                       'BAND': 1,
                                       'FIELD': 'DN',
                                       'EIGHT_CONNECTEDNESS': False,
                                       'EXTRA': '',
                                       'OUTPUT': scratch_folder + '/pixel_polygons.shp'})

    processing.run("native:reprojectlayer", {'INPUT': scratch_folder + '/pixel_polygons.shp',
                                             'TARGET_CRS': model_coordinate_system,
                                             'OPERATION': '+proj=noop',
                                             'OUTPUT': scratch_folder + '/hls_pixels_polygons_projected.shp'})

    selectAndExportFeatures(scratch_folder + '/hls_pixels_polygons_projected.shp',
                            '("DN" = 1) or ("DN" = 2)',
                            scratch_folder + '/selected_pixel_polygons.shp')

    # Dissolve polygons
    hls_dissolve = processing.run("native:dissolve", {'INPUT': scratch_folder + '/selected_pixel_polygons.shp',
                                                      'FIELD': [],
                                                      'SEPARATE_DISJOINT': True,
                                                      'OUTPUT': 'TEMPORARY_OUTPUT'})['OUTPUT']
    hls_dissolve_geometry = \
        processing.run("qgis:exportaddgeometrycolumns", {'INPUT': hls_dissolve,
                                                         'CALC_METHOD': 0,
                                                         'OUTPUT': scratch_folder + '/hls_dissolve_geometry.shp'})['OUTPUT']

    # Select polygons based on area
    selectAndExportFeatures(scratch_folder + '/hls_dissolve_geometry.shp',
                            '"area" > ' + str(tdp_diameter * tdp_diameter),
                            scratch_folder + '/large_enough_areas.shp')

    # Calculate centroids
    initial_hlzs = processing.run("native:centroids", {'INPUT': scratch_folder + '/large_enough_areas.shp',
                                                       'ALL_PARTS': True,
                                                       'OUTPUT': 'TEMPORARY_OUTPUT'})['OUTPUT']

    output_hlzs = processing.run("native:reprojectlayer", {'INPUT': initial_hlzs,
                                                           'TARGET_CRS': model_coordinate_system,
                                                           'OPERATION': '+proj=noop',
                                                           'OUTPUT': output})['OUTPUT']
    # Add raster layer to map
    hlz_layer = QgsVectorLayer(output_hlzs, 'Possible HLZs')
    instance.addMapLayer(hlz_layer)

    # Add layer style and trigger repaint
    hlz_layer.loadNamedStyle(style_path)
    hlz_layer.triggerRepaint()

    return output_hlzs


def selectAndExportFeatures(input_filename, input_expression, output_filename):
    """
    Function to select features within an input file based on an input expression and then export them to a new file
    """
    temp_layer = QgsVectorLayer(input_filename, 'temp_layer')
    temp_layer.selectByExpression(input_expression, QgsVectorLayer.SetSelection)
    QgsVectorFileWriter.writeAsVectorFormat(temp_layer, output_filename, 'utf-8', driverName='ESRI Shapefile',
                                            onlySelected=True)
