# -*- coding: utf-8 -*-

"""
/***************************************************************************
 OpenLidarTools
                                 A QGIS QGISplugin
 Open LiDAR Toolbox
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-03-10
        copyright            : (C) 2021 by Benjamin Štular, Edisa Lozić,
                                                            Stefan Eichert
        email                : stefaneichert@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Benjamin Štular, Edisa Lozić, Stefan Eichert'
__date__ = '2021-03-10'
__copyright__ = '(C) 2021 by Benjamin Štular, Edisa Lozić, Stefan Eichert'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

"""
Model exported as python.
Name : Lidar Pipeline
Group : OpenLidarToolbox
With QGIS : 31604
"""

import inspect
import os

import processing
from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.core import (
    QgsProcessingAlgorithm, QgsProcessingMultiStepFeedback,
    QgsProcessingParameterBoolean, QgsProcessingParameterCrs,
    QgsProcessingParameterFile, QgsProcessingParameterNumber,
    QgsProcessingParameterString, QgsProcessingUtils)
from .utils.utils import getHelpText, setCrs, randomfilename


class BaseData(QgsProcessingAlgorithm):

    def initAlgorithm(self, config=None):

        self.addParameter(
            QgsProcessingParameterFile(
                'InputFilelaslaz',
                'Input LAS/LAZ file',
                behavior=QgsProcessingParameterFile.File,
                fileFilter='Lidar Files (*.las *.laz)',
                defaultValue=None))
        self.addParameter(
            QgsProcessingParameterBoolean(
                'classLas',
                'The input LAS/LAZ file is already classified',
                optional=False,
                defaultValue=False))
        self.addParameter(
            QgsProcessingParameterBoolean(
                'LowNoise',
                'Remove low noise',
                optional=False,
                defaultValue=False))
        self.addParameter(QgsProcessingParameterCrs(
            'CRS',
            'Source File Coordinate System',
            defaultValue=setCrs()))
        self.addParameter(
            QgsProcessingParameterNumber(
                'SetCellSize',
                'Cell Size',
                type=QgsProcessingParameterNumber.Double,
                minValue=0,
                maxValue=1.79769e+308,
                defaultValue=0.5))
        self.addParameter(QgsProcessingParameterString(
            'prefix',
            'Name prefix for layers',
            multiLine=False,
            defaultValue='',
            optional=True))
        self.addParameter(QgsProcessingParameterBoolean(
            'TIN',
            'TLI',
            optional=False,
            defaultValue=True))
        self.addParameter(QgsProcessingParameterBoolean(
            'IDW',
            'IDW',
            optional=False,
            defaultValue=True))
        self.addParameter(QgsProcessingParameterBoolean(
            'GPD',
            'Ground Point Density',
            optional=False,
            defaultValue=True))
        self.addParameter(QgsProcessingParameterBoolean(
            'LVD',
            'Low Vegetation Density',
            optional=False,
            defaultValue=True))

    def processAlgorithm(self, parameters, context, model_feedback):

        feedback = QgsProcessingMultiStepFeedback(18, model_feedback)
        results = {}
        outputs = {}

        #if las/laz file is already classified, skip classification
        if parameters['classLas']:
            las_height_classify_file = parameters['InputFilelaslaz']
        else:
            alg_params = {
                'InputFilelaslaz': parameters['InputFilelaslaz'],
                'LAS':
                    QgsProcessingUtils.generateTempFilename('lasheightCl.las'),
                'LowNoise': parameters['LowNoise']}
            outputs['ClassifyLaslaz'] = processing.run(
                'Open LiDAR Toolbox:ToClassLas',
                alg_params,
                context=context,
                feedback=feedback,
                is_child_algorithm=True)
            las_height_classify_file = \
                outputs['ClassifyLaslaz']['classifiedLAZ']
            results['LAS'] = outputs['ClassifyLaslaz']['classifiedLAZ']

        step = 1
        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        #convert from laz to las if necessary (needed for WBT)
        if parameters['InputFilelaslaz'][-4:] == '.laz':
            alg_params = {
                'ADDITIONAL_OPTIONS': '',
                'APPEND_LAX': False,
                'CPU64': False,
                'CREATE_LAX': False,
                'GUI': False,
                'INPUT_LASLAZ': parameters['InputFilelaslaz'],
                'OUTPUT_LASLAZ':
                    QgsProcessingUtils.generateTempFilename('lasheight.las'),
                'REPORT_SIZE': False,
                'VERBOSE': False}
            las_height_classify_file = alg_params['OUTPUT_LASLAZ']
            outputs['Laszip'] = processing.run(
                'LAStools:LasZip',
                alg_params,
                context=context,
                feedback=feedback,
                is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        #ground point density by WBT
        alg_params = {
            'exclude_cls': '0,1,3,4,5,7,8',
            'input': las_height_classify_file,
            'maxz': None,
            'minz': None,
            'radius': 10,
            'resolution': parameters['SetCellSize'],
            'returns': 0,
            'output': QgsProcessingUtils.generateTempFilename('gpd.tif')}
        ground_point_density_file = alg_params['output']
        outputs['Lidarpointdensity1_ground'] = processing.run(
            'wbt:LidarPointDensity',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # resampleGPD
        alg_params = {
            'GRASS_RASTER_FORMAT_META': '',
            'GRASS_RASTER_FORMAT_OPT': '',
            'GRASS_REGION_CELLSIZE_PARAMETER': parameters['SetCellSize'],
            'GRASS_REGION_PARAMETER': ground_point_density_file,
            'input': ground_point_density_file,
            'output': QgsProcessingUtils.generateTempFilename('GPDres.tif')}
        GPDfileR = alg_params['output']
        outputs['Resamplegpd'] = processing.run(
            'grass7:r.resample',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # Warp (reproject)
        alg_params = {
            'DATA_TYPE': 0,
            'EXTRA': '',
            'INPUT': GPDfileR,
            'MULTITHREADING': False,
            'NODATA': None,
            'OPTIONS': '',
            'RESAMPLING': 0,
            'SOURCE_CRS': parameters['CRS'],
            'TARGET_CRS': parameters['CRS'],
            'TARGET_EXTENT': None,
            'TARGET_EXTENT_CRS': None,
            'TARGET_RESOLUTION': None,
            'OUTPUT': QgsProcessingUtils.generateTempFilename('GPDres.tif')}
        outputs['WarpReproject'] = processing.run(
            'gdal:warpreproject',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)
        GPDfileR = alg_params['OUTPUT']
        results['GPD'] = GPDfileR

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1


        # LidarPointDensity2_LowVeg
        alg_params = {
            'exclude_cls': '0,1,2,4,5,6,7',
            'input': las_height_classify_file,
            'maxz': None,
            'minz': None,
            'radius': 10,
            'resolution': parameters['SetCellSize'],
            'returns': 0,
            'output': QgsProcessingUtils.generateTempFilename('lvd.tif')
        }
        low_vegetation_file = alg_params['output']
        outputs['Lidarpointdensity2_lowveg'] = processing.run(
            'wbt:LidarPointDensity',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # resampleLVD
        alg_params = {
            'GRASS_RASTER_FORMAT_META': '',
            'GRASS_RASTER_FORMAT_OPT': '',
            'GRASS_REGION_CELLSIZE_PARAMETER': parameters['SetCellSize'],
            'GRASS_REGION_PARAMETER': ground_point_density_file,
            'input': low_vegetation_file,
            'output': QgsProcessingUtils.generateTempFilename('LVD.tif')
        }
        low_vegetation_density_file_resampled = alg_params['output']
        outputs['Resamplelvd'] = processing.run(
            'grass7:r.resample',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # Warp (reproject)
        alg_params = {
            'DATA_TYPE': 0,
            'EXTRA': '',
            'INPUT': low_vegetation_density_file_resampled,
            'MULTITHREADING': False,
            'NODATA': None,
            'OPTIONS': '',
            'RESAMPLING': 0,
            'SOURCE_CRS': parameters['CRS'],
            'TARGET_CRS': parameters['CRS'],
            'TARGET_EXTENT': None,
            'TARGET_EXTENT_CRS': None,
            'TARGET_RESOLUTION': None,
            'OUTPUT': QgsProcessingUtils.generateTempFilename('LVD.tif')
        }

        outputs['WarpReproject'] = processing.run(
            'gdal:warpreproject',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)
        low_vegetation_density_file_resampled = alg_params['OUTPUT']
        results['LVD'] = low_vegetation_density_file_resampled

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # LidarTINGridding
        alg_params = {
            'exclude_cls': '0,1,3,4,5,7',
            'input': las_height_classify_file,
            'max_triangle_edge_length': None,
            'maxz': None,
            'minz': None,
            'parameter': 0,
            'resolution': parameters['SetCellSize'],
            'returns': 0,
            'output': QgsProcessingUtils.generateTempFilename('tin.tif')}
        triangulated_network_grid = alg_params['output']
        outputs['Lidartingridding'] = processing.run(
            'wbt:LidarTINGridding',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # resampleTIN
        alg_params = {
            'GRASS_RASTER_FORMAT_META': '',
            'GRASS_RASTER_FORMAT_OPT': '',
            'GRASS_REGION_CELLSIZE_PARAMETER': parameters['SetCellSize'],
            'GRASS_REGION_PARAMETER': ground_point_density_file,
            'input': triangulated_network_grid,
            'output': QgsProcessingUtils.generateTempFilename('tinRes.tif')}
        tin_file_resampled = alg_params['output']
        outputs['ResampleTin'] = processing.run(
            'grass7:r.resample',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # Warp (reproject)
        alg_params = {
            'DATA_TYPE': 0,
            'EXTRA': '',
            'INPUT': tin_file_resampled,
            'MULTITHREADING': False,
            'NODATA': None,
            'OPTIONS': '',
            'RESAMPLING': 0,
            'SOURCE_CRS': parameters['CRS'],
            'TARGET_CRS': parameters['CRS'],
            'TARGET_EXTENT': None,
            'TARGET_EXTENT_CRS': None,
            'TARGET_RESOLUTION': None,
            'OUTPUT': QgsProcessingUtils.generateTempFilename(
                'tin_file_resampled.tif')}

        outputs['WarpReproject'] = processing.run(
            'gdal:warpreproject',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        tin_file_resampled = alg_params['OUTPUT']
        results['DEM'] = tin_file_resampled

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # LidarIdwInterpolation
        alg_params = {
            'exclude_cls': '0,1,3,4,5,7,8,9',
            'input': las_height_classify_file,
            'maxz': None,
            'minz': None,
            'parameter': 0,
            'radius': 5,
            'resolution': parameters['SetCellSize'],
            'returns': 0,
            'weight': 2,
            'output': QgsProcessingUtils.generateTempFilename('idw.tif')}
        lidarIDW = alg_params['output']
        outputs['Lidaridwinterpolation'] = processing.run(
            'wbt:LidarIdwInterpolation',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # resampleIDW
        alg_params = {
            'GRASS_RASTER_FORMAT_META': '',
            'GRASS_RASTER_FORMAT_OPT': '',
            'GRASS_REGION_CELLSIZE_PARAMETER': parameters['SetCellSize'],
            'GRASS_REGION_PARAMETER': ground_point_density_file,
            'input': lidarIDW,
            'output': QgsProcessingUtils.generateTempFilename('IDWres.tif')}
        IDWfileR = alg_params['output']
        outputs['ResampleIDW'] = processing.run(
            'grass7:r.resample',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # Warp (reproject)
        alg_params = {
            'DATA_TYPE': 0,
            'EXTRA': '',
            'INPUT': IDWfileR,
            'MULTITHREADING': False,
            'NODATA': None,
            'OPTIONS': '',
            'RESAMPLING': 0,
            'SOURCE_CRS': parameters['CRS'],
            'TARGET_CRS': parameters['CRS'],
            'TARGET_EXTENT': None,
            'TARGET_EXTENT_CRS': None,
            'TARGET_RESOLUTION': None,
            'OUTPUT': QgsProcessingUtils.generateTempFilename('IDWres.tif')
        }

        outputs['WarpReproject'] = processing.run(
            'gdal:warpreproject',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True)
        IDWfileR = alg_params['OUTPUT']
        results['IDW'] = IDWfileR

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # Load layer into project
        if parameters['IDW']:
            alg_params = {
                'INPUT': IDWfileR,
                'NAME': f"{parameters['prefix']}IDW"}
            outputs['LoadLayerIntoProject'] = processing.run(
                'native:loadlayer',
                alg_params,
                context=context,
                feedback=feedback,
                is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # Load layer into project
        if parameters['GPD']:
            alg_params = {
                'INPUT': GPDfileR,
                'NAME': f"{parameters['prefix']}Ground Point Density"}
            outputs['LoadLayerIntoProject'] = processing.run(
                'native:loadlayer',
                alg_params,
                context=context,
                feedback=feedback,
                is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # Load layer into project
        if parameters['TIN']:
            alg_params = {
                'INPUT': tin_file_resampled,
                'NAME': f"{parameters['prefix']}TLI"}
            outputs['LoadLayerIntoProject'] = processing.run(
                'native:loadlayer',
                alg_params,
                context=context,
                feedback=feedback,
                is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}
        step += 1

        # Load layer into project
        if parameters['LVD']:
            alg_params = {
                'INPUT': low_vegetation_density_file_resampled,
                'NAME': f"{parameters['prefix']}Low Vegetation Density"}
            outputs['LoadLayerIntoProject'] = processing.run(
                'native:loadlayer',
                alg_params,
                context=context,
                feedback=feedback,
                is_child_algorithm=True)

        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}

        return results

    def name(self):
        return 'basedata'

    def displayName(self):
        return 'Create base data'

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def icon(self):
        cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        icon = QIcon(
            os.path.join(os.path.join(cmd_folder, 'icons/2_4_basedata.png')))
        return icon

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def shortHelpString(self):
        title = """
            <html><body><h2>Algorithm description</h2>
            <p>This is a pipeline that takes an airborne LiDAR point cloud to 
            produce rasters needed for further processing or used directly in 
            archaeological (or similar) workflows.</p>
            <h2>Input parameters</h2>"""

        conclusion = """
            <br><br>
            Create base data incorporates parts of Lastools, Whitebox tools, 
            GDAL, GRASS GIS, and QGIS core tools.
            <br><br>
            <p><b>References:</b><br><br> Štular, B.; Eichert, S.; Lozić, E. 
            Airborne LiDAR Point Cloud Processing for Archaeology. Pipeline and 
            QGIS Toolbox. Remote Sens. 2021, 16, 3225. 
            <a href="https://doi.org/10.3390/rs13163225">
            https://doi.org/10.3390/rs13163225</a></p>
            <br><a href="https://stefaneichert.github.io/OpenLidarToolbox/">
            Website</a>
            <br><p align="right">Algorithm author: Benjamin Štular, Edisa Lozić,
            Stefan Eichert </p><p align="right">Help author: Benjamin Štular, 
            Edisa Lozić, Stefan Eichert</p></body></html>"""

        helpText = title + getHelpText(['lasfile', 'noise', 'crs', 'cell',
                                        'prefix', 'gpd', 'lvd', 'las',
                                        'tli', 'idw']) +\
                   conclusion

        helpText = " ".join((helpText.replace('\n','')).split())

        return helpText

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return BaseData()
