# -*- coding: utf-8 -*-
"""
/***************************************************************************
 OpenHazardsPH
                                 A QGIS plugin
 A plugin to load publicly available geospatial hazard data created by various agencies in the Philippines.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-01-19
        git sha              : $Format:%H$
        copyright            : (C) 2020 by BNHR
        email                : hi@bnhr.xyz
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtWidgets import QTableWidget, QTableWidgetItem
from qgis.core import QgsMessageLog, QgsVectorLayer, QgsRasterLayer, QgsProject

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .open_hazards_ph_dockwidget import OpenHazardsPHDockWidget
import os.path

# Import urlopen
from urllib.request import urlopen

URLS = {"lipad": "https://raw.githubusercontent.com/maning/open-hazards-ph/master/lipad.txt",
        "noah": "https://raw.githubusercontent.com/maning/open-hazards-ph/master/noah.txt",}

class OpenHazardsPH:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'OpenHazardsPH_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Open Hazards PH')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'OpenHazardsPH')
        self.toolbar.setObjectName(u'OpenHazardsPH')

        #print "** INITIALIZING OpenHazardsPH"

        self.pluginIsActive = False
        self.dockwidget = None


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('OpenHazardsPH', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/open_hazards_ph/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Get Philippine Hazard Data'),
            callback=self.run,
            parent=self.iface.mainWindow())

    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING OpenHazardsPH"

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD OpenHazardsPH"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Open Hazards PH'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    #--------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            #print "** STARTING OpenHazardsPH"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = OpenHazardsPHDockWidget()
                self.dockwidget.dataSourceComboBox.addItems([*URLS])
                self.dockwidget.dataSourceComboBox.currentTextChanged.connect(self.get_data_list)
                self.get_data_list()
                self.dockwidget.loadVectorButton.clicked.connect(self.load_vector_data)
                self.dockwidget.loadRasterButton.clicked.connect(self.load_raster_data)

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

    def get_data_list(self):
        # QgsMessageLog.logMessage(list_str, "OpenHazardsPH")
        data_source = self.dockwidget.dataSourceComboBox.currentText()
        data_list = []
        with urlopen(URLS[data_source]) as hazard_list:
            list_str = str(hazard_list.read())
            list_arr = list_str[2:-1].split("\\n")

            for item in list_arr:
                i2 = item.split(" ")
                i3 = list(filter(None, i2))
                if len(i3) == 5:
                     i4 = {"datetime":"{date}T{time}".format(date=i3[0], time=i3[1]), "size": "{size} {units}".format(size=i3[2], units=i3[3]), "file": i3[4]}
                     data_list.append(i4)

        # QgsMessageLog.logMessage(data_list[2]['file'], "OpenHazardsPH")

        self.dockwidget.dataTable.setRowCount(len(data_list) - 1)
        for row in range(0, len(data_list)):
            for col in range(0,2):
                if col == 0:
                    self.dockwidget.dataTable.setItem(row, col, QTableWidgetItem(data_list[row]['file']))
                elif col == 1:
                    self.dockwidget.dataTable.setItem(row, col, QTableWidgetItem(data_list[row]['size']))


    def load_vector_data(self):

        base_url = '/vsicurl/https://open-hazards-ph.s3.amazonaws.com/'
        file = self.dockwidget.dataTable.selectedItems()[0].text()
        # QgsMessageLog.logMessage(file, "OpenHazardsPH")
        name = file.split("/")[-1].split(".")[0]

        data = QgsVectorLayer('{base_url}{file}'.format(base_url=base_url, file=file), name, 'ogr')
        QgsProject.instance().addMapLayer(data)


    def load_raster_data(self):

        base_url = '/vsicurl/https://open-hazards-ph.s3.amazonaws.com/'
        file = self.dockwidget.dataTable.selectedItems()[0].text()
        name = file.split("/")[-1].split(".")[0]

        data = QgsRasterLayer('{base_url}{file}'.format(base_url=base_url, file=file), name, 'gdal')
        QgsProject.instance().addMapLayer(data)
