"""
Bundle Dependencies Script for Open Geodata Browser Plugin
Downloads and extracts Python packages into shared_libs folder

Usage: python bundle_dependencies.py
"""

import os
import sys
import subprocess
import shutil
import tarfile
import zipfile
from pathlib import Path


class DependencyBundler:
    """Handle downloading and extracting dependencies"""
    
    def __init__(self):
        self.plugin_dir = Path(__file__).parent
        self.shared_libs_dir = self.plugin_dir / 'shared_libs'
        self.packages = [
            'pystac-client',
            'open-geodata-api',
            'requests'
        ]
    
    def run(self):
        """Main bundling workflow"""
        print("\n" + "=" * 70)
        print("📦 Open Geodata Browser - Dependency Bundler")
        print("=" * 70)
        
        self.create_shared_libs_dir()
        
        if not self.download_packages():
            print("\n✗ Failed to download packages")
            return False
        
        if not self.extract_packages():
            print("\n✗ Failed to extract packages")
            return False
        
        self.create_init_file()
        self.cleanup_downloads()
        self.print_summary()
        
        return True
    
    def create_shared_libs_dir(self):
        """Create or clean shared_libs directory"""
        print(f"\n📁 Setting up shared_libs directory...")
        
        if self.shared_libs_dir.exists():
            print(f"  Cleaning existing: {self.shared_libs_dir}")
            shutil.rmtree(self.shared_libs_dir)
        
        self.shared_libs_dir.mkdir(exist_ok=True)
        print(f"  ✓ Created: {self.shared_libs_dir}")
    
    def download_packages(self):
        """Download packages without dependencies"""
        print(f"\n⬇️  Downloading packages...")
        
        try:
            cmd = [
                sys.executable,
                '-m',
                'pip',
                'download',
                '-d',
                str(self.shared_libs_dir),
                '--no-deps'
            ] + self.packages
            
            result = subprocess.run(
                cmd,
                capture_output=True,
                text=True,
                check=False
            )
            
            if result.returncode == 0:
                print("  ✓ Successfully downloaded packages")
                return True
            else:
                print(f"  ✗ Download failed:\n{result.stderr}")
                return False
        
        except Exception as e:
            print(f"  ✗ Error downloading: {e}")
            return False
    
    def extract_packages(self):
        """Extract downloaded packages"""
        print(f"\n📦 Extracting packages...")
        
        extracted_count = 0
        
        # Extract .tar.gz files
        for tar_file in self.shared_libs_dir.glob("*.tar.gz"):
            try:
                print(f"  Extracting: {tar_file.name}")
                with tarfile.open(tar_file, 'r:gz') as tar:
                    tar.extractall(self.shared_libs_dir)
                extracted_count += 1
            except Exception as e:
                print(f"  ⚠️  Failed to extract {tar_file.name}: {e}")
        
        # Extract .whl files
        for whl_file in self.shared_libs_dir.glob("*.whl"):
            try:
                print(f"  Extracting: {whl_file.name}")
                with zipfile.ZipFile(whl_file, 'r') as zip_ref:
                    zip_ref.extractall(self.shared_libs_dir)
                extracted_count += 1
            except Exception as e:
                print(f"  ⚠️  Failed to extract {whl_file.name}: {e}")
        
        if extracted_count > 0:
            print(f"  ✓ Extracted {extracted_count} packages")
            return True
        else:
            print("  ✗ No packages extracted")
            return False
    
    def create_init_file(self):
        """Create __init__.py in shared_libs"""
        print(f"\n📝 Creating __init__.py...")
        
        init_file = self.shared_libs_dir / '__init__.py'
        init_file.write_text('# Bundled dependencies for Open Geodata Browser\n')
        
        print(f"  ✓ Created: {init_file}")
    
    def cleanup_downloads(self):
        """Remove downloaded archive files"""
        print(f"\n🧹 Cleaning up downloaded files...")
        
        removed_count = 0
        
        for tar_file in self.shared_libs_dir.glob("*.tar.gz"):
            tar_file.unlink()
            removed_count += 1
        
        for whl_file in self.shared_libs_dir.glob("*.whl"):
            whl_file.unlink()
            removed_count += 1
        
        print(f"  ✓ Removed {removed_count} archive files")
    
    def print_summary(self):
        """Print summary of bundled packages"""
        print("\n" + "=" * 70)
        print("✓ BUNDLING COMPLETE!")
        print("=" * 70)
        
        py_packages = [
            d for d in self.shared_libs_dir.iterdir()
            if d.is_dir() and not d.name.startswith('.')
        ]
        
        print(f"\n📦 Bundled packages in shared_libs:")
        for pkg in sorted(py_packages):
            size_mb = sum(f.stat().st_size for f in pkg.rglob('*') if f.is_file()) / (1024 * 1024)
            print(f"  • {pkg.name} ({size_mb:.2f} MB)")
        
        total_size = sum(
            f.stat().st_size for f in self.shared_libs_dir.rglob('*') if f.is_file()
        ) / (1024 * 1024)
        
        print(f"\n📊 Total size: {total_size:.2f} MB")
        print(f"📁 Location: {self.shared_libs_dir}")
        print("\n✓ Ready to package plugin!")
        print("=" * 70 + "\n")


def main():
    """Main entry point"""
    bundler = DependencyBundler()
    
    try:
        success = bundler.run()
        sys.exit(0 if success else 1)
    except KeyboardInterrupt:
        print("\n\n✗ Bundling cancelled by user")
        sys.exit(1)
    except Exception as e:
        print(f"\n✗ Unexpected error: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)


if __name__ == '__main__':
    main()
