"""
Open Geodata Browser - QGIS Plugin
Provides access to satellite imagery from multiple STAC providers
"""

import sys
import os
from pathlib import Path
from qgis.core import QgsMessageLog, Qgis


def setup_bundled_dependencies():
    """Setup bundled dependencies from shared_libs folder"""
    plugin_dir = Path(__file__).parent
    shared_libs_path = plugin_dir / 'shared_libs'
    
    if not shared_libs_path.exists():
        QgsMessageLog.logMessage(
            'No shared_libs folder found - dependencies must be installed separately',
            'Open Geodata Browser',
            Qgis.Info
        )
        return False
    
    # Add shared_libs to path if not already there
    if str(shared_libs_path) not in sys.path:
        sys.path.insert(0, str(shared_libs_path))
        QgsMessageLog.logMessage(
            f'Added shared_libs to Python path: {shared_libs_path}',
            'Open Geodata Browser',
            Qgis.Info
        )
    
    # Try to import from bundled libs
    try:
        import open_geodata_api
        QgsMessageLog.logMessage(
            '✓ open-geodata-api loaded from bundled libs',
            'Open Geodata Browser',
            Qgis.Info
        )
        
        import pystac_client
        QgsMessageLog.logMessage(
            '✓ pystac-client loaded from bundled libs',
            'Open Geodata Browser',
            Qgis.Info
        )
        
        import requests
        QgsMessageLog.logMessage(
            '✓ requests loaded from bundled libs',
            'Open Geodata Browser',
            Qgis.Info
        )
        
        return True
    except ImportError as e:
        QgsMessageLog.logMessage(
            f'Failed to load bundled dependencies: {str(e)}',
            'Open Geodata Browser',
            Qgis.Warning
        )
        return False


def check_system_dependencies():
    """Check if dependencies are already installed in system Python"""
    try:
        import open_geodata_api
        import pystac_client
        import requests
        QgsMessageLog.logMessage(
            '✓ All dependencies found in system Python',
            'Open Geodata Browser',
            Qgis.Success
        )
        return True
    except ImportError:
        return False


def classFactory(iface):
    """Load GeodataBrowser class"""
    # Try bundled first, then system
    bundled_ok = setup_bundled_dependencies()
    system_ok = check_system_dependencies()
    
    if not (bundled_ok or system_ok):
        QgsMessageLog.logMessage(
            'Dependencies not found - plugin will prompt for installation',
            'Open Geodata Browser',
            Qgis.Warning
        )
    
    from .geodata_browser import GeodataBrowser
    return GeodataBrowser(iface)
